#define PROG_NAME "fni_view"
#define PROG_DESC "3D interactive visualization of a height map"
#define PROG_VERS "2.0"

#define PROG_C_COPYRIGHT "Copyright  2005 Universidade Estadual Fluminense (UFF)."

/* Last edited on 2010-08-15 00:03:20 by stolfi */

#define PROG_HELP \
  "  " PROG_NAME " \\\n" \
  "    [ -scale {ZSCALE} ] \\\n" \
  "    [ -isMask {ISMASK} ] \\\n" \
  "    [ -colorize {VMIN} {VMAX} ] \\\n" \
  "    [ -channel {CHAN} ] \\\n" \
  "    [ -texture {TEXTURE_FILE} ] \\\n" \
  "    [ < ] {HEIGHT_FILE}"

#define PROG_INFO \
  "NAME\n" \
  "  " PROG_NAME " - " PROG_DESC "\n" \
  "\n" \
  "SYNOPSIS\n" \
  PROG_HELP "\n" \
  "\n" \
  "DESCRIPTION\n" \
  "  Reads an image map from file {HEIGHT_FILE}.  Displays a selected channel of it" \
  " as a terrain in 3D.\n" \
  "\n" \
  "  If \"-texture\" is specified, reads also a second image from" \
  " {TEXTURE_FILE}, and uses it asa texture map to colorize the terrain.\n" \
  "\n" \
  "  The interface allows the user to move the camera around.\n" \
  "\n" \
  "  The sample value in channel {CHAN}, column {x}, row {y} of the input image is" \
  " taken as the Z coordinate of a 3D terrain, above the" \
  " point {(x,y)} of the {Z=0} plane.  The terrain is" \
  " displayed by conecting those points with quadrangular" \
  " patches formed by 4 or 8 triangles.  The sample values may be optionally magnified" \
  " by a scale factor before displaying.\n" \
  "\n" \
  "  The texture image may have either the same dimensions as the" \
  " height image, or exactly one column and one row less than it.  In the" \
  " first case, the texture pixel in column {x}, row {y} is taken" \
  " as the color of the terrain above the unit square with" \
  " diagonal {(x-1/2,y-1/2)} and {(x+1/2,y+1/2)}.  In the" \
  " second case, the pixel in column {x}, row {y} of the texture map is" \
  " taken as the color of the unit-side square whose diagonal" \
  " corners are {(x,y)} and {(x+1,y+1)}.\n" \
  "\n" \
  " If no texture map is specified, the terrain is colorized" \
  " with colors computed from the sample values.\n" \
  "\n" \
  "  The input files may be in the PNM (PBM/PGM/PPM) format" \
  " (see {jspnm_image.h}) or in the  float-valued multi-channel" \
  " image format (FNI) (see {float_image.h}).  If the" \
  " argument {HEIGHT_FILE} is \"-\" or omitted, the program reads it" \
  " from standard input, assuming the FNI format.\n" \
  "\n" \
  "  If the main image is in the PNM format, its samples are" \
  " converted to floats in the range {[0_1]} as specified by" \
  " the \"-isMask\" option. \n" \
  "\n" \
  "OPTIONS\n" \
  "  -channel {CHAN}\n" \
  "    Specifies the channel of the height file that is to be used as the" \
  " height map.  Channel numbers start at 0; the program fails if the" \
  " channel does not exist in the input file.  The default is \"-channel 0\".\n" \
  "\n" \
  "  -scale {ZSCALE}\n" \
  "    Specifies a multiplying scale factor for the heights" \
  " recovered from the height map.  The default is \"-scale 1.0\".\n" \
  "\n" \
  "  -isMask {ISMASK}\n" \
  "    This optional Boolean argument specifies the interpretation" \
  " of integer sample values when the main input file is in the PNM format, specifically" \
  " how the integer values {0..MAXVAL} are mapped to the range {[0_1]}.  If {ISMASK} is true (\"T\" or 1)," \
  " " sample_conv_0_1_isMask_true_INFO "  If {ISMASK} is false (\"F\" or 0)," \
  " " sample_conv_0_1_isMask_false_INFO "  The default is \"-isMask F\". This" \
  " option does not affect the reading of PNM texture files.\n" \
  "\n" \
  "  -texture {TEXTURE_FILE}\n" \
  "    Specifies the name of the FNI file containing the texture" \
  " map.  If {TEXTURE_FILE} is \"-\", reads the texture map from" \
  " standard input too (after the height map, if applicable).\n" \
  "    The texture map should have one column and one row" \
  " less than the height map.\n" \
  "\n" \
  "  -colorize {VMIN} {VMAX}\n" \
  "  -colorize auto\n" \
  "    Specifies the range of sample values to assume when computing" \
  " pseudocolors.  This argument is used only if the texture map is" \
  " omitted or turned off.  In the first option, samples" \
  " in the range {[VMIN _ VMAX]} are mapped to colors of an appropriate" \
  " pseudocolor scale; samples outside that range are mapped to the end" \
  " points of the scale.  The \"auto\" option will set {VMIN} and {VMAX} to" \
  " the maximum and minimum" \
  " samples in the current channel.  The default is \"-colorize auto\".\n" \
  "\n" \
  "DOCUMENTATION OPTIONS\n" \
  argparser_help_info_HELP_INFO "\n" \
  "\n" \
  "SEE ALSO\n" \
  "  pnm_to_fni(1), fni_to_pnm(1), {float_image.h}.\n" \
  "\n" \
  "AUTHOR\n" \
  "  Created 2005 by Rafael Saracchini, IC-UFF.\n" \
  "\n" \
  "MODIFICATION HISTORY\n" \
  "  apr/2006 by Jorge Stolfi, IC-UNICAMP: Recast to use jslibs, argparser, etc..\n" \
  "  jul/2009 by R. Saracchini: Changed to use lights and display list.\n" \
  "  jun/2010 by R. Saracchini: Added mouse support.\n" \
  "  jul/2010 by Jorge Stolfi: Added PGM/PPM input.\n" \
  "  jul/2010 by Jorge Stolfi: Added support for same-size textures.\n" \
  "  jul/2010 by Jorge Stolfi: Added \"-colorize\" option.\n" \
  "  aug/2010 by Jorge Stolfi: Added \"-isMask\" option.\n" \
  "\n" \
  "WARRANTY\n" \
  argparser_help_info_NO_WARRANTY "\n" \
  "\n" \
  "RIGHTS\n" \
  "  " PROG_C_COPYRIGHT ".\n" \
  "\n" \
  argparser_help_info_STANDARD_RIGHTS

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <values.h>
#include <assert.h>
#include <string.h>

#include <GL/glu.h>
#include <GL/glut.h>

#include <argparser.h>
#include <float_image.h>
#include <float_pnm_image_io.h>
#include <sample_conv.h>
#include <frgb.h>
#include <frgb_path.h>
#include <bool.h>
#include <jsfile.h>
#include <r3.h>

#include <fvw_GL.h>
#include <fvw_paint.h>
#include <fvw_paint_node_colored.h>
#include <fvw_paint_cell_colored.h>
#include <fvw_paint_self_colored.h>

#define INF INFINITY

#define fvw_zoom_step_ratio (15.0/16.0)
  /* Zoom in/out kbd commands multiply/divide the obs distance by this amount. */

#define fvw_Z_scale_step_ratio (pow(2.0,1.0/12.0))
  /* Scale incr/decr kbd commands multiply/divide the Z scale factor by this amount. */

/* COMMAND-LINE OPTIONS */

typedef struct options_t
  { char* height;         /* Name of height map. */
    int channel;          /* Initial channel to display. */
    double scale;         /* Initial scale to use for all channels. */
    bool_t isMask;        /* TRUE if input samples are to be converted as in a mask. */
    char* texture;        /* Name of texture map, or NULL if not specified. */
    bool_t colorize_auto; /* TRUE to get pseudo-color height range from height map. */
    float colorize_vmin;  /* Min sample value for pseudo-color, or {NAN}. */
    float colorize_vmax;  /* Max sample value for pseudo-color, or {NAN}. */
  } options_t;

/* GLOBAL VARIABLES USED BY GL METHODS */

typedef struct fvw_state_t 
  { 
    /* Height map: */
    float_image_t *ht;   /* Height image (const). */
    int channel;         /* Current channel being displayed (mutable). */
    double_vec_t scale;  /* Current Z scale factor for each channel, or INF if undefined (mutable). */
    double defScale;     /* Default Z scale for new channels (mutable). */
    
    /* Texture map: */
    float_image_t *tx;   /* Texture image (const). */
    bool_t node_tx;      /* TRUE if {tx} is suitable for node-painting (const). */
    bool_t cell_tx;      /* TRUE if {tx} is suitable for cell-painting (const). */
    bool_t texturize;    /* When TRUE, use texture map (mutable). */
    
    /* Reference plane: */
    bool_t refPlane;     /* When TRUE, show reference plane (mutable). */
    
    /* Current window dimensions (set by {fvw_reshape_method}): */
    GLint window_HSize; /* Width (mutable). */
    GLint window_VSize; /* Height (mutable). */

    /* Observer's position relative to center of bbox: */
    GLfloat azimuth;    /* Azimuth from X axis (degrees, mutable). */
    GLfloat elevation;  /* Elevation from XY plane (degrees, mutable). */
    GLfloat distance;   /* Distance (pixels, mutable). */

    double RAD;         /* Nominal radius of terrain (for reasonable Z scale). */
    
    /* The following data needs to be recomputed whenever {channel} changes. */
    int vrange_channel;     /* Channel for which the bbox was computed (mutable). */

    /* Bounding box for pixel corners in channel {vrange_channel}: */
    bool_t vrange_auto; /* TRUE means ajdust {vmin,vmax} to the channel samples. */
    float vmin, vmax;   /* Low and high UNSCALED Z coordinate (pixels, mutable). */
    
    /* Mouse state: */
    int mouse_x,mouse_y; /* Position of last mouse event (mutable). */
  } fvw_state_t;

static fvw_state_t *fvw_state = NULL; /* The state displayed in the GL window. */

/* INTERNAL PROTOTYPES */

fvw_state_t *fvw_create_state(options_t *o);
  /* Creates and initializes a window data record, suitable for use
    used by the methods {fvw_display_method} etc. */

float_image_t *fvw_read_image(char *name, bool_t isMask, int *NCP, int *NXP, int *NYP);
  /* Reads a float image, in the format of {float_image_write} (if the
    {name} ends in ".fni") or in PBM/PGM/PPM format (if {name} ends in
    ".pbm", ".pgm", ".ppm", ".pnm"). If {name} is "-", reads from
    {stdin} assuming {float_image_write} format. The procedure also
    sets {*NCP,*NXP,*NYP} to the number of channels, columns, and rows.*/

float_image_t *fvw_read_float_image(char *name);
  /* Reads a float image, in the format of {float_image_write}, from
    the named file (or {stdin} if {name} is "-"). */

options_t *fvw_parse_options(int argc, char **argv);
  /* Parses the command line arguments and packs them as an {options_t}. */

int main(int argc,char** argv);

/* MAJOR PAINTING FUNCTIONS */

void fvw_paint_everything(fvw_state_t *w);
  /* Repaints the terrain according to the data and parameters in {w}. */

void fvw_recompute_vrange(fvw_state_t *w);
  /* Recomputes the sample range {w->vmin,w->vmax} for the current channel
    {w->channel}. Usually called after a change in {w->channel}.
    Also sets {w->vrange_channel} to {w->channel}. */

void fvw_set_perspective(fvw_state_t *w);
  /* Sets the GL perspective view parameters according to the 
    window size, observer position, and bounding box data 
    stored in {w}. */

void fvw_set_lights(fvw_state_t *w);
  /* Sets the GL lighting parameters according to the state {w}. */

/* MEDIUM-LEVEL PAINTING */

void fvw_paint_height_map
  ( float_image_t *ht, 
    int c, 
    double zscale, 
    float vmin,
    float vmax,
    float_image_t *tx,
    bool_t node_tx,
    bool_t cell_tx
  );
  /* Paints the height map consisting of channel {c} of image {ht},
    with the Z coordinate being the sample values scaled by {zscale}.
    If {tx} is not null, it must have either the same size as {ht}, or
    one pixel less in both axes; in either case, paints the terrain
    with color {tx}. If {tx} is NULL, paints with colors based on the
    unscaled sample values relative to the unscaled sample range {[vmin _ vmax]}. */

void fvw_paint_reference_plane(fvw_state_t *w, float z);
  /* Paints a horizontal rectangle at height {z} 
    (without any scaling). */

/* GL WINDOW METHODS */

void fvw_display_method(void);
  /* Processes redisplay requests by the window manager. */

void fvw_reshape_method(int width, int height);
  /* Processes window reshape events. */

void fvw_keyboard_method(unsigned char key, int x, int y);
  /* Processes keyboard events. */
     
void fvw_passivemouse_method( int x, int y);
  /* Processes passive mouse events (no mouse clicked) */

void fvw_activemouse_method( int x, int y);
  /* Processes active mouse events (mouse clicked) */

void fvw_special_method(int key, int x, int y);
  /* Processes special user input events. */

/* IMPLEMENTATIONS */

int main(int argc, char** argv)
  {
    fvw_GL_initialize_libraries(&argc, argv);
  
    options_t *o = fvw_parse_options(argc, argv);

    fvw_state = fvw_create_state(o);
    
    fvw_GL_initialize_window
      ( o->height,
        fvw_display_method,
        fvw_reshape_method,
        fvw_keyboard_method,
	fvw_passivemouse_method,
	fvw_activemouse_method,
        fvw_special_method
      );
    
    fvw_GL_start_viewing();
    
    return 0;
  }

fvw_state_t *fvw_create_state(options_t *o)
  { 
    fvw_state_t *w = (fvw_state_t *)notnull(malloc(sizeof(fvw_state_t)), "no mem");

    w->window_HSize = fvw_GL_default_window_HSize;
    w->window_VSize = fvw_GL_default_window_VSize;

    /* Read height map, check and save the channel index: */
    int ht_NC, ht_NX, ht_NY;
    w->ht = fvw_read_image(o->height, o->isMask, &ht_NC, &ht_NX, &ht_NY);
    fprintf(stderr, "height map has %d channels, %d columns, %d rows\n", ht_NC, ht_NX, ht_NY); 
    demand(ht_NC > 0, "invalid num of channels in height map");
    
    /* The initial channel is user-specified: */
    w->channel = o->channel;
    if ((w->channel < 0) || (w->channel >= ht_NC))
      { fprintf(stderr, "invalid height map channel %d, using 0\n", w->channel);
        w->channel = 0;
      }

    /* The default scale for new channels is user-specified: */
    w->defScale = o->scale;

    /* The channel scale vector is initially filled with {+INF}: */
    w->scale = double_vec_new(ht_NC);
    { int c; for (c = 0; c < ht_NC; c++) { w->scale.e[c] = +INF; } }
    
    /* Read texture map, if any,and check its size: */
    int tx_NC, tx_NX, tx_NY; /* Channels, columns and rows of texture map. */
    if (o->texture != NULL)
      { bool_t tx_isMask = FALSE; /* Assume that texture has smooth pixel distr. */
        w->tx = fvw_read_image(o->texture, tx_isMask, &tx_NC, &tx_NX, &tx_NY);
        fprintf(stderr, "texture map has %d channels, %d columns, %d rows\n", tx_NC, tx_NX, tx_NY); 
        demand((tx_NC == 1) || (tx_NC == 3), "invalid num of channels in texture map");
        w->node_tx = (tx_NX == ht_NX) && (tx_NY == ht_NY);
        w->cell_tx = (tx_NX == ht_NX-1) && (tx_NY == ht_NY-1);
        demand(w->node_tx || w->cell_tx, "texture file has wrond size");
        w->texturize = TRUE;
      }
    else
      { w->tx = NULL; 
        w->texturize = w->node_tx = w->cell_tx = FALSE;
      }

    /* Do not show reference plane initially: */
    w->refPlane = FALSE;
    
    /* Set colorize options from command line: */
    w->vrange_auto = o->colorize_auto;
    w->vmin = o->colorize_vmin;
    w->vmax = o->colorize_vmax;
    
    /* Nominal radius of data: */
    double SZX = (double)ht_NX;
    double SZY = (double)ht_NY;
    w->RAD = sqrt((SZX*SZX + SZY*SZY)/2);
    
    /* Initial observer's position: */
    w->azimuth = 0;
    w->elevation = 30;
    w->distance = 2*w->RAD;

    /* Set {vrange_channel} to -1 to force a bbox computation: */
    w->vrange_channel = -1;
    
    return w;
  }

float_image_t *fvw_read_image(char *name, bool_t isMask, int *NCP, int *NXP, int *NYP)
  { float_image_t *fim = NULL;
    /* Decide type by file name: */
    int len = strlen(name);
    if ((strcmp(name,"-") == 0) || ((len > 4) && (strcmp(name+len-4,".fni") == 0)))
      { /* Float image file: */
        fim = fvw_read_float_image(name);
      }
    else if ((len > 4) && (name[len-4] == '.') && (name[len-3] == 'p') && (name[len-1] == 'm'))
      { /* PBM/PGM/PPM file: */
        fim = float_pnm_image_read(name, isMask, 1.0, 0.0, TRUE, TRUE, TRUE);
      }
    else
      { fprintf(stderr, "file name = \"%s\"\n", name);
        demand(FALSE, "invalid file name");
      }
    (*NCP) = fim->sz[0];
    (*NXP) = fim->sz[1];
    (*NYP) = fim->sz[2];
    demand (((*NCP) > 0) && ((*NXP) > 0) && ((*NYP) > 0), "image is empty\n");
    return fim;
  }

float_image_t *fvw_read_float_image(char *name)
  { FILE *rd = open_read(name, TRUE);
    float_image_t *fim = float_image_read(rd);
    if (rd != stdin) { fclose(rd); }
    return fim;
  }

void fvw_paint_everything(fvw_state_t *w)
  {
    /* Set the GL graphics modes for depth painting: */
    glShadeModel(GL_SMOOTH);
    glDepthMask(GL_TRUE);
    glEnable(GL_DEPTH_TEST);
    
    /* Get the channel {c} to display: */
    int c = w->channel;
    
    /* Get the Z scale factor for this channel: */
    if (fabs(w->scale.e[c]) == INF) { w->scale.e[c] = w->defScale; }
    double zscale = w->scale.e[c];
    
    /* Make {zscale} the default for any other `new' channels: */
    w->defScale = zscale;
    
    /* Recompute the sample range if needed: */
    if ((w->vrange_auto) && (c != w->vrange_channel)) { fvw_recompute_vrange(w); }
    assert(! isnan(w->vmin));
    assert(! isnan(w->vmax));
    
    /* Get  the height image dimensons: */
    int HNC, HNX, HNY;
    float_image_get_size(w->ht, &HNC, &HNX, &HNY);
    
    /* Compute the center of attention: */
    GLfloat ctrx = (double)HNX/2;
    GLfloat ctry = (double)HNY/2;
    GLfloat ctrz = zscale * (w->vmin + w->vmax)/2;

    /* Compute observer's position: */
    double az =  M_PI*w->azimuth/180;   /* Observer's azimuth in radians. */
    double ev =  M_PI*w->elevation/180; /* Observer's elevation in radians. */
    double ca = cos(az), sa = sin(az);  
    double ce = cos(ev), se = sin(ev);
    double R = w->distance;

    GLfloat obsx = ctrx + R * ca * ce;
    GLfloat obsy = ctry + R * sa * ce;
    GLfloat obsz = ctrz + R * se;

    glPushMatrix();
    
    /* Set the view transformation matrix: */
    gluLookAt( obsx, obsy, obsz, ctrx, ctry, ctrz, 0.0, 0.0, 1.0);

    /* Place the lights: */
    fvw_set_lights(w);
    
    /* Paint the reference plane, if any: */
    if (w->refPlane)
      { float zplane = zscale*w->vmin - 0.1*w->RAD;
        fvw_paint_reference_plane(w, zplane);
      }

    /* Paint the terrain: */
    fvw_paint_height_map(w->ht, c, zscale, w->vmin, w->vmax, (w->texturize ? w->tx : NULL), w->node_tx, w->cell_tx);
    
    glPopMatrix();
    
    glDisable(GL_DEPTH_TEST);
  }    

void fvw_set_perspective(fvw_state_t *w)
  {
    /* Set viewport: */
    glViewport(0, 0, (GLint)w->window_HSize, (GLint)w->window_VSize);

    /* Set perspective matrix: */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    float angheight = 40.0; /* Angular height of window (degrees) */
    float aspect = (float)w->window_HSize/(float)w->window_VSize;
    gluPerspective(angheight, aspect, 0.1, w->distance + 2.0*w->RAD);
    glMatrixMode(GL_MODELVIEW);
  }

void fvw_set_lights(fvw_state_t *w)
  {
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    double ambi = 0.1;                /* Amount of ambient light. */
    double spec = 0.0;                /* Amount of shine-scatterable light. */
    double diff = 1.0 - ambi - spec;  /* Amount of diffusible light. */

    if (ambi > 0)
      { GLfloat light_ambient[] = { ambi, ambi, ambi, 1.0 };
        glLightfv(GL_LIGHT0, GL_AMBIENT,  light_ambient);
      }

    if (diff > 0)
      { GLfloat light_diffuse[] = { diff, diff, diff, 1.0 };
        glLightfv(GL_LIGHT0, GL_DIFFUSE,  light_diffuse);
      }

    if (spec > 0)
      { GLfloat light_specular[] = { spec, spec, spec, 1.0 };
        glLightfv(GL_LIGHT0, GL_SPECULAR,  light_specular);
      }

    /* Choose light's position relative to observer: */
    double rev = w->elevation/90.0;  /* Observer's rel elevation, in {[-1_+1]}. */
    double daz = +30;     /* Azimuth rel to observer (degrees). */
    double dev = rev*30;  /* Elevation rel to observer (degrees). */
    /* Compute light  direction vector: */
    double az =  M_PI*(w->azimuth + daz)/180;   /* Light's azimuth (radians). */
    double ev =  M_PI*(w->elevation + dev)/180; /* Observer's elevation (radians). */
    double ca = cos(az), sa = sin(az);  
    double ce = cos(ev), se = sin(ev);
    GLfloat dirx = ca * ce;
    GLfloat diry = sa * ce;
    GLfloat dirz = se;
    
    /* Place light at infinity: */
    GLfloat pos[] = { dirx, diry, dirz, 0.0 };
    glLightfv(GL_LIGHT0, GL_POSITION, pos);
  }

void fvw_recompute_vrange(fvw_state_t *w)
  {
    /* Get current channel to be displayed: */
    int c = w->channel;
    
    /* Get range of samples in this channel: */
    float vmin = +INF;
    float vmax = -INF;
    float_image_update_sample_range(w->ht, c, &vmin, &vmax);

    w->vmin = vmin; w->vmax = vmax; 
    w->vrange_channel = c;
  }

void fvw_paint_height_map
  ( float_image_t *ht, 
    int c, 
    double zscale, 
    float vmin,
    float vmax,
    float_image_t *tx,
    bool_t node_tx,
    bool_t cell_tx
  )
  {
    if (ht == NULL) { return; }
    if (fvw_debug_paint) { fprintf(stderr, "+ %s\n", __FUNCTION__); }

    /* Get  the height image dimensons: */
    int HNC, HNX, HNY;
    float_image_get_size(ht, &HNC, &HNX, &HNY);

    /* Set surface finish: */
    glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    
    if (tx == NULL)
      { /* No texture map, paint with height-derived colors: */
        fvw_paint_self_colored_height_map(ht, c, zscale, vmin, vmax);
      }
    else if (cell_tx)
      { /* Texture colors are associated with grid cells: */
        fvw_paint_cell_colored_height_map(ht, c, zscale, tx);
      }
    else if (node_tx)
      { /* Texture colors are associated with grid corners: */ 
        fvw_paint_node_colored_height_map(ht, c, zscale, tx);
      }
    else
      { assert(FALSE); }
    glDisable(GL_COLOR_MATERIAL);
    if (fvw_debug_paint) { fprintf(stderr, "- %s\n", __FUNCTION__); }
  }

void fvw_paint_reference_plane(fvw_state_t *w, float z)
  {
    /* Get the height image dimensons: */
    int HNC, HNX, HNY;
    float_image_get_size(w->ht, &HNC, &HNX, &HNY);
    
    /* X and Y extents are those of the bitmap: */
    float xmin = 0.0;
    float xmax = (float)HNX;
    float ymin = 0.0;
    float ymax = (float)HNY;
    
    glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);

    glColor3f(0.650, 0.975, 0.925);
    
    glBegin(GL_QUADS);
    glVertex3f(xmin, ymin, z);
    glVertex3f(xmax, ymin, z);
    glVertex3f(xmax, ymax, z);
    glVertex3f(xmin, ymax, z);
    glEnd();
    
    glDisable(GL_COLOR_MATERIAL);
  }

/* GL EVENT-HANDLING METHODS */

void fvw_display_method(void)
  {
    if (fvw_debug_GL) { fprintf(stderr, "+ %s\n", __FUNCTION__); }
    fvw_state_t *w = fvw_state;

    /* Clear the window: */
    glClearColor(0.750, 0.750, 0.750, 1.000); /* Light gray, opaque. */
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

    /* Paint the terrain and stuff: */
    fvw_paint_everything(w);
    
    /* Display the result: */
    glutSwapBuffers();
    if (fvw_debug_GL) { fprintf(stderr, "- %s\n", __FUNCTION__); }
  }

void fvw_reshape_method(int width, int height)
  {
    if (fvw_debug_GL) { fprintf(stderr, "+ %s\n", __FUNCTION__); }

    fvw_state_t *w = fvw_state;
    
    /* Save window size: */
    w->window_HSize = width;
    w->window_VSize = height;
    
    /* Adjust the perspective parameters to the window's current size: */
    fvw_set_perspective(w);

    if (fvw_debug_GL) { fprintf(stderr, "- %s\n", __FUNCTION__); }
  }

void fvw_keyboard_method(unsigned char key, int x, int y)
  {
    if (fvw_debug_GL) { fprintf(stderr, "+ %s\n", __FUNCTION__); }

    fvw_state_t *w = fvw_state;
    
    switch (key) 
      { 
        case 27: 
        case 'q':
        case 'Q':
          /* Quit: */
          exit(0);
          break;

        case 't':
          /* Toggle texturization: */
          w->texturize = !w->texturize;
          glutPostRedisplay();
          break;

        case 'p':
          /* Toggle reference plane: */
          w->refPlane = !w->refPlane;
          glutPostRedisplay();
          break;

        case 'Z':
          /* Zoom in, down to a limit: */
          w->distance = fmax(w->distance*fvw_zoom_step_ratio, w->RAD/16);
          glutPostRedisplay();
          break;

        case 'z':
          /* Zoom out, up to a limit: */
          w->distance = fmin(w->distance/fvw_zoom_step_ratio, w->RAD*16);
          glutPostRedisplay();
          break;

        case 'S':
          /* Increase Z scale: */
          w->scale.e[w->channel] *= fvw_Z_scale_step_ratio;
          glutPostRedisplay();
          break;

        case 's':
          /* Reduce Z scale: */
          w->scale.e[w->channel] /= fvw_Z_scale_step_ratio;
          glutPostRedisplay();
          break;

        case 'c':
          /* Cycle through channels of height map: */
          { int NC = w->ht->sz[0]; w->channel = (w->channel + 1) % NC; }
          glutPostRedisplay();
          break;

        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
          /* Select named channel of height map: */
          { int c = key - '0';
            int NC = w->ht->sz[0];
            w->channel = (c < NC ? c : NC-1);
          }
          glutPostRedisplay();
          break;
      }

    if (fvw_debug_GL) { fprintf(stderr, "- %s\n", __FUNCTION__); }
  }

void fvw_passivemouse_method( int x, int y)
  {
    fvw_state_t *w = fvw_state;
    
    w->mouse_x = x;
    w->mouse_y = y;
  }

void fvw_activemouse_method( int x, int y)
  {
    fvw_state_t *w = fvw_state;
    
    int xvec, yvec;
    xvec = w->mouse_x - x;
    yvec = w->mouse_y - y;

    float az = w->azimuth + xvec;
    float el = w->elevation - yvec;
    el = fmax(-88.0, fmin(+88.0, el));
    
    w->elevation = el;
    w->azimuth = az;

    w->mouse_x = x;
    w->mouse_y = y;

    glutPostRedisplay();
  }

void fvw_special_method(int key, int x, int y)
  {
    if (fvw_debug_GL) { fprintf(stderr, "+ %s\n", __FUNCTION__); }

    fvw_state_t *w = fvw_state;
    
    switch (key)
      { 
        case GLUT_KEY_UP:
          w->elevation = fmin(w->elevation + 2, +88.0);
          glutPostRedisplay();
          break;

        case GLUT_KEY_DOWN:
          w->elevation = fmax(w->elevation - 2, -88.0);
          glutPostRedisplay();
          break;

        case GLUT_KEY_LEFT:
          w->azimuth = w->azimuth + 2;
          glutPostRedisplay();
          break;

        case GLUT_KEY_RIGHT:
          w->azimuth = w->azimuth - 2;
          glutPostRedisplay();
          break;
      }

    if (fvw_debug_GL) { fprintf(stderr, "- %s\n", __FUNCTION__); }
  }

/* COMMAND LINE PARSING */

options_t *fvw_parse_options(int argc, char **argv)
  {
    /* Initialize argument parser: */
    argparser_t *pp = argparser_new(stderr, argc, argv);
    argparser_set_help(pp, PROG_NAME " version " PROG_VERS ", usage:\n" PROG_HELP);
    argparser_set_info(pp, PROG_INFO);
    argparser_process_help_info_options(pp);
    
    /* Allocate the program's option record: */
    options_t *o = (options_t *)malloc(sizeof(options_t)); 
    
    /* Parse the texture map name: */
    if (argparser_keyword_present(pp, "-texture"))
      { o->texture = argparser_get_next(pp); }
    else
      { o->texture = NULL; }

    /* Parse the initial channel index to display: */
    if (argparser_keyword_present(pp, "-channel"))
      { o->channel = argparser_get_next_int(pp, 0, float_image_max_size-1); }
    else
      { o->channel = 0; }

    /* Parse the colorization range: */
    o->colorize_vmin = NAN; /* Default. */
    o->colorize_vmax = NAN; /* Default. */
    if (argparser_keyword_present(pp, "-colorize"))
      { if (argparser_keyword_present_next(pp, "auto"))
          { o->colorize_auto = TRUE; }
        else
          { o->colorize_auto = FALSE;
            o->colorize_vmin = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX);
            o->colorize_vmax = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX);
          }
      }
    else
      { o->colorize_auto = TRUE; }

    /* Parse the initial scale to display: */
    if (argparser_keyword_present_next(pp, "-scale"))
      { o->scale = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX); }
    else
      { o->scale = 1.00; }

    /* Parse the integer decoding options: */
    if (argparser_keyword_present(pp, "-isMask"))
      { o->isMask = argparser_get_next_bool(pp); }
    else
      { o->isMask = FALSE; }
    
    /* Go to the positional arguments: */
    argparser_skip_parsed(pp);
    
    /* Parse the height map name and its selected channel: */
    if ((argparser_next(pp) != NULL) && (! argparser_next_is_keyword(pp)))
      { o->height = argparser_get_next(pp); }
    else
      { o->height = "-"; }
    
    /* Check for spurious arguments: */
    argparser_finish(pp);
    
    return o;
  }

