#define PROG_NAME "photo_to_normal"
#define PROG_DESC "compute normal maps from N photos taken with different lights"
#define PROG_VERS "1.0"

/* Copyright  2004 by the Fluminense Federal University (UFF). */
/* See the copyright, authorship, and warranty notice at end of file. */
/* Last edited on 2008-05-25 00:45:50 by stolfi */

#define PROG_HELP \
  PROG_NAME "\\\n" \
  "    -scene {SCENE_IMG_1} .. {SCENE_IMG_NF} \\\n" \
  "    [ -maxval {MAXVAL} ] [ -noise {SDEV} ] \\\n" \
  "    { -gauge \\\n" \
  "        {POSX} {POSY} \\\n" \
  "        {GAUGE_NRM} \\\n" \
  "        {GAUGE_IMG_1} .. {GAUGE_IMG_NF} \\\n" \
  "    }.. \\\n" \
  "    -outPrefix {PREFIX} \\\n" \
  "    " argparser_help_info_HELP

#define PROG_INFO \
  "NAME\n" \
  "  " PROG_NAME " - " PROG_DESC "\n" \
  "\n" \
  "SYNOPSIS\n" \
  "  " PROG_HELP "\n" \
  "\n" \
  "DESCRIPTION\n" \
  "  The program reads a certain number {NF} of images of" \
  " the same scene, all taken with the same camera position but under" \
  " different lighting conditions (/light fields/); and outputs a" \
  " /normal map/, an /intrinsic color map/, and a " \
  " /strangeness map/ for that scene.\n" \
  "\n" \
  "LIGHT FIELD GAUGES\n" \
  "  The normal vector at a pixel is estimated with the help of one" \
  " or more /light field gauges/, objects with known shape, white" \
  " intrinsic color, and the same finish as the scene's surface.  Each" \
  " gauge should be photographed under the same light fields" \
  " used for the scene photos.\n" \
  "\n" \
  "  Presently the program accepts only one gauge.\n" \
  "\n" \
  "INPUT IMAGES\n" \
  "  The input images are read from files \"{SCENE_IMG_1}\", ...," \
  " \"{SCENE_IMG_NF}\", in the float-valued multichannel (FNI) format" \
  " (see {float_image.h} for a description).  The number {NF} of" \
  " light fields must be at least three, preferably more.  All" \
  " these images must have the same dimensions and the same number" \
  " {NC} of color channels.  The light fields must be as distinct as" \
  " possible, e.g. generated by a single round lamp placed" \
  " in different positions.  They must be chosen so that every" \
  " visible point on the scene's surface is fully illuminated in" \
  " at least three of the images.\n" \
  "\n" \
  "  For each gauge, the user must also provide a normal" \
  " map, which is read from the file \"{GAUGE_NRM}\"; and" \
  " a list of {NF} photos of that gauge, read from files" \
  " \"{GAUGE_IMG_1}\", ..., \"{GAUGE_IMG_NF}\"; where each" \
  " gauge photo must have been taken unde the same light" \
  " field as the corresponding scene photo." \
  "\n" \
  "OUTPUT IMAGES\n" \
  "  The normal map is a three-channel image file called" \
  " \"{PREFIX}-nrm.fni\" (also in FNI format), where the value" \
  " of each pixel is a mean normal vector of the scene's" \
  " surface within that pixel.\n" \
  "\n" \
  "  The intrinsic color map is an image file called" \
  " \"{PREFIX}-clr.fni\" (also in FNI format).  It has the same" \
  " number {NC} channels as the input image.  The value of each" \
  " pixel is the /intrinsic color/ of the scene's surface at" \
  " that point --- that is, the color that the surface would" \
  " have if it were illuminated perpendicularly by white light" \
  " of unit intensity.\n" \
  "\n" \
  "  The strangeness map is a single-channel image file called" \
  " \"{PREFIX}-dif.fni\" (also in FNI format).  The value of each" \
  " pixel is the /discrepancy/ between the relative lightness" \
  " of that pixel in the input images, and the best-matching" \
  " pixel in the light field gauge images, even accounting for" \
  " possible differences of intrinsic color. The discrepancy" \
  " ranges from zero to 2.0. Pixels with complex optical properties" \
  " or anomalous light fields --- such as projected" \
  " shadows, caustics, specular highlights, transparency, etc. ---" \
  " are likely to have high discrepancies.\n" \
  "\n" \
  "OPTIONS\n" \
  "  -scene {SCENE_IMG_1} .. {SCENE_IMG_NF}\n" \
  "    Specifies the names of FNI images showing the scene" \
  " under different light fields.  Use the file name" \
  " \"-\" to read any of these image(s) from the standard input.\n" \
  "  -gauge {POSX} {POSY} {GAUGE_NRM} {GAUGE_IMG_1} .. {GAUGE_IMG_NF}\n" \
  "    Specifies an additional gauge in the scece, and" \
  " its attributes: the nominal position {(POSX,POSY)}, the" \
  " name of the FNI file containing its normal" \
  " map, and the names of {NF} FNI files containing photos" \
  " of the gauge under the same light fields" \
  " used for the corresponding scenes images.  Use the" \
  " file name \"-\" to read any of these image(s) from the" \
  " standard input.\n" \
  "  -maxval {MAXVAL}\n" \
  "    Specifies the effective maximum pixel value" \
  " of the input scene images, as originally produced by the" \
  " camera.  This argument is used by the program to estimate" \
  " the standard deviation of the input image noise due to" \
  " quantization --- namely, {1/(MAXVAL + 1)/sqrt(12)}.  As a" \
  " special case, {MAXVAL = 0} means that the scene image" \
  " samples should be considered exact (free from quantization" \
  " noise). The default {MAXVAL} is 0.\n" \
  "\n" \
  "  -noise {SDEV}\n" \
  "    Specifies the standard deviation of any noise present" \
  " in each sample of the input scene images, in addition" \
  " to the quantization noise.  The default is 0." \
  "\n" \
  argparser_help_info_HELP_INFO "\n" \
  "\n" \
  "SEE ALSO\n" \
  "  virtual-gauge(1), normal_to_slope(1), slope_to_height(1)," \
  " pnm_to_fni(1), fni_to_pnm(1).\n" \
  "\n" \
  "AUTHOR\n" \
  "  Created jul/2005 by Rafael Saracchini, IC-UFF.\n" \
  "  Modified mar/2006 by Jorge Stolfi, UNICAMP:\n" \
  "    * FNI format for all input/output.\n" \
  "    * Reformed the command line options."

#define stringify(x) #x

#define _GNU_SOURCE
#include <stdio.h>
#include <math.h>
#include <values.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <float_image.h>
#include <argparser.h>
#include <jsfile.h>
#include <jstime.h>
#include <affirm.h>
#include <vec.h>
#include <r2.h>
#include <r3.h> 

#include <pst_basic.h>
#include <pst_signature.h>
#include <pst_argparser.h>

typedef struct gauge_opts_t  /* Parameters of a gauge from the command line: */
  { char* normal;        /* Filename of the normal map of gauge {g}. */
    r2_t pos;            /* Nominal position of gauge {g} in scene images. */
    name_vec_t photos;   /* {photos[i]} is the filename of gauge's image under light {i}. */
  } gauge_opts_t;

vec_typedef(gauge_opts_vec_t,gauge_opts_vec,gauge_opts_t);

typedef struct options_t
  { int NF;              /* Number of light fields. */
    name_vec_t photos;   /* {photos[i]} is the filename of scene image under light {i}. */
    int NG;              /* Number of gauge objects. */
    gauge_opts_vec_t Go; /* {Go[g]} contains the command line arguments of gauge {g}. */
    int maxval;          /* Max pixel value in original (quantized) scene images, or 0. */
    double noise;        /* Standard deviation of extra sample noise in scene images. */
    char *outPrefix;     /* Prefix for output file names. */
  } options_t;

image_vec_t ptn_read_images(name_vec_t *name, int *NCP, int *NXP, int *NYP);
  /* Reads a list of multichannel float-valued images {img[*]} from
    the files named in {name[*]}. All images must have the same
    dimensions, channel count, and color space. Returns in {*NXP,*NYP}
    the the width and height of those images, and in {*NCP} their
    channel count. */

float_image_t *ptn_read_image(char *name, int *NCP, int *NXP, int *NYP);
  /* Reads a float image from the named file (or {stdin} if {name} is
    "-"). If {*NCP,*NXP,NYP} are -1, stores into those variablesthe
    numbers of channels, columns, and rows of the image. Othwerwise
    these image attributes must match the values of those
    variables. */

void ptn_free_images(image_vec_t *img);
  /* Reclaims the images {img[*]} and sets {img} to the empty vector. */

void ptn_write_normals(char *name, float_image_t *NRM);
  /* Writes the normal map {NRM} as a three-channel float image file
    "{name}-nrm.fni" in the format of {float_image_write}. */

void ptn_write_colors(char *name, float_image_t *CLR);
  /* Writes the intrinsic color map {CLR} as a multichannel float image
    "{name}-clr.fni" in the format of {float_image_write}. */

void ptn_write_strangeness(char *name, float_image_t *DIF);
  /* Writes the strangeness map {DIF} as a single-channel float image
    "{name}-dif.fni" in the format of {float_image_write}. */

void ptn_write_float_image(char *name, char *tag, float_image_t *I);
  /* Writes the image {I} as a float image "{name}-{tag}.fni" in the format
    of {float_image_write}. */

options_t *ptn_parse_options(int argc, char **argv);
  /* Parses the command line options, returns them as an {options_t} record. */

gauge_opts_t ptn_parse_gauge_specs(argparser_t *pp, int NF);
   /* Parses the gauge arguments after a "-gauge" keyword, appending
     them to the gauge data in {o}. Requires the number of light
     fields {NF}. */

int main(int argc, char** argv)
  {
    /* Parse options from the command line: */
    options_t *o = ptn_parse_options(argc, argv);

    /* Build signature-to-normal tables from gauge images: */
    fprintf(stderr, "reading gauge images and building light table...\n");
    int NG = o->NG;
    light_table_t *tab[NG];
    int g;
    int GNC = -1;  /* All gauge photos must have the same num of channels. */
    for (g = 0; g < NG; g++)
      { /* Get options of of gauge {g}: */
        gauge_opts_t *gop = &(o->Go.e[g]); 
        /* Read normal map of gauge {g}: */
        int GNX = -1, GNY = -1;
        int GNRMNC = -1;
        float_image_t *GNRM = ptn_read_image(gop->normal, &GNRMNC, &GNX, &GNY);
        /* Read images of gauge {g} (must have size {GNX,GNY}): */
        image_vec_t G = ptn_read_images(&(gop->photos), &GNC, &GNX, &GNY);  
        tab[g] = pst_signature_build_table (&(gop->pos), GNRM, &G, FALSE);
        float_image_free(GNRM);
        ptn_free_images(&G);
      }

    /* Read scene images {S[i]}, the scene under light field {i}: */
    fprintf(stderr, "reading scene images...\n");
    int SNC = GNC; /* Channels of scene photos must match gauge photos. */
    int SNX = -1, SNY = -1; /* Number of columns and rows of scene images. */
    image_vec_t S = ptn_read_images(&(o->photos), &SNC, &SNX, &SNY);
    
    demand(SNC == GNC, "channel cound mismatch between gauge and scene images");

    fprintf(stderr, "processing scene images...\n");
    float_image_t *NRM = float_image_new(3,   SNX, SNY); /* Normal image. */
    float_image_t *CLR = float_image_new(SNC, SNX, SNY); /* Self-color image. */
    float_image_t *DIF = float_image_new(1,   SNX, SNY); /* Strangeness map. */

    double tini = user_cpu_time_usec();
    pst_signature_normals_from_photos(NG, tab, &S, o->maxval, o->noise, NRM, CLR, DIF);

    fprintf(stderr, "\n");
    
    /* Report time: */
    double trun = user_cpu_time_usec() - tini;
    fprintf(stderr, "%14.6f s total time\n", trun/1.0e6);
    fprintf(stderr, "%14.3f s per pixel\n", trun/(SNX*SNY));
    
    /* Write output images: */
    ptn_write_normals(o->outPrefix, NRM);
    ptn_write_colors(o->outPrefix, CLR);
    ptn_write_strangeness(o->outPrefix, DIF);
    return 0;
  }

void ptn_write_normals(char *name, float_image_t *NRM)
  { ptn_write_float_image(name, "nrm", NRM); }

void ptn_write_colors(char *name, float_image_t *CLR)
  { ptn_write_float_image(name, "clr", CLR); }

void ptn_write_strangeness(char *name, float_image_t *DIF)
  { ptn_write_float_image(name, "dif", DIF); }

void ptn_write_float_image(char *name, char *tag, float_image_t *I)
  { char *fileName;
    fileName = NULL; asprintf(&fileName, "%s-%s.fni", name, tag);
    FILE *wr = open_write(fileName, TRUE);
    float_image_write(wr, I);
    if (wr == stdout) { fflush(wr); } else { fclose(wr); }
    free(fileName);
  }

image_vec_t ptn_read_images(name_vec_t *name, int *NCP, int *NXP, int *NYP)
  { int NF = name->ne; /* Number of files: */
    image_vec_t imv = image_vec_new(name->ne);
    int i;
    for (i = 0; i < NF; i++)
      { imv.e[i] = ptn_read_image(name->e[i], NCP, NXP, NYP); }
    image_vec_trim(&imv, NF);
    return imv;
  }
  
float_image_t *ptn_read_image(char *name, int *NCP, int *NXP, int *NYP)    
  { FILE *rd = open_read(name, TRUE);
    float_image_t *img = float_image_read(rd);
    if (rd != stdin) { fclose(rd); }
    /* Get dimensions and number of channels: */
    int NC = img->sz[0];
    int NX = img->sz[1]; 
    int NY = img->sz[2];

    if ((*NCP) < 0) 
      { (*NCP) = NC; } 
    else 
      { demand(NC == (*NCP), "photo size mismatch: channels"); }

    if ((*NXP) < 0) 
      { (*NXP) = NX; } 
    else 
      { demand(NX == (*NXP), "photo size mismatch: columns"); }

    if ((*NYP) < 0) 
      { (*NYP) = NY; } 
    else 
      { demand(NY == (*NYP), "photo size mismatch: rows"); }
      
    return img;
  }

void ptn_free_images(image_vec_t *img)
  { int i;
    for (i = 0; i < img->ne; i++) { float_image_free(img->e[i]); }
    image_vec_trim(img, 0);
  }

options_t *ptn_parse_options(int argc, char **argv)
  {
    argparser_t *pp = argparser_new(stderr, argc, argv);
    argparser_set_help(pp, PROG_NAME " version " PROG_VERS ", usage:\n" PROG_HELP);
    argparser_set_info(pp, PROG_INFO);
    argparser_process_help_info_options(pp);
    
    options_t *o = (options_t *)notnull(malloc(sizeof(options_t)), "no mem"); 
    
    /* Parse list {o->photos} of scene images, obtain {o->NF}: */
    argparser_get_keyword(pp, "-scene");
    o->NF = -1;
    o->photos = pst_parse_file_name_list(pp, &(o->NF));
    if (o->NF < 1) { argparser_error(pp, "no scene images given"); }
      
    /* Parse list of gauge images, obtain {o->NG}: */
    o->NG = 0;
    o->Go = gauge_opts_vec_new(0);
    while (argparser_keyword_present(pp, "-gauge"))
      { /* Another gauge: */
        gauge_opts_t gop = ptn_parse_gauge_specs (pp, o->NF);
        gauge_opts_vec_expand(&(o->Go), o->NG);
        o->Go.e[o->NG] = gop;
        o->NG++;
      }
    gauge_opts_vec_trim(&(o->Go), o->NG);
    
    /* Parse standard deviation of image noise: */
    if (argparser_keyword_present(pp, "-noise"))
      { o->noise = argparser_get_next_double(pp, 0.0, 1.0e+20); }
    else
      { o->noise = 0.0; }
    
    if (argparser_keyword_present(pp, "-maxval"))
      { o->maxval = argparser_get_next_int(pp, 1, MAXINT/2); }
    else
      { o->maxval = 0; }
    
    argparser_get_keyword(pp, "-outPrefix");
    o->outPrefix = argparser_get_next(pp);

    /* Global consistency checks: */
    if (o->NF < 3) { argparser_error(pp, "too few scene images"); }
    if (o->NG < 1) { argparser_error(pp, "must specify at least one gauge"); }
    if (strlen(o->outPrefix) == 0) 
      { argparser_error(pp, "empty \"-outPrefix\""); }

    argparser_finish(pp);
    
    return o;
  }

gauge_opts_t ptn_parse_gauge_specs(argparser_t *pp, int NF)
  { 
    gauge_opts_t gop;
    /* Get nominal position {o.pos[NG]} of gauge: */
    double posx = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX);
    double posy = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX);
    gop.pos = (r2_t){{posx, posy}};

    /* Get name of gauge normal map: */
    { char *name = pst_parse_next_file_name(pp);
      if (name == NULL) 
        { argparser_error(pp, "missing filename of gauge normal map name"); }
      gop.normal = name;
    }
    
    /* Get names of gauge images under each light field: */
    assert(NF >= 0);
    gop.photos = pst_parse_file_name_list(pp, &(NF));
    
    return gop;
  }

vec_typeimpl(gauge_opts_vec_t,gauge_opts_vec,gauge_opts_t);


/* COPYRIGHT, AUTHORSHIP, AND WARRANTY NOTICE:
**
**   Copyright  2004 by the Fluminense Federal University (UFF).
**
** Created on jul/2005 by Rafael Saracchini, IC-UFF.
** Modified by Jorge Stolfi, mar/2006.
**
** Permission to use, copy, modify, and redistribute this software and
** its documentation for any purpose and without fee is hereby
** granted, provided that: (1) the copyright notice at the top of this
** file and this copyright, authorship, and warranty notice is retained
** in all derived source files and documentation; (2) no executable
** code derived from this file is published or distributed without the
** corresponding source code; and (3) these same rights are granted to
** any recipient of such code, under the same conditions.
** This software is provided "as is", WITHOUT ANY EXPLICIT OR IMPLICIT
** WARRANTIES, not even the implied warranties of merchantibility and
** fitness for a particular purpose. END OF NOTICE.
*/
