#include <camera_setup.h>
#include <camera_errors.h>
#include <camera_names.h>
#include <stdlib.h>
#include <string.h>

int determine_videomode(int format,int mode,int coding,int depth,dc1394video_mode_t* videomode,dc1394color_coding_t* colorcoding){
  /*coding:
    0 MONO
    1 RGB
    2 RAW
    3 YUV
  */
  *videomode = DC1394_VIDEO_MODE_MAX + 1;
  *colorcoding = DC1394_COLOR_CODING_MAX +1;
  if(format == 7){
    switch(mode){
      case 0: *videomode = DC1394_VIDEO_MODE_FORMAT7_0; break;
      case 1: *videomode = DC1394_VIDEO_MODE_FORMAT7_1; break;
      case 2: *videomode = DC1394_VIDEO_MODE_FORMAT7_2; break;
      case 3: *videomode = DC1394_VIDEO_MODE_FORMAT7_3; break;
      case 4: *videomode = DC1394_VIDEO_MODE_FORMAT7_4; break;
      case 5: *videomode = DC1394_VIDEO_MODE_FORMAT7_5; break;
      case 6: *videomode = DC1394_VIDEO_MODE_FORMAT7_6; break;
      case 7: *videomode = DC1394_VIDEO_MODE_FORMAT7_7; break;
      default:
	break;
    }
  }else{
    switch(format){
      case 0:
	if((coding == 3) && (depth == 444)){
	  *videomode = DC1394_VIDEO_MODE_160x120_YUV444;
	  *colorcoding = DC1394_COLOR_CODING_YUV444;
	}
	break;
      case 1:
	if((coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_320x240_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}
	break;
      case 2:
	if( (coding == 3) && (depth == 411) ){
	  *videomode = DC1394_VIDEO_MODE_640x480_YUV411;
	  *colorcoding = DC1394_COLOR_CODING_YUV411;
	}else if((coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_640x480_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}else if( (coding == 1) && (depth == 8)) {
	  *videomode = DC1394_VIDEO_MODE_640x480_RGB8;
	  *colorcoding = DC1394_COLOR_CODING_RGB8;
	}else if( (coding == 0) && (depth ==  8)){
	  *videomode = DC1394_VIDEO_MODE_640x480_MONO8;
	  *colorcoding = DC1394_COLOR_CODING_MONO8;
	}else if ((coding == 0) && (depth == 16) ){
	  *videomode = DC1394_VIDEO_MODE_640x480_MONO16;
	  *colorcoding = DC1394_COLOR_CODING_MONO16;
	}
	break;
      case 3:
	if( (coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_800x600_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}else if( (coding == 1) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_800x600_RGB8;
	  *colorcoding = DC1394_COLOR_CODING_RGB8;
	}else if( (coding == 0) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_800x600_MONO8;
	  *colorcoding = DC1394_COLOR_CODING_MONO8;
	}else if( (coding == 0) && (depth == 16) ){
	  *videomode = DC1394_VIDEO_MODE_800x600_MONO16;
	  *colorcoding = DC1394_COLOR_CODING_MONO16;
	}
	break;
      case 4:
	if( (coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_1024x768_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}else if( (coding == 1) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1024x768_RGB8;
	  *colorcoding = DC1394_COLOR_CODING_RGB8;
	}else if( (coding == 0) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1024x768_MONO8;
	  *colorcoding = DC1394_COLOR_CODING_MONO8;
	}else if( (coding == 0) && (depth == 16) ){
	  *videomode = DC1394_VIDEO_MODE_1024x768_MONO16;
	  *colorcoding = DC1394_COLOR_CODING_MONO16;
	}
	break;
      case 5:
	if( (coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_1280x960_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}else if( (coding == 1) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1280x960_RGB8;
	  *colorcoding = DC1394_COLOR_CODING_RGB8;
	}else if( (coding == 0) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1280x960_MONO8;
	  *colorcoding = DC1394_COLOR_CODING_MONO8;
	}else if( (coding == 0) && (depth == 16) ){
	  *videomode = DC1394_VIDEO_MODE_1280x960_MONO16;
	  *colorcoding = DC1394_COLOR_CODING_MONO16;
	}
	break;
      
      case 6:
	if( (coding == 3) && (depth == 422) ){
	  *videomode = DC1394_VIDEO_MODE_1600x1200_YUV422;
	  *colorcoding = DC1394_COLOR_CODING_YUV422;
	}else if( (coding == 1) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1600x1200_RGB8;
	  *colorcoding = DC1394_COLOR_CODING_RGB8;
	}else if( (coding == 0) && (depth == 8) ){
	  *videomode = DC1394_VIDEO_MODE_1600x1200_MONO8;
	  *colorcoding = DC1394_COLOR_CODING_MONO8;
	}else if( (coding == 0) && (depth == 16) ){
	  *videomode = DC1394_VIDEO_MODE_1600x1200_MONO16;
	  *colorcoding = DC1394_COLOR_CODING_MONO16;
	}
	break;
      default:
	break;
    }
  }
  if((*videomode) > DC1394_VIDEO_MODE_MAX ){
    return 0;
  }
  //if we have format 7 we need to determine the color coding
  if(format == 7){
    switch(coding){
      case 0:
	if(depth == 8){ *colorcoding = DC1394_COLOR_CODING_MONO8;}
	else if(depth == 16){ *colorcoding = DC1394_COLOR_CODING_MONO16;}
	break;
      case 1:
	if(depth == 8){ *colorcoding = DC1394_COLOR_CODING_RGB8;}
	else if(depth == 16){ *colorcoding = DC1394_COLOR_CODING_RGB16;}
	break;
      case 2:
	if(depth == 8){ *colorcoding = DC1394_COLOR_CODING_RAW8;}
	else if(depth == 16){ *colorcoding = DC1394_COLOR_CODING_RAW16;}
	break;
      case 3:
	if(depth == 411){ *colorcoding = DC1394_COLOR_CODING_YUV411;}
	else if(depth == 422){ *colorcoding = DC1394_COLOR_CODING_YUV422;}
	else if(depth == 444){ *colorcoding = DC1394_COLOR_CODING_YUV444;}
	break;
      default:
	break;
    }
  }
  if((*colorcoding) > DC1394_COLOR_CODING_MAX){
    return 0;
  }
  return 1;
}




dc1394_t* session_init(void){
  return dc1394_new();
}


dc1394error_t session_close(dc1394_t* session){
  dc1394_free (session);
  return DC1394_SUCCESS;
}

int session_get_num_devices(dc1394_t* session){
  dc1394camera_list_t* list;
  //dc1394error_t err = 
  dc1394_camera_enumerate (session, &(list));
  int  num_devices = list->num;
  dc1394_camera_free_list(list);
  return num_devices;
}

void session_list_devices(dc1394_t* session,FILE* arq){
  dc1394camera_list_t* list;
  //dc1394error_t err = 
  dc1394_camera_enumerate (session, &(list));
  int  num_devices = list->num;
  fprintf(arq,"Found %d devices:\n",num_devices);
  int i;
  for(i = 0; i < num_devices; i++){
    fprintf(arq,"Device [%d]\n GUID %llx UNIT %d\n",i,list->ids[i].guid,list->ids[i].unit);
  }
  dc1394_camera_free_list(list);
}

dc1394camera_t* camera_get_device(dc1394_t* session,  int deviceNum){
  dc1394camera_t* camera = NULL;
  dc1394camera_list_t* list;
  //dc1394error_t err = 
  dc1394_camera_enumerate (session, &(list));
  int  num_devices = list->num;
  if((deviceNum >= 0 ) && (deviceNum < num_devices)){
    camera = dc1394_camera_new (session, list->ids[deviceNum].guid);
  }
  dc1394_camera_free_list(list);
  return camera;
}

void camera_print_capabilities(dc1394camera_t* camera, FILE* arq){
  
  fprintf(arq,"------CAM-INFORMATIONS------\n");
  dc1394_camera_print_info(camera,arq);
  fprintf(arq,"------VIDEO-MODES------------\n");
  fprintf(arq,"Supported Modes: \n\n");
  int i;
  dc1394video_modes_t camera_modes;
  dc1394error_t err = dc1394_video_get_supported_modes(camera, &(camera_modes));
  for(i = 0; i < camera_modes.num; i++){
    dc1394video_mode_t video_mode = camera_modes.modes[i];
    uint32_t w,h;
    dc1394_get_image_size_from_video_mode(camera, video_mode, &w, &h);
    fprintf(arq,"%s - %dx%d ",camfirewire_video_mode_name(video_mode),w,h);
    if(!dc1394_is_video_mode_scalable(video_mode)){
      fprintf(arq,"fps (");
      dc1394framerates_t framerate_list;
      err = dc1394_video_get_supported_framerates(camera, video_mode,&framerate_list);
      int j;
      for( j= 0; j < framerate_list.num; j++){
	dc1394framerate_t framerate = framerate_list.framerates[j];
	float fps;
	err = dc1394_framerate_as_float(framerate,&fps);
	fprintf(arq," %3.2f ",fps);
      }
      fprintf(arq,")");
    }else{
      fprintf(arq," Scalable - Color Codings available:\n");
      dc1394color_codings_t codings;
      err =  dc1394_format7_get_color_codings(camera,video_mode, &(codings));
      int j;
      for( j= 0; j < codings.num; j++){
	fprintf(arq,"\t%s\n",camfirewire_color_coding_name(codings.codings[j]));
      }
    }
    fprintf(arq,"\n");
  }
  
  fprintf(arq,"------FEATURES---------------\n");
  dc1394featureset_t featureset;
  dc1394_feature_get_all(camera,&featureset);
  dc1394_feature_print_all(&featureset, arq);
  fprintf(arq,"-----------------------------\n");
  
}

dc1394error_t camera_release(dc1394camera_t* camera){
  dc1394_video_set_transmission(camera, DC1394_OFF);
  dc1394_capture_stop(camera);
  dc1394_camera_free(camera); 
  return DC1394_SUCCESS;
}

dc1394error_t session_reset_device(dc1394_t* session,int deviceNum){
  dc1394camera_list_t * list;
  dc1394error_t err=dc1394_camera_enumerate (session, &list);
  DC1394_ERR_RTN(err,"Failed to enumerate cameras");

  if((deviceNum < 0) || (deviceNum >= list->num ) ){
    fprintf(stderr,"Device %d is not valid within range [0,%d]\n",deviceNum,list->num -1);
    return DC1394_FAILURE;
  }
  
  dc1394camera_t* camera = dc1394_camera_new (session, list->ids[deviceNum].guid);
  
  if (!camera) {
    dc1394_log_error("Failed to initialize camera with guid %llx", list->ids[deviceNum].guid);
    return DC1394_FAILURE;
  }
  
  fprintf (stderr,"Using camera with GUID %llx: ", camera->guid);
  fprintf (stderr,"Reseting bus...");
  if (dc1394_reset_bus (camera) != DC1394_SUCCESS){
    fprintf (stderr,"ERROR\n");
  }else{
    fprintf (stderr,"OK\n");
  }
  dc1394_camera_free (camera);

  dc1394_camera_free_list (list);
  return DC1394_SUCCESS;
}


dc1394error_t session_reset_all_devices(dc1394_t* session){
  dc1394camera_list_t * list;
  dc1394error_t err=dc1394_camera_enumerate (session, &list);
  DC1394_ERR_RTN(err,"Failed to enumerate cameras");
  int i;
  for(i = 0; i < list->num; i++){
    dc1394camera_t* camera = dc1394_camera_new (session, list->ids[i].guid);
    if (!camera) {
      dc1394_log_error("Failed to initialize camera with guid %llx", list->ids[i].guid);
      return DC1394_FAILURE;
    }
   
    fprintf (stderr,"Using camera with GUID %llx: ", camera->guid);
    fprintf (stderr,"Reseting bus...");
    if (dc1394_reset_bus (camera) != DC1394_SUCCESS){
      fprintf (stderr,"ERROR\n");
    }else{
      fprintf (stderr,"OK\n");
    }
    dc1394_camera_free (camera);
  }
  dc1394_camera_free_list (list);
  return DC1394_SUCCESS;
}

dc1394error_t camera_auto_init(dc1394camera_t* camera){
  dc1394error_t err;
  
  /*Investigate the camera parameters*/
  dc1394video_modes_t video_modes;
  err = dc1394_video_get_supported_modes(camera,&video_modes);
  DC1394_ERR_RTN(err,"Could not discover video modes");
  dc1394video_mode_t selected_video_mode = video_modes.modes[0];
  
  dc1394framerates_t framerates;
  err = dc1394_video_get_supported_framerates(camera, selected_video_mode,&framerates);
  DC1394_ERR_RTN(err,"Could not discover framerates");
  dc1394framerate_t selected_framerate = framerates.framerates[0];
 
  /*Now sets all the parameters*/
  return camera_init(camera,
		      selected_video_mode,
		      DC1394_COLOR_CODING_MIN,
		      DC1394_ISO_SPEED_400,
		      selected_framerate,
		      NULL,
		      NULL);
  
}

dc1394error_t camera_init(dc1394camera_t* camera,
			  dc1394video_mode_t videomode,
			  dc1394color_coding_t colorcoding,
			  dc1394speed_t isospeed,
			  dc1394framerate_t framerate,
			  r2_t* roiDimensions,
			  r2_t* roiPosition
			  )
  {
  
  dc1394error_t err;
    
  err = dc1394_video_set_iso_speed(camera, isospeed);
  DC1394_ERR_RTN(err,"Could not set iso speed");
  
  err = dc1394_video_set_mode(camera, videomode);
  DC1394_ERR_RTN(err,"Could not set video mode");

  err = dc1394_video_set_framerate(camera, framerate);
  DC1394_ERR_RTN(err,"Could not set framerate");

  err = dc1394_capture_setup(camera,1, DC1394_CAPTURE_FLAGS_DEFAULT);
  DC1394_ERR_RTN(err,"Could not set capture flags");
  
  if((videomode >= DC1394_VIDEO_MODE_FORMAT7_0) && (videomode <= DC1394_VIDEO_MODE_FORMAT7_7) ){
    /*Format 7 specific options*/
   
    fprintf(stderr,"Selected Format is Scalable\n");
    
    uint32_t max_w,max_h;
    dc1394_get_image_size_from_video_mode(camera, videomode, &max_w, &max_h);
    DC1394_ERR_RTN(err,"Could not determine frame dimensions\n");
    
    uint32_t packet_size;
    int32_t left = 0;
    int32_t top = 0;
    int32_t width = max_w;
    int32_t height = max_h;
    
    err = dc1394_format7_get_recommended_packet_size(camera,videomode,&packet_size);
    
    DC1394_ERR_RTN(err,"Could not get packet size");
    
    if(roiDimensions != NULL){
      width = roiDimensions->c[0];
      height = roiDimensions->c[1];
    }
    if(roiPosition !=  NULL){
      left = roiPosition->c[0];
      top = roiPosition->c[1];
    }
    
    err =  dc1394_format7_set_roi(camera,videomode,colorcoding,packet_size,left,top,width,height);
    DC1394_ERR_RTN(err,"Could not set ROI");
   
    err = dc1394_format7_set_color_coding(camera, videomode,colorcoding);
    DC1394_ERR_RTN(err,"Could not set color coding\n");
    
  }
  
  return DC1394_SUCCESS;
}

void camera_print_settings(dc1394camera_t* camera, FILE* arq){
    fprintf(arq,"-----------------------\n");
    dc1394error_t err;
    
    dc1394framerate_t framerate;
    float fps;
    err = dc1394_video_get_framerate(camera, &framerate);
    err = dc1394_framerate_as_float(framerate,&fps);
    
    dc1394video_mode_t video_mode;
    uint32_t h,w;
    err = dc1394_video_get_mode(camera,&video_mode);
    dc1394_get_image_size_from_video_mode(camera, video_mode, &w, &h);
    
    dc1394operation_mode_t mode;
    err = dc1394_video_get_operation_mode(camera, &mode);
    
    dc1394speed_t speed;
    err = dc1394_video_get_iso_speed(camera,&speed);
    
    
    fprintf(arq,"DEVICE %llx UNIT: %d\n",camera->guid,camera->unit);
    fprintf(arq,"ISO SPEED: %s \n",camfirewire_iso_speed_name(speed));
    fprintf(arq,"VIDEO MODE: %s (%dx%d)\n",camfirewire_video_mode_name(video_mode),w,h);
    fprintf(arq,"FRAMERATE: %3.2f \n",fps);
    fprintf(arq,"OPERATION MODE: %s \n",camfirewire_operation_mode_name(mode));
    if(dc1394_is_video_mode_scalable(video_mode)){
       dc1394color_coding_t coding;
       dc1394_format7_get_color_coding(camera,video_mode,&coding);
       fprintf(arq,"COLOR CODING: %s\n",camfirewire_color_coding_name(coding));
     }
    fprintf(arq,"-----------------------\n");
}

dc1394error_t camera_copy_frame(dc1394video_frame_t* orig,dc1394video_frame_t* dest){
  *dest = *orig;
  uint64_t total_bytes = orig->total_bytes;
  unsigned char * image = (unsigned char *)malloc(sizeof(unsigned char)* total_bytes);
  dest->image = image;
  memcpy (dest->image, orig->image,total_bytes);
  return DC1394_SUCCESS;
}

dc1394error_t camera_flush_DMA(dc1394camera_t* camera){
  dc1394video_frame_t* frame =  calloc(1,sizeof(dc1394video_frame_t)) ;
  int count = 0;
  
  do{
   dc1394error_t  err=dc1394_capture_dequeue(camera, DC1394_CAPTURE_POLICY_POLL,&frame);
     DC1394_ERR_RTN(err,"Could not dequeue frame");
    count++;
  }while(frame != NULL);
//  dc1394error_t  err=dc1394_capture_dequeue(camera, DC1394_CAPTURE_POLICY_POLL,&frame);//      DC1394_ERR_RTN(err,"Could not dequeue frame");
 // DC1394_ERR_RTN(err,"Could not dequeue frame");
  //fprintf(stderr,"Flushed %d frames from DMA\n",1);
  free(frame);
  return DC1394_SUCCESS; 
}