#include <stdio.h>
#include <stdlib.h>
#include <argparser.h>
#include <jsfile.h>
#include <float_image.h>
#include <limits.h>

#define PROG_NAME "fni_cut"

#define PROG_HELP \
  PROG_NAME " \\\n" \
  "    -top {NUM} \\\n" \
  "    -left {NUM} \\\n" \
  "    -width {NUM} \\\n" \
  "    -height {NUM} \\\n" \
  "    -right {NUM} \\\n" \
  "    -bottom {NUM} \\\n" \
  "    [-in {INPUT FILE}] \\\n" \
  "    [-out {OUTPUT FILE}] \\\n" \
  "    " argparser_help_info_HELP ""

#define PROG_INFO \
  "NAME\n" \
  "  Etc. etc..\n" \
  "\n" \
  "SYNOPSIS\n" \
  PROG_HELP "\n" \
  "\n" \
  "OPTIONS" \
  "  Etc. etc.."


struct options_t{
  int width;
  int height;
  int top;
  int left;
  int bottom;
  int right;
  bool_t flip_y;
  char* inputFNI;
  char* outputFNI;
};

typedef struct options_t options_t;

float_image_t* cutFNI(float_image_t* img, int left, int top, int width, int height);
/*Cut and new FNI image from {img} given the cutting coordinates */

float_image_t* cutFNI(float_image_t* img, int left, int top, int width, int height){
  int iNX,iNY,iNC;
  iNC = img->sz[0];
  iNX = img->sz[1];
  iNY = img->sz[2];

  int NC,NX,NY;
  NC = iNC;
  NX = width;
  NY = height;
  float_image_t* novo = float_image_new(NC,NX,NY);
    
  int c,x,y;
  for(x = 0; x < NX; x++){
    for(y = 0; y < NY; y++){
      for(c = 0; c < NC; c++){
	int ix,iy;
	ix = x+left;
	iy = y+top;
	if( (ix >= iNX) || (iy >= iNY)) continue;
	float val = float_image_get_sample(img,c,ix,iy);
	float_image_set_sample(novo,c,x,y,val);
      }
    }
  }

  return novo;
}

options_t* parse_args(int argc, char** argv);
/*Parse the arguments*/

options_t* parse_args(int argc, char** argv){
  options_t* o = (options_t*)malloc(sizeof(options_t));
  argparser_t *pp = argparser_new(stderr, argc, argv);
  argparser_set_help(pp, PROG_HELP);
  argparser_set_info(pp, PROG_INFO);
  argparser_process_help_info_options(pp);
  o->width = -1;
  o->height = -1;
  o->top = -1;
  o->left = -1;
  o->bottom = -1;
  o->right = -1;

  if (argparser_keyword_present(pp, "-top")) {
    		o->top = argparser_get_next_int(pp, 0, INT_MAX);
  }
  if (argparser_keyword_present(pp, "-left")) {
    		o->left = argparser_get_next_int(pp, 0, INT_MAX);
  }
  if (argparser_keyword_present(pp, "-width")) {
    		o->width = argparser_get_next_int(pp, 0, INT_MAX);
  }
  if (argparser_keyword_present(pp, "-height")) {
    		o->height = argparser_get_next_int(pp, 0, INT_MAX);
  }

  if (argparser_keyword_present(pp, "-right")) {
    		o->right = argparser_get_next_int(pp, 0, INT_MAX);
		if((o->width == -1) && (o->left != -1) ){
		  o->width = o->right - o->left;
		}
		if((o->width != -1) && (o->left == -1) ){
		  o->left = o->right - o->width;
		}
  }
  if (argparser_keyword_present(pp, "-bottom")) {
    		o->bottom = argparser_get_next_int(pp, 0, INT_MAX);
		if((o->height == -1) && (o->top != -1) ){
		  o->height= o->bottom - o->top;
		}
		if((o->height != -1) && (o->top == -1) ){
		  o->top = o->bottom - o->height;
		}
  }

  fprintf(stderr,"Cutting region will be %dx%d+%d+%d\n",o->width,o->height,o->left,o->top);

  if((o->left < 0) || (o->top < 0) || (o->width < 0) || (o->height < 0) ){
    argparser_error(pp, "Must specify good coordinates to perform a cut !");
  }

  o->inputFNI = NULL;
  if (argparser_keyword_present(pp, "-in")) {
    o->inputFNI =  argparser_get_next(pp);
  }else{
    fprintf(stderr,"Input will be read from standard in\n");
  }
  
  o->outputFNI = NULL;
  if (argparser_keyword_present(pp, "-out")) {
    o->outputFNI =  argparser_get_next(pp);
  }else{
    fprintf(stderr,"Output will be written in standard out\n");
  }

  argparser_finish(pp);
  return o;
}

float_image_t* ReadFNI(char* filename);
/*Read an FNI file, it sets to read from stdin if no filename is specified*/
float_image_t* ReadFNI(char* filename){
  FILE* infile = stdin;
  if(filename != NULL) infile = open_read(filename,TRUE);
  float_image_t* img = float_image_read(infile);
  if(filename != NULL) fclose(infile);
  return img;
}

void WriteFNI(char* filename, float_image_t* img);
/*writes and FNi file, it writes into stdout if no filename is specified*/

void WriteFNI(char* filename, float_image_t* img){
  FILE* outfile = stdout;
  if(filename != NULL) outfile = open_write(filename,TRUE);
  float_image_write(outfile,img);
  if(filename != NULL) fclose(outfile);
}

int main(int argc, char** argv){
  options_t* o = parse_args(argc,argv);

  float_image_t* in_img = ReadFNI(o->inputFNI);
  float_image_t* out_img = cutFNI(in_img,o->left,o->top,o->width,o->height);
  WriteFNI(o->outputFNI,out_img);
  float_image_free(in_img);
  float_image_free(out_img);
  return 0;
}
