/*
Implementation of the method descrbed in the article
Surface Height Recovery from Surface Normals using Manifold Embedding
by Antonio Robles-Kelly and Edwin R. Hancock
published at 2004 in IEEE ????
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float_image.h>
#include <jsfile.h>

struct normal_point_t{
	double x,y;
	double nx,ny,nz;
};
typedef struct normal_point_t normal_point;

double h(normal_point si,normal_point sj, int dx,int dy);
double d(float_image_t* normal_map,normal_point sk,normal_point sl);
normal_point getNormalPoint(float_image_t* im,int x,int y);
float_image_t* compute_heights(float_image_t* normal_map);
float_image_t* expand_map(float_image_t* in_map);

double h(normal_point si,normal_point sj, int dx,int dy){
	//this is the euclidean distance between two points
// 	double r = sqrt( ((si.x-sj.x)*(si.x-sj.x)) + ((si.y-sj.y)*(si.y-sj.y))  );
	double hr;
	if( abs(dx) == abs(dy) ){
		double rx = sj.x-si.x;
		double ry = sj.y-si.y;
	// 	if(r == 0) return 0;
		double hx = (rx/2.0)*((si.nx/si.nz) + (sj.nx/sj.nz));
		double hy = (ry/2.0)*((si.ny/si.nz) + (sj.ny/sj.nz));
		//double wx = (sj.x - si.x)/r;
		//double wy = (sj.y - si.y)/r;
		hr = hx + hy;
	}else if( abs(dx) == 1){
		double rx = sj.x-si.x;
		double hx = (rx)*((si.nx/si.nz) + (sj.nx/sj.nz));
		hr = hx;
	}else if(abs(dy) == 1){
		double ry = sj.y-si.y;
		double hy = (ry)*((si.ny/si.nz) + (sj.ny/sj.nz));
		hr = hy;
	}
	return hr;
}

double d(float_image_t* normal_map,normal_point sk,normal_point sl){
	//we need find the geodesic path
	int distx,disty;
	distx = sl.x - sk.x;
	disty = sl.y - sk.y;
	int dirx,diry;
	dirx = 1;
	double res = 0;
	if(distx < 0) dirx = -1;
	if(distx == 0) dirx = 0;
	diry = 1;
	if(disty < 0) diry = -1;
	if(disty == 0) diry = 0;
	int i;
	normal_point pi = sk;
	//printf("%lf %lf\n",sl.x,sl.y);
	if ( (distx == 0) && (disty == 0) ) return 0;
	if(abs(distx) > abs(disty)){
	//	puts("path X");
		for(i = 1; i <= (distx*dirx); i++){
			int px,py;
			double a,b;
			a = (sl.y - sk.y)/(float)(sl.x - sk.x);
			b = sl.y - (a*sl.x);
			px = (i*dirx) + sk.x;
			py = round(a*px + b);
			normal_point pj = getNormalPoint(normal_map,px,py);
			int dx,dy;
			dx = pj.x - pi.x;
			dy = pj.y - pi.y;
			res+=h(pi,pj,dx,dy);
		//	printf("%f %f\n",pi.x,pi.y);
			pi = pj;
		}
	}else{
	//	puts("path Y");
		for(i = 1; i <= (disty*diry); i++){
			int px,py;
			double a,b;
			a = (sl.x - sk.x)/(float)(sl.y - sk.y);
			b = sl.x - (a*sl.y);
			py = (i*diry) + sk.y;
			px = round(a*py + b);
			normal_point pj = getNormalPoint(normal_map,px,py);
			int dx,dy;
			dx = pj.x - pi.x;
			dy = pj.y - pi.y;
			res+=h(pi,pj,dx,dy);
			//printf("%f %f\n",pi.x,pi.y);
			pi = pj;
		}
	}
//	printf("%f %f\n",sl.x,sl.y);
	return res;
	
	
}


normal_point getNormalPoint(float_image_t* im,int x,int y){
	normal_point pt;
	pt.x = x;
	pt.y = y;
	/*pt.nx = im->pixel[y][x].x;*/
	pt.nx = float_image_get_sample(im,0,x,y);
	/*pt.ny = im->pixel[y][x].y;*/
	pt.ny = float_image_get_sample(im,1,x,y);
	/*pt.nz = im->pixel[y][x].z;*/
	pt.nz = float_image_get_sample(im,2,x,y);
	return pt;
}

float_image_t* compute_heights(float_image_t* normal_map){
	float_image_t* height_map;
	int nx,ny;
	nx = normal_map->sz[1];
	ny = normal_map->sz[2];
	height_map =float_image_new(1,nx,ny);
	int x,y;
	double N;
	
	N = nx*ny;
	fprintf(stderr,"%dx%d -> %f\n",nx,ny,N);
	int total;
	normal_point s1,si,sl;
	s1 = getNormalPoint(normal_map,0,0);
	total = 0;
	fprintf(stderr,"\n");
	for(x = 0; x < nx; x++){
		for(y = 0; y < ny; y++){
			fprintf(stderr,"\033[1A");
			fprintf(stderr,"%04d of %04d  %6.6f%%\n",total,nx*ny,total*100/N);
			int lx,ly;
			si = getNormalPoint(normal_map,x,y);
			double sum1,sum2;
			sum1 = 0;
			sum2 = 0;
			for(lx = 0 ; lx < nx; lx++){
				for(ly = 0; ly < ny; ly++){
					sl = getNormalPoint(normal_map,lx,ly);
					sum1 += d(normal_map,si,sl);
					sum2 += d(normal_map,s1,sl);
				}
			}
			double height = (1/N)*(sum1-sum2);
			//height_map->pixel[y][x] = height;	
			float_image_set_sample(height_map,0,x,y,height);
			total++;
		}
		
	}
	return height_map;
}

float_image_t* expand_map(float_image_t* in_map){
	float_image_t* out_map;
	int NX = in_map->sz[1];
	int NY = in_map->sz[2];
	out_map = float_image_new(1,NX+1,NY+1);
	int x,y;
	int nx,ny;
	nx = NX+1;
	ny = NY+1;
	puts("STARTING EXPAND\n");
	for(x = 0;x < nx;x++){
		for(y = 0; y < ny; y++){
			double sum = 0;
			int tot = 0;
			fprintf(stderr,"\033[1A");
			fprintf(stderr,"[%04d %04d]\n",x,y);
			int i,j;
			for(i = x-1; i <= x; i++){
				for(j = y-1;j<= y; j++){
					if(i < 0) continue;
					if(j < 0) continue;
					if(i >= (nx-1)) continue;
					if(j >= (ny-1)) continue;
					//sum+=in_map->pixel[j][i];
					sum+= float_image_get_sample(in_map,0,i,j);
					tot++;
				}
			}
			
			//out_map->pixel[y][x] = sum/(float)tot;
			float_image_set_sample(out_map,0,x,y,sum/(float)tot);
		}
	}
	puts("FOI");
	return out_map;
}

float_image_t* load_fni_file(char* prefix,char* suffix);
float_image_t* load_fni_file(char* prefix,char* suffix){
	char* nome_arq = NULL;
	asprintf(&nome_arq,"%s_%s.fni",prefix,suffix);
	FILE* arq = open_read(nome_arq,TRUE);
	float_image_t* fi = float_image_read(arq);
	free(nome_arq);
	fclose(arq);
	return fi;
	
}
void save_fni_file(char* prefix,char* suffix, float_image_t* fo);
void save_fni_file(char* prefix,char* suffix, float_image_t* fo){
	char* nome_arq = NULL;
	asprintf(&nome_arq,"%s_%s.fni",prefix,suffix);
	FILE* arq = open_write(nome_arq,TRUE);
	float_image_write(arq,fo);
	free(nome_arq);
	fclose(arq);
}

int main(int argc, char** argv){
	if(argc != 3){
		fprintf(stderr,"Usage:\nkelly_hancockn <input-prefix> <output-prefix>\n");
		return 1;
	}
	char* intag = argv[1];
	char* outtag = argv[2];
	float_image_t* normal_map;
	float_image_t* height_map;
	float_image_t* height_map2;
	normal_map = load_fni_file(intag,"normals");
	height_map = compute_heights(normal_map);
	puts("Computed heights");
	height_map2 = expand_map(height_map);
	save_fni_file(outtag,"alturas",height_map2);
	return 0;
}
