#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <r3.h>
#include <assert.h>
#include <argparser.h>
#include <float_image.h>
#include <jsfile.h>


#define PROG_NAME "smith_bors"

#define PROG_HELP \
  PROG_NAME " \\\n" \
  "    -input {INTAG} \\\n" \
  "    -output {OUTTAG} \\\n" \
  "    -max_iter {MAX_ITER} \\\n" \
  "    -eta0 {ETA0} \\\n" \
  "    -eps {EPS} \\\n" \
  "    " argparser_help_info_HELP ""

#define PROG_INFO \
  "NAME\n" \
  "  Etc. etc..\n" \
  "\n" \
  "SYNOPSIS\n" \
  PROG_HELP "\n" \
  "\n" \
  "OPTIONS" \
  "  Etc. etc.."



struct options_t{
	char* intag;
	char* outtag;
	double eps;
	int max_iter;
	double eta0;
};
typedef struct options_t options_t;


double MCE(float_image_t* normal_map, float_image_t* height_map);
double MCE(float_image_t* normal_map, float_image_t* height_map){
	int NX,NY;
	NX = normal_map->sz[1];
	NY = normal_map->sz[2];
	int i,j;
	double sum= 0.0;
	for(i = 0; i < NX; i++){
		for(j = 0; j < NY; j++){
			r3_t N,Nhat;
			N = (r3_t){{float_image_get_sample(normal_map,0,i,j),float_image_get_sample(normal_map,1,i,j),float_image_get_sample(normal_map,2,i,j)}};
			if(i != 0){
				Nhat.c[0] = float_image_get_sample(height_map,0,i-1,j) - float_image_get_sample(height_map,0,i,j);
			}else{
				Nhat.c[0] = float_image_get_sample(height_map,0,i,j);
			}
			if(j != 0){
				Nhat.c[1] = float_image_get_sample(height_map,0,i,j-1) - float_image_get_sample(height_map,0,i,j);
			}else{
				Nhat.c[1] = float_image_get_sample(height_map,0,i,j);
			}
			Nhat.c[2] =  1;
			r3_dir(&Nhat,&Nhat);
			//assert( !isnan(Nhat.c[2]));
			sum+= r3_dot(&N,&Nhat);
		}
	}
	return sum/(float)(NX*NY);
}


void updateHeight(float_image_t* height_map, int x, int y, double delta, double eta);
void updateHeight(float_image_t* height_map, int x, int y, double delta, double eta){
	int NX = height_map->sz[1];
	int NY = height_map->sz[2];
	if( (x >= 0 ) && (x < NX) && (y >=0 ) && (y < NY)){
		double height = float_image_get_sample(height_map,0,x,y);
		height+= eta*delta;
		float_image_set_sample(height_map,0,x,y,height);
	}
}

double getDerivativeComponent(float_image_t* normal_map,int x, int y, int c);
double getDerivativeComponent(float_image_t* normal_map,int x, int y, int c){	
	int NX = normal_map->sz[1];
	int NY = normal_map->sz[2];
	if(x < 0) x = -x -1;
	if(x >= NX) x = (2*NX) -1 -x;
	if(y < 0) y = -y -1;
	if(y >= NY) y = (2*NY) -1 -y;
	
		return float_image_get_sample(normal_map,c,x,y)/float_image_get_sample(normal_map,2,x,y);
	
}

double getAvgDerivativeComponent(float_image_t* normal_map,int xa, int ya, int xb, int yb, int c);
double getAvgDerivativeComponent(float_image_t* normal_map,int xa, int ya, int xb, int yb, int c){

	return 0.5*getDerivativeComponent(normal_map,xa,ya,c) + 0.5*getDerivativeComponent(normal_map,xb,yb,c);
}

float_image_t* compute_heights(float_image_t* normal_map,double eps, int max_iter, double eta0);
float_image_t* compute_heights(float_image_t* normal_map,double eps, int max_iter, double eta0){
	int NX = normal_map->sz[1];
	int NY = normal_map->sz[2];
	bool_t original = FALSE;
	float_image_t* height_map = float_image_new(1,NX+1,NY+1);
	int x,y;
	for(x = 0; x <= NX; x++){
		for(y = 0; y <= NY; y++){
			float_image_set_sample(height_map,0,x,y,0);
		}
	}
	int iter = 1;
	double mce_error = -1;
	while( (iter <= max_iter)  && (fabs(1 - mce_error) > eps) ){
		float_image_set_sample(height_map,0,0,0,0);
		
		double eta = eta0/(1 + (iter/(float)max_iter));
		for(x = 0; x <= NX; x++){
			for(y = 0; y <= NY; y++){
				//update north
				double dzym;
				double dzxp;
				double dzxm;
				double dzyp;
				
				if(original){
					dzxm = getDerivativeComponent(normal_map,x-1, y-1,0);
					dzxp = getDerivativeComponent(normal_map,x,y-1,0);
					dzym = getDerivativeComponent(normal_map,x-1,y-1,1);
					dzyp = getDerivativeComponent(normal_map,x-1, y,1);
					
					
				}else{
					dzxm = getAvgDerivativeComponent(normal_map,x-1 , y-1 ,  x-1, y ,  0);
					dzxp = getAvgDerivativeComponent(normal_map,x   , y-1 ,  x,   y ,  0);
					dzym = getAvgDerivativeComponent(normal_map,x-1 , y-1 ,  x,   y-1, 1);
					dzyp = getAvgDerivativeComponent(normal_map,x-1 , y   ,  x,   y  , 1);	
				}
				updateHeight(height_map,x,y+1,+dzyp,eta);
				updateHeight(height_map,x+1,y,+dzxp,eta);
				//updateHeight(height_map,x-1,y,-dzxm,eta);
				//updateHeight(height_map,x,y-1,-dzym,eta);
				
			}
		}
		
		//double mean = float_image_get_sample(height_map,0,0,1) + float_image_get_sample(height_map,0,1,0) + float_image_get_sample(height_map,0,1,1);
		//mean = mean/3.0;
		//float_image_set_sample(height_map,0,0,0,mean);
		mce_error = MCE(normal_map, height_map);
		fprintf(stderr,"\033[1A");
		fprintf(stderr,"ITER %06d of %06d - MCE %10.8f\n",iter,max_iter,mce_error);
		iter++;
	}
	if( (iter <= max_iter)) puts ("foi 2");
	if( (fabs(1 - mce_error) > eps) ) puts("foi 1");

	return height_map;
}



float_image_t* load_fni_file(char* prefix,char* suffix);
float_image_t* load_fni_file(char* prefix,char* suffix){
	char* nome_arq = NULL;
	asprintf(&nome_arq,"%s_%s.fni",prefix,suffix);
	FILE* arq = open_read(nome_arq,TRUE);
	float_image_t* fi = float_image_read(arq);
	free(nome_arq);
	fclose(arq);
	return fi;
	
}
void save_fni_file(char* prefix,char* suffix, float_image_t* fo);
void save_fni_file(char* prefix,char* suffix, float_image_t* fo){
	char* nome_arq = NULL;
	asprintf(&nome_arq,"%s_%s.fni",prefix,suffix);
	FILE* arq = open_write(nome_arq,TRUE);
	float_image_write(arq,fo);
	free(nome_arq);
	fclose(arq);
}

options_t *parse_options(int argc, char **argv);
options_t *parse_options(int argc, char **argv)
{
	argparser_t *pp = argparser_new(stderr, argc, argv);
	argparser_set_help(pp, PROG_HELP);
  	argparser_set_info(pp, PROG_INFO);
  	argparser_process_help_info_options(pp);
	
	options_t* o = (options_t*)malloc(sizeof(options_t));	
	argparser_get_keyword(pp,"-input");
	o->intag = argparser_get_next(pp);
	argparser_get_keyword(pp,"-output");
	o->outtag = argparser_get_next(pp);
	argparser_get_keyword(pp,"-max_iter");
	o->max_iter = argparser_get_next_int(pp,0,100000);
	argparser_get_keyword(pp,"-eta0");
	o->eta0 = argparser_get_next_double(pp,0,100000);
	argparser_get_keyword(pp,"-eps");
	o->eps = argparser_get_next_double(pp,0,100000);
	return o;
}

int main(int argc, char** argv){
	
	options_t* o = parse_options(argc,argv);
	float_image_t* normal_map;
	float_image_t* height_map;
	normal_map = load_fni_file(o->intag,"normals");
	height_map = compute_heights(normal_map,o->eps,o->max_iter,o->eta0);
	puts("Computed heights");
	save_fni_file(o->outtag,"alturas",height_map);
	return 0;
}
