#! /usr/bin/python3
# Last edited on 2024-09-19 08:38:28 by stolfi

from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_box_vertices import make_vertices
from slicing_box_faces_orig import make_edges_faces_orig
from slicing_box_faces_mono import monotonize
from slicing_box_faces_convex import convexify
from slicing_box_faces_triang import triangulate
from slicing_lib import min_edge_length, min_triangle_width

import OBJ_format
import POV_format
 
# The object is a hollow box without top or bottom, with round
# indentations at the top and bottom of each side wall.  Its shape is
# determined, as described below, by the following parameters:
#
#   {Ni}       Half the number of facets in each indentation.
#   {S_box}    Half of {X} and {Y} extend of box.
#   {H_box}    Hlaf of {Z} extent of box.
#   {T_box}    Box wall thickness.
#   {H_den}    Half-width of indentations.
#
# The the box has a basic square horizontal cross-section, with chamfers
# at the vertical corners (so the cross-section is actually two nested
# octagons). The outer surface of the box spans {[-S_box _ +S_box]} in
# {X} and {Y}, and {[-H_box _ +H_box]} in {Z}. The inner surface has the
# same {Z} range but the {X} and {Y} ranges are reduced by the wall
# thickness {T_box} at either end.
#
# The walls of the box are identified an index {ks} in {0..3} in CCW
# order seen from {+Z}. An inner/outer bit {kr} in {0,1} identifies the
# two sides of each wall.
#
# The object is symmetric in that the shape of wall {ks+1} (modulo 4) is
# that if wall {ks}, rotated +90 degrees around the {Z}-axis. Each wall
# is also symmetric in the other two direction: specifically, wall 0 is
# symmetric under negation of the {Y} and/or {Z} coordinates.
#
# The indentation on the top or bottom of each wall is a chain of
# {2*Ni} rectangular faces, making {2*Ni-1} concave dihedral angles,
# spanning the thickness of the wall. On wall 0, these indentation faces
# are parallel to the {X} axis. The {2*Ni+1} vertices of these faces
# that lie on the outer surface of the wall lie on an arc of a circle whose
# chord is horizontal, at {Z} coordinate {±H_box}. The half-length of
# that chord is {H_den}. Indentation faces are identified
# by the wall index {ks}, an additional lowe/upper bit {kz}, and a face
# index {kf} in {0..2*Ni-1}.
#
# The chamfer along each vertical edge of the box, inner or outer, makes
# angles of 45 degrees with the adjacent wall surfaces. The depth of the
# chamfer is such that it touches the extreme vertices of the bottom and
# top indentations.
#
# Main faces on the walls of the object (the /plazas/) consist therefore
# of two chains of {2*Ni} edges, along the indentations, connected by two
# vertical edges of length {2*H_box}.
#
# COMMAND LINE
#
# The program takes two parameters from the command line: {Tag} and {Ni}.
# The other parameters are defined internally.

def main():
  Tag = sys.argv[1]
  Ni = int(sys.argv[2])
  
  S_box = 40
  T_box = 12
  H_den = 26
  H_box = 50
  prec = 4 if Ni < 10000 else 5 if Ni < 100000 else 6
  
  Vind, Vlst = make_vertices(Ni, S_box, H_box, T_box, H_den, prec, verbose = False)
  Elst, Flst = make_edges_faces_orig(Ni, Vind, Vlst, prec, verbose = False)
  Emon, Fmon = monotonize(Ni, Vind, Vlst, Elst, Flst, prec, verbose = False)
  Ecvx, Fcvx = convexify(Ni, Vind, Vlst, Elst, Flst, prec, verbose = False)
  Etri, Ftri = triangulate(Ni, Vind, Vlst, Elst, Flst, prec, verbose = False)
  
  dmin = min_edge_length(Vlst, Etri)
  sys.stderr.write("min edge length = %.16f\n" % dmin)
   
  hmin = min_triangle_width(Vlst, Ftri)
  sys.stderr.write("min triangle width = %.16f\n" % hmin)
 
  opref = f"out/{Tag}_{Ni:08d}"
  OBJ_format.write_object(opref + "_orig.obj",   Vlst, Elst, Flst, prec)
  OBJ_format.write_object(opref + "_mono.obj",   Vlst, Emon, Fmon, prec)
  OBJ_format.write_object(opref + "_convex.obj", Vlst, Ecvx, Fcvx, prec)
  OBJ_format.write_object(opref + "_triang.obj", Vlst, Etri, Ftri, prec)

  if Ni < 20:
    Np = 0; Zp_min = -1; Zp_max = +1
    emag = 1.0
    vmag = 1.0
    POV_format.write_object(opref + "_orig.inc",   Vlst, Elst, Flst, Np, Zp_min, Zp_max, emag, vmag, prec)
    POV_format.write_object(opref + "_mono.inc",   Vlst, Emon, Fmon, Np, Zp_min, Zp_max, emag, vmag, prec)
    POV_format.write_object(opref + "_convex.inc", Vlst, Ecvx, Fcvx, Np, Zp_min, Zp_max, emag, vmag, prec)
    POV_format.write_object(opref + "_triang.inc", Vlst, Etri, Ftri, Np, Zp_min, Zp_max, emag, vmag, prec)

  return 0

main()  

 
