#! /usr/bin/python3
# Last edited on 2024-09-18 22:43:09 by stolfi
 
from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_lib import make_edge, make_face, check_repeated_edge, prte, prtf

def convexify(Ni, Vind, Vlst, Elst, Flst, prec, verbose):
  # See the main program and {make_vertices} for the description of the object.
  #
  # Subdivides the faces of the original mesh of the object into
  # convex polygons. Assumes that the vertices are given by the structured
  # table {Vind} and flat list {Vlst}, the edges are given by the flat
  # list {Elst}, and the faces are given by the flat list {Flst}. The
  # lists {Vlst,Elst,Flst} are indexed from 1; element 0 is not used.
  #
  # Returns new flat lists {Fcvx} and {Ecvx} of faces and edges
  # of the convex refinement of the mesh.  No new vertices are created. The
  # original edges are preserved with their original indices,
  # and the new diagonal edges are appended after them, with {etype=1}.
  # The faces are replaced by the convex polygons.
  #
  # Original faces "fu.{ki}" (upper chain), and "fs.{ky}" (stoke faces) are
  # preserved, with the same labels, since they are convex. Each plaza
  # "fp.{kx}" is split into {Ni} triangles, with labels "fp.{kx}.{ki}" 
  # where {ki} is in {0..Ni-1}.
  # 
  # The diagonals added across the plazas will have labels
  # "dp.{kx}.{ki}" where {ki} ranges in {1..Ni-1}.

  Nv_in = len(Vlst)-1  # Total vertex count (input and output).
  Ne_in = len(Elst)-1  # Total input edge count.
  Nf_in = len(Flst)-1  # Total input faces.
  
  Nv_ot = 2*(Ni+2)   # Expected output vertex count.
  Ne_ot = 5*Ni + 4   # Expected output edge count.
  Nf_ot = 3*Ni + 2   # Expected output face (convex polygon) count.
  
  sys.stderr.write("Expecting %d edges\n" % Ne_ot)
  sys.stderr.write("Expecting %d faces\n" % Nf_ot)

  assert Nv_ot == Nv_in, "{Ni,Vlst} inconsistent"

  Fcvx = [None]*(Nf_ot+1)
  Ecvx = Elst.copy() + [None]*(Ne_ot-Ne_in)
  
  Eset = set()  # Set of edges as pairs, to check for repetitions.
  for je in range(1,Ne_in+1): 
    e = Elst[je]
    Eset.add((e[0],e[1]))
  
  ne = Ne_in # Number of edges created so far.
  nf = 0     # Number of faces created so far.
 
  def Svdi(kv_org, kv_dst, elab):
    # Adds the diagonal edge from {Vlst[kv_org]} to {Vlst[kv_dst]} to {Ecvx},
    # reoriented in increasing index sense.
    nonlocal ne, nf
    if kv_org > kv_dst: kv_org, kv_dst = kv_dst, kv_org
    etype = 1
    e = make_edge(Vlst[kv_org], Vlst[kv_dst], ne+1, etype, elab, prec)
    assert e != None
    if verbose: prte(e, prec)
    ne += 1
    Ecvx[ne] = e
    check_repeated_edge(ne, Ecvx, Eset, prec)
    return None
    
  def Cvx(Civ,flab):
    # Adds to {Ecvx} the convex face with vertices
    # {Vlst[Civ[0..deg-1]]}, where {deg}= len(Civ)},
    # assumed to be CCW seen from outside.
    nonlocal ne, nf
    nf += 1
    t = make_face(Civ, nf, flab, Vlst, prec)
    if verbose: prtf(t, Vlst, prec)
    Fcvx[nf] = t

  for f in Flst[1:]:
    Nx,Ny,Nz,Fiv,kf,flab = f;
    if verbose: 
      sys.stderr.write("\n")
      sys.stderr.write("  IN: ")
      prtf(f, Vlst, prec)
      sys.stderr.write("\n")
    
    if flab[0:2] == "fu" or flab[0:2] == "fs":
      # Upper chain or spoke face:
      assert len(Fiv) == 4
      Cvx(Fiv,flab)
    elif flab[0:2] == "fp":
      # Plaza face:
      assert len(Fiv) == Ni+2;
      kx = int(flab[3])
      assert kx == 0 or kx == 1
      for ki in range(Ni):
        if verbose: sys.stderr.write("\n")
        kva = Vind['vu'][kx][ki]
        kvb = Vind['vu'][kx][ki+1]
        kvc = Vind['vc'][kx]
        if kx == 0:
          Cvx((kva,kvb,kvc), flab + f".{ki}")
        else:
          Cvx((kvc,kvb,kva), flab + f".{ki}")
        if ki < Ni-1:
          Svdi(kvb,kvc,"d" + flab[1:] + f".{ki+1}")
    else:
      assert False, f"invalid face label '{flab}'"

  if nf < Nf_ot: sys.stderr.write("!! missing some faces\n")
  assert nf <= Nf_ot

  if ne < Ne_ot: sys.stderr.write("!! missing some edges\n")
  assert ne <= Ne_ot

  return Ecvx, Fcvx
