#! /usr/bin/python3
# Last edited on 2024-09-18 22:43:04 by stolfi
 
from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_lib import make_edge, make_face, check_repeated_edge, prte, prtf

def triangulate(Ni, Vind, Vlst, Elst, Flst, prec, verbose):
  # See the main program and {make_vertices} for the description of the object.
  #
  # Subdivides the faces of the original mesh of the object Into
  # triangles. Assumes that the vertices are given by the structured
  # table {Vind} and flat list {Vlst}, the edges are given by the flat
  # list {Elst},and the faces are given by the flat list {Flst}. The
  # lists {Vlst,Elst,Flst} are indexed from 1; element 0 is not used.
  #
  # Returns new flat lists {Ftri} and {Etri} of faces and edges
  # of the triangulation.  No new vertices are created. The
  # original edges are preserved with their original indices,
  # and the new diagonal edges are appended after them, with {etype=1}.
  # The faces are replaced by the triangles.
  #
  # Original faces "fu.{ki}" (upper chain) and "fs.{ky}" (stokes) are
  # split into two triangles, with the same label plus ".{kt}" appended
  # where {kt} is either 0 or 1. Each plaza "fp.{kx}" is split into {Ni}
  # triangles, with labels "fp.{kx}.{ki}" where {ki} is in {0..Ni-1}.
  # 
  # The diagonal added to a quadrangular face will have the same label
  # as the face, with "d" instead of "f". The diagonals of plazas will
  # have labels "dp.{kx}.{ki}" where {ki} ranges in {1..Ni-1}.
  
  Nv_in = len(Vlst)-1  # Total vertex count (input and output).
  Ne_in = len(Elst)-1  # Total input edge count.
  Nf_in = len(Flst)-1  # Total input faces.
  
  Nv_ot = 2*(Ni+2)      # Expected output vertex count.
  Ne_ot = 6*Ni + 6      # Expected output edge count.
  Nf_ot = 4*Ni + 4      # Expected output face (triangle) count.
  
  sys.stderr.write("Expecting %d edges\n" % Ne_ot)
  sys.stderr.write("Expecting %d faces\n" % Nf_ot)

  assert Nv_ot == Nv_in, "{Ni,Vlst} inconsistent"

  Etri = Elst.copy() + [None]*(Ne_ot-Ne_in)
  Ftri = [None]*(Nf_ot+1)
  
  Eset = set()  # Set of edges as pairs, to check for repetitions.
  for je in range(1,Ne_in+1): 
    e = Elst[je]
    Eset.add((e[0],e[1]))
  
  ne = Ne_in # Number of edges created so far.
  nf = 0     # Number of faces created so far.
 
  def Svdi(kv_org, kv_dst, elab):
    # Adds the diagonal edge from {Vlst[kv_org]} to {Vlst[kv_dst]} to {Etri},
    # reoriented in increasing index sense.
    nonlocal ne, nf
    if kv_org > kv_dst: kv_org, kv_dst = kv_dst, kv_org
    etype = 1
    e = make_edge(Vlst[kv_org], Vlst[kv_dst], ne+1, etype, elab, prec)
    assert e != None
    if verbose: prte(e, prec)
    ne += 1
    Etri[ne] = e
    check_repeated_edge(ne, Etri, Eset, prec)
    return None
    
  def Tri(kva,kvb,kvc,flab):
    # Adds to {Etri} the triangle with vertices {Vlst[kva],Vlst[kvb],Vlst[kvc]}
    # assumed to be CCW seen from outside.
    nonlocal ne, nf
    nf += 1
    t = make_face((kva,kvb,kvc), nf, flab, Vlst, prec)
    if verbose: prtf(t, Vlst, prec)
    Ftri[nf] = t

  for f in Flst[1:]:
    Nx,Ny,Nz,Fiv,kf,flab = f;
    if verbose: 
      sys.stderr.write("\n")
      sys.stderr.write("  IN: ")
      prtf(f, Vlst, prec)
      sys.stderr.write("\n")
    
    if flab[0:2] == "fu" or flab[0:2] == "fs":
      # Upper chain or spoke face:
      assert len(Fiv) == 4
      Tri(Fiv[0],Fiv[1],Fiv[3], flab + ".0")
      Tri(Fiv[1],Fiv[2],Fiv[3], flab + ".1")
      Svdi(Fiv[1],Fiv[3],"d" + flab[1:])
    elif flab[0:2] == "fp":
      # Plaza face:
      assert len(Fiv) == Ni+2;
      kx = int(flab[3])
      assert kx == 0 or kx == 1
      for ki in range(Ni):
        if verbose: sys.stderr.write("\n")
        kva = Vind['vu'][kx][ki]
        kvb = Vind['vu'][kx][ki+1]
        kvc = Vind['vc'][kx]
        if kx == 0:
          Tri(kva,kvb,kvc, flab + f".{ki}")
        else:
          Tri(kvc,kvb,kva, flab + f".{ki}")
        if ki < Ni-1:
          Svdi(kvb,kvc,"d" + flab[1:] + f".{ki+1}")
    else:
      assert False, f"invalid face label '{flab}'"

  sys.stderr.write("generated %d faces (expected %d)\n" % (nf, Nf_ot))
  if nf < Nf_ot: sys.stderr.write("!! missing some faces\n")
  assert nf <= Nf_ot

  sys.stderr.write("generated %d edges (expected %d)\n" % (ne, Ne_ot))
  if ne < Ne_ot: sys.stderr.write("!! missing some edges\n")
  assert ne <= Ne_ot

  return Etri, Ftri
