# /usr/bin/python3
# Test program for module {example_path}
# Last edited on 2021-02-18 21:16:00 by jstolfi

import example_path
import path
import contact
import job_parms
import hacks
import pyx
import rn
import sys
from math import sqrt, sin, cos, floor, ceil, inf, nan, pi

parms = job_parms.typical()

pyx.unit.set(uscale=0.5, wscale=0.5, vscale=0.5)

def plot_path_and_contacts(name, ph, CS, xbg, ybg, wdfill, axes, dots, arrows, matter):
  # Plots the path {ph} and contact list {CS} to file
  # "tests/out/example_path_TST_{name}.{ext}" where {ext} is "eps", "png", and
  # "jpg".
  #
  # If {xbg} and {ybg} are not {None}, plots a white frame 
  # around the box from ({0,0)} to {(xbg,ybg)} to establish the
  # figure's bounding box.
  #
  # Assumes that {wdfill} is the scan-line spacing for solid raster line
  # fill. If {axes} is true, plots the axes of traces too. If {arrows} is
  # true, plots arrows in traces too. If {matter} is true, shows also the
  # approximate area covered by the extruded material.
  
  c = pyx.canvas.canvas()

  if xbg != None and ybg != None:
    hacks.plot_frame(c, pyx.color.rgb.white, 0.50, None, xbg, ybg, 0.25)

  ctrace =  pyx.color.rgb(0.050, 0.850, 0.000) # Color of nominal trace area.
  ccont =   pyx.color.rgb(1.000, 0.000, 0.000) # Color of contacts.

  waxes = 0.05*wdfill  # Width of jumps and axis lines.
  wcont = 0.10*wdfill  # Width of contacts.
  path.plot_standard(c, ph, (0,0), None, ctrace, waxes, axes, dots, arrows, matter)

  for ct in CS: contact.plot(c, ct, (0,0), wcont, ccont)

  hacks.write_plot(c, "tests/out/example_path_TST_" + name)

def test_simple():
  name = "simple"
  ph = example_path.simple(parms)
  CS = []
  xbg = 6
  ybg = 3
  wdfill = parms['solid_raster_width']
  axes = True
  dots = True
  arrows = False
  matter = True
  plot_path_and_contacts(name, ph, CS, xbg, ybg, wdfill, axes, dots, arrows, matter)

def test_onion(Rc, Rf, nt):
  name = "onion%05.2f_%05.2f_%03d" % (Rc,Rf,nt)
  wdc = parms['contour_trace_width']
  wdf = parms['solid_raster_width']
  Rest = max(Rc,Rc) + wdf/2
  Rbox = ceil(Rest + wdf) + 1
  ctr = (Rbox,Rbox)
  phase = 0.5*pi
  ph, Rin, Rot = example_path.onion(ctr, Rc, wdc, Rf, wdf, phase, nt, parms)
  CS = []
  xbg = 2*Rbox
  ybg = 2*Rbox
  axes = True
  dots = True
  arrows = True
  matter = True
  plot_path_and_contacts(name, ph, CS, xbg, ybg, wdf, axes, dots, arrows, matter)

def test_scanline_fill(alt):
  name = "scanline_fill_" + ("ALTDIR" if alt else "UNIDIR")

  wdfill = 2*parms['solid_raster_width']
  wd = wdfill # Width of traces

  ph, ct = example_path.scanline_fill(alt, wd, parms)
  CS = [ ct ]
  xbg = 10
  ybg = 7
  axes = False
  dots = False
  arrows = False
  matter = False
  plot_path_and_contacts(name, ph, CS, xbg, ybg, wdfill, axes, dots, arrows, matter)

def test_gearloose():
  name = "gearloose"
  R = 10
  ph = example_path.gearloose(R, False, parms)
  CS = []
  xbg = 2*R+2 
  ybg = 2*R+2
  wdfill = parms['solid_raster_width']
  axes = False
  dots = False
  arrows = False
  matter = False
  plot_path_and_contacts(name, ph, CS, xbg, ybg, wdfill, axes, dots, arrows, matter)

def test_raster_rectangle():
  name = "raster_rectangle"
  wdfill = parms['solid_raster_width']
  
  # Create the test path {ph} and contacts {CS}:
  wd = wdfill
  n = 7
  dp = (1,1)
  xsz = 3
  step = -1.5*wd
  use_jumps = False
  ph = example_path.raster_rectangle(n, dp, xsz, step, wd, use_jumps, parms)
  CS = []
  
  # Plot it:
  xbg = 1 +xsz + 1
  ybg = 1 + (n-1)*abs(step) + 1
  axes = False
  dots = False
  arrows = True
  matter = True
  plot_path_and_contacts(name, ph, CS, xbg, ybg, wdfill, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

test_onion(5,0.3,36)
test_onion(5,0.3,35)
test_onion(5,7,8)
test_raster_rectangle()
test_scanline_fill(False)
test_scanline_fill(True)
test_gearloose()
test_simple()
