/** @file ruler.hpp
  * Definitions for measurement type and operations 
  */

#pragma once
#include <iostream>
#include <chrono>

namespace ruler{

/*
 * measure
 * -------
 */

template<class rep, class ratio>
using measure		= std::chrono::duration<rep, ratio>;

using nanometer		= measure<int64_t, std::nano>;
using micrometer	= measure<int64_t, std::micro>;
using millimeter	= measure<int64_t, std::milli>;
using centimeter	= measure<int64_t, std::centi>;
using meter		= measure<int64_t, std::ratio<1>>;
using kilometer		= measure<int64_t, std::kilo>;

/*
 * pixel density
 * -------------
 */

template<class rep, class ratio>
using pixel_density = std::chrono::duration<rep, ratio>;

using ppm	= pixel_density<double, std::ratio<1>>;
using ppcm	= pixel_density<double, std::ratio<100, 1>>;
using ppmm	= pixel_density<double, std::ratio<1000, 1>>;

/*
 * ruler defined literals
 * ----------------------
 */

namespace literals{

nanometer	operator"" _nm(unsigned long long n);
millimeter	operator"" _mm(unsigned long long n);
centimeter	operator"" _cm(unsigned long long n);
meter		operator"" _m(unsigned long long n);
kilometer	operator"" _km(unsigned long long n);
ppm		operator"" _ppm(unsigned long long n);
ppcm		operator"" _ppcm(unsigned long long n);
ppmm		operator"" _ppmm(unsigned long long n);
ppm		operator"" _ppm(long double n);
ppcm		operator"" _ppcm(long double n);
ppmm		operator"" _ppmm(long double n);

} // end of namespace literals
} // end of namespace ruler

/*
 * stream operators
 * ----------------
 */

std::ostream& operator<<(std::ostream& out, ruler::ppm const& ppm);
std::ostream& operator<<(std::ostream& out, ruler::ppcm const& ppcm);
std::ostream& operator<<(std::ostream& out, ruler::ppmm const& ppmm);

