#include <iostream>
#include <fstream>

#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>

#include <cdt2d/cdt2d.hpp>

#include "shapetor-options.hpp"

using namespace std::string_literals;
using shape_representation = shapetor_options::shape_representation;
namespace fs = std::filesystem;

/*
 * auxiliar functions
 * ------------------
 */

cv::Mat open_binary_representation(fs::path const& path)
{

	auto binary = cv::imread(
		path,
		cv::IMREAD_GRAYSCALE
	);

	if(binary.data == nullptr){
		throw std::runtime_error(
			"could not open the binary image: "s
			+ path.c_str()
		);
	}

	return binary;
}

/*
 * convertions
 * -----------
 */

void convert_polygon_to_binary(shapetor_options const& options)
{
	std::ifstream polygon_file(options.input_paths().front());
	cdt::polygon2d polygon;
	polygon_file >> polygon;

	if(options.verbose()){
		std::cout
			<< "_______\n"
			<< "Polygon\n"
			<< "-------\n"
			<< polygon
			<< '\n';
	}

	cv::Mat binary;
	cdt::polygon_to_binary(
		polygon,
		binary,
		options.resolution(),
		options.margin()
	);

	cv::imwrite(options.input_paths().at(1), binary);
}

void convert_binary_to_cdt2d(shapetor_options const& options)
{
	auto binary = open_binary_representation(options.input_paths().at(0));
	cv::Mat cdt;

	cdt::binary_to_cdt2d(
		binary,
		cdt,
		options.resolution(),
		options.k(),
		options.delta(),
		options.margin()
	);

	cv::imwrite(options.input_paths().at(1), cdt);
}

void convert(shapetor_options const& options)
{
	if(options.from_representation() == shape_representation::POLYGON
		&& options.to_representation() == shape_representation::BINARY){
		convert_polygon_to_binary(options);
	}else if(options.from_representation() == shape_representation::BINARY
		&& options.to_representation() == shape_representation::CDT2D){
		convert_binary_to_cdt2d(options);
	}else{
		std::stringstream ss;

		ss	<< "conversion from "
			<< options.from_representation()
			<< " to "
			<< options.to_representation()
			<< " was not implemented";
		throw std::runtime_error(ss.str());
	}
}

void dispatch_command(shapetor_options const& options)
{
	switch(options.command()){
	case shapetor_options::cmd::CONVERT:
		convert(options);
		break;
	case shapetor_options::cmd::UNION:
		throw std::runtime_error("union command was not implemented");
		break;
	case shapetor_options::cmd::MINUS:
		throw std::runtime_error("minus command was not implemented");
		break;
	case shapetor_options::cmd::INTERSECTION:
		throw std::runtime_error(
			"intersection command was not implemented"
		);
		break;
	case shapetor_options::cmd::NONE:
		throw std::runtime_error("none command was not implemented");
		break;
	}
}

int main(int const argc, char const* const argv[])
try{

	shapetor_options options(argc, argv);

	if(options.verbose()){
		std::cout
			<< "________________\n"
			<< "Shapetor options\n"
			<< "----------------\n"
			<< options;
	}

	switch(options.state()){
	case shapetor_options::parsing_state::HELP:
		std::cout << options.usage() << '\n';
		std::cout << options.all_options_descriptions();
		break;
	case shapetor_options::parsing_state::VERSION:
		std::cout << "shapetor version 0.0.0\n";
		break;
	case shapetor_options::parsing_state::NONE:
		throw std::runtime_error(
			"parsing state cannot be none after"
			" construction"
		);
		break;
	case shapetor_options::parsing_state::OPTIONS_READY:
		dispatch_command(options);
		break;
	}

	return EXIT_SUCCESS;

}catch(boost::program_options::error& e){
	std::cerr << "parsing error: " << e.what() << '\n';
	return EXIT_FAILURE;
}catch(shapetor_options::usage_error& e){
	std::cerr << "usage error: " << e.what() << '\n';
	std::cerr << shapetor_options::usage(argv[0]) << '\n';
	return EXIT_FAILURE;
}catch(std::runtime_error& e){
	std::cerr << "runtime error: " << e.what() << '\n';
	return EXIT_FAILURE;
}
