# Procedures to group ratser filling elements in various ways.
# Last edited on 2021-05-27 15:25:59 by jstolfi

import regroup_rasters_IMP

# The procedures in this module assume that the parameter {OPHS} is a
# list of oriented paths that are elements of a solid fill, each
# possibly assigned to a /group/ whose index is accessed through
# {path_hp.get_group}. The procedures split or redefine those groups
# into new groups according to various criteria, and attach the new
# group indices to the paths of {OPHS} with {path_hp.set_group}.
# 
# Each path in {OPHS} should consist of a single raster trace. The paths
# must be pairwise disjoint and all parallel to some unit direction
# vector {xdir}. They must be sorted in /scanline order/, which is
# defined by increasing projection on an axis {ydir} perpendicular to
# {xdir}, ignoring roundoff errors; with ties broken by increasing
# projection along {xdir}. The orientations of the paths and traces are
# ignored by these procedures.
#
# These procedures use the contact information of each path as 
# accessed by {path_hp.get_contacts} and {path_hp.set_contacts}.
# These lists must have all contacts which have both sides 
# on paths of {OPHS}, and only those.  Contacts between {OPHS} and 
# other paths (such as contours) should not be in those lists.
#
# Every procedure that is supposed to change the group assignments will
# renumber the new groups with consecutive integers starting at 0, even
# if the partition of {OPHS} into groups remains the same. They return
# the number of new groups created.

def merge_all_groups(OPHS):
  # Ignores the current group assignments of the paths in {OPHS}, and reassigns
  # all to group 0.
  return regroup_rasters_IMP.merge_all_groups(OPHS)

def define_groups_by_contour(OPHS, CRS):
  # Ignores the current group assignment and instead separats the filler
  # elements from {OPHS} into new groups, one for each contour (closed
  # oriented path) in the list {CRS}. Each filler will be assigned to
  # the group with index {icr} such that {ocr=CRS[icr]} is the
  # innermost contour that contains it.
  #
  # It is assumed that all fillers are in the interior of the slice.
  # Therefore, there will not be any filler assigned to a group index
  # {icr} if {icr >= len(CRS)}, if {CRS[icr]} is an inner (hole)
  # contour, or if contour {CRS[icr]} is an outer contour but there is
  # no filler inside it except inside nested contours.
  return regroup_rasters_IMP.define_groups_by_contour(OPHS, CRS)
  
def split_groups_at_forks(OPHS): 
  # Splits the current groups of fillers from {OPHS} into maximal
  # sub-groups connected by single contacts.
  #
  # Specifically, each of the original groups is split into maximal
  # sub-groups such that, in every sub-group whose fillers are
  # {GPH[0..m-1]}, and for every {i} in {1..m-1}, there is a contact
  # between {GPH[i-1]} and {GPH[i]}, which is the only contact of on the
  # upper side of {GPH[i-1]} and the only contact in the lower side of
  # {GPH[i]}
  #
  # The new groups will be identified by consecutive non-negative
  # integers starting from 0, even if no splits occurred. Returns the
  # number of those new groups.
  return regroup_rasters_IMP.split_groups_at_forks(OPHS)
  
def break_large_groups(OPHS, ydir, max_lines):
  # Splits each current group, if needed, into sug-groups with at most 
  # {max_lines} whole scan-lines each.  
  #
  # In particular, if {max_lines} is 1, each new group will be contained
  # in a single scan-line. If that operation is applied after
  # {split_groups_at_forks} each group will be a single filler element.
  #
  # The new groups are renumbered with consecutive integers starting at
  # 0, even if no splits occurred. Returns the number of new groups.
  regroup_rasters_IMP.break_large_groups(OPHS, ydir, max_lines)
