/** @file iterator.hpp
  * Implementation of adaptors to help common 
  * objects manipulation
  */

#pragma once
#include <type_traits>

#include <opencv2/core/core.hpp>

namespace cdt{

template <bool C, class A, class B>
struct type_if_else{};

template <class A, class B>
struct type_if_else<true, A, B>{
	typedef A type;
};

template <class A, class B>
struct type_if_else<false, A, B>{
	typedef B type;
};

/*
 * adaptors for CGAL
 * -----------------
 */

template<class T>
class vertices_iterator_adaptor{
protected:
	T& m_reference;
public:
	explicit vertices_iterator_adaptor(T& a_reference) noexcept
		: m_reference(a_reference)
	{}

	virtual ~vertices_iterator_adaptor() = default;

	auto begin()
	{
		return m_reference.vertices_begin();
	}

	auto begin() const
	{
		return m_reference.vertices_begin();
	}

	auto end()
	{
		return m_reference.vertices_end();
	}

	auto end() const
	{
		return m_reference.vertices_end();
	}

};

template<class T>
class holes_iterator_adaptor{
protected:
	T& m_reference;
public:
	explicit holes_iterator_adaptor(T& a_reference) noexcept
		: m_reference(a_reference)
	{}

	virtual ~holes_iterator_adaptor() = default;

	auto begin()
	{
		return m_reference.holes_begin();
	}

	auto begin() const
	{
		return m_reference.holes_begin();
	}

	auto end()
	{
		return m_reference.holes_end();
	}

	auto end() const
	{
		return m_reference.holes_end();
	}

};

template<class T>
class edges_iterator_adaptor{
protected:
	T& m_reference;
public:
	explicit edges_iterator_adaptor(T& a_reference) noexcept
		: m_reference(a_reference)
	{}

	virtual ~edges_iterator_adaptor() = default;

	auto begin()
	{
		return m_reference.edges_begin();
	}

	auto begin() const
	{
		return m_reference.edges_begin();
	}

	auto end()
	{
		return m_reference.edges_end();
	}

	auto end() const
	{
		return m_reference.edges_end();
	}

};

/**
  * build an adaptor to transverse the vertices of a
  * CGAL object.
  */
template<class T>
auto vertices(T& ref)
{
	return vertices_iterator_adaptor<T>{ref};
}

/**
  * build an adaptor to transverse the holes of a
  * CGAL object.
  */
template<class T>
auto holes(T& ref)
{
	return holes_iterator_adaptor<T>{ref};
}

/**
  * build an adaptor to transverse the edges of a
  * CGAL object.
  */
template<class T>
auto edges(T& ref)
{
	return edges_iterator_adaptor<T>{ref};
}

/*
 * adaptors for opencv
 * -------------------
 */

template<class pixel_type, class M>
class pixels_iterator_adaptor{
public:
	pixels_iterator_adaptor() = delete;
	virtual ~pixels_iterator_adaptor() = default;

	explicit pixels_iterator_adaptor(M& a_reference) noexcept
		: m_reference(a_reference)
	{}

	auto begin()
	{
		return m_reference.template begin<pixel_type>();
	}

	auto begin() const
	{
		return m_reference.template begin<pixel_type>();
	}

	auto end()
	{
		return m_reference.template end<pixel_type>();
	}

	auto end() const
	{
		return m_reference.template end<pixel_type>();
	}

protected:
	M& m_reference;
};

/**
  * build an adaptor to transverse the pixels of an opencv image
  */
template<class T>
auto pixels(cv::Mat& img)
{
	return pixels_iterator_adaptor<T, cv::Mat>(img);
}

/**
  * build an adaptor to transverse the pixels of an opencv image
  */
template<class T>
auto pixels(cv::Mat const& img)
{
	return pixels_iterator_adaptor<T, cv::Mat const>(img);
}

template<class T, class I, class M>
class pixel_point_iterator : public I {
protected:
	// type of the reference that will be returned in operator*
	using return_reference_type = typename type_if_else<
		std::is_same<I, cv::MatIterator_<T>>::value,
		T,
		T const
	>::type;

public:
	pixel_point_iterator() = delete;
	virtual ~pixel_point_iterator() = default;

	pixel_point_iterator(
		I it,
		cv::Point2i& a_coordinate,
		M& a_img)
		: I(it),
		  m_coordinate(a_coordinate),
		  m_rows(a_img.rows),
		  m_cols(a_img.cols)
	{}

	pixel_point_iterator& operator++()
	{
		++m_coordinate.x;
		if(m_coordinate.x == m_cols){
			m_coordinate.x = 0;
			++m_coordinate.y;
		}

		I::operator++();

		return *this;
	}

	pixel_point_iterator operator++(int)
	{
		++m_coordinate.x;
		if(m_coordinate.x == m_cols){
			m_coordinate.x = 0;
			++m_coordinate.y;
		}

		I::operator++(0);

		return *this;
	}

	std::tuple<return_reference_type&, cv::Point2i&> operator*()
	{
		return { I::operator*(), m_coordinate };
	}

protected:
	cv::Point2i& m_coordinate;
	int32_t m_rows;
	int32_t m_cols;

};

template<class T, class I, class M>
class pixel_point_iterator_adaptor{
public:
	pixel_point_iterator_adaptor() = delete;
	virtual ~pixel_point_iterator_adaptor() = default;

	pixel_point_iterator_adaptor(M& a_img)
		: m_reference(a_img),
		  m_point{0, 0}
	{}

	auto begin()
	{
		return pixel_point_iterator<T, I, M>(
			m_reference.template begin<T>(),
			m_point,
			m_reference
		);
	}

	auto begin() const
	{
		return pixel_point_iterator<T, I, M>(
			m_reference.template begin<T>(),
			m_point,
			m_reference
		);
	}

	auto end()
	{
		return pixel_point_iterator<T, I, M>(
			m_reference.template end<T>(),
			m_point,
			m_reference
		);
	}

	auto end() const
	{
		return pixel_point_iterator<T, I, M>(
			m_reference.template end<T>(),
			m_point,
			m_reference
		);
	}

protected:
	M& m_reference;
	cv::Point2i m_point;
};

/**
  * build an adaptor to transverse the pixels and the coordinate 
  * of an opencv image
  */
template<class T>
auto pixels_at(cv::Mat& img)
{
	return pixel_point_iterator_adaptor<
		T,
		cv::MatIterator_<T>,
		cv::Mat
	>(img);
}

/**
  * build an adaptor to transverse the pixels and the coordinate 
  * of an opencv image
  */
template<class T>
auto pixels_at(cv::Mat const& img)
{
	return pixel_point_iterator_adaptor<
		T,
		cv::MatConstIterator_<T>,
		cv::Mat const
	>(img);
}

}
