// The DCC abacus

// Assumes defined the following things:

//   These are absolute:
//     abacus_width              = width (X) of abacus, including frame
//     abacus_height             = height (Z) of abacus, including frame
//     abacus_frame_width        = width of abacus frame, in both directions
//     abacus_frame_thickness    = thickness (Y) of abacus frame
//     abacus_stick_R            = radius of slider-bearing sticks

//     abacus_frame_texture        = texture for frame (long axis is X axis)
//     abacus_unit_slider_{texture,interior} = texture and interior for the "units" sliders
//     abacus_five_slider_{texture,interior} = texture and interior for the "fives" sliders
//     abacus_stick_texture        = texture for the sticks

// Defines 
//  abacus  = 
//     the whole abacus, centered on the origin and perpendicular
//     to the Y axis, with sticks parallel to X axis.

#include "dcc-abacus-slider.inc"

#declare abacus_space_width = abacus_width - 2 * abacus_frame_width;
#declare abacus_space_height = abacus_height - 2 * abacus_frame_width;

#declare abacus_space_HX = abacus_space_width / 2;
#declare abacus_space_HZ = abacus_space_height / 2;

#declare abacus_slider_scaled_width = (abacus_space_width - abacus_frame_width) / 9;

#declare abacus_slider_scale = abacus_slider_scaled_width / (1.05 * abacus_slider_width);

// Properly rotated, scaled, and Y-displaced slider:
// (Assumes the abacus is lying flat on a table.)
#declare abacus_scaled_slider = 
  object{abacus_slider
    rotate <0,0,90>
    scale abacus_slider_scale
    translate <0, abacus_stick_R - abacus_slider_scale * abacus_slider_hole_R, 0>
  }

#declare abacus_frame_HY = abacus_frame_thickness / 2;

#declare abacus_frame_long_side =
  intersection{
    box{
      < -(abacus_space_HX + abacus_frame_width), 
        -abacus_frame_HY, 
        -abacus_frame_width
      >, 
      < +(abacus_space_HX + abacus_frame_width), 
        +abacus_frame_HY, 
        0
      >
    }
    plane{<-1,00,+1>, 0 translate <-abacus_space_HX,0,0>}
    plane{<+1,00,+1>, 0 translate <+abacus_space_HX,0,0>}
  }

#declare abacus_frame_short_side =
  intersection{
    box{
      < -(abacus_space_HZ + abacus_frame_width),
        -abacus_frame_HY,
        -abacus_frame_width
      >, 
      < +(abacus_space_HZ + abacus_frame_width),
        +abacus_frame_HY, 
        0
      >
    }
    plane{<-1,00,+1>, 0 translate <-abacus_space_HZ,0,0>}
    plane{<+1,00,+1>, 0 translate <+abacus_space_HZ,0,0>}
   }
  

#declare abacus_frame_bottom_side =
  object{abacus_frame_long_side
    translate <0,0,-abacus_space_HZ>
    texture{abacus_frame_texture}
  }

#declare abacus_frame_top_side =
  object{abacus_frame_long_side 
    scale <1,1,-1>
    translate <0,0,+abacus_space_HZ>
    texture{abacus_frame_texture}
  }

#declare abacus_frame_left_side =
  object{abacus_frame_short_side
    translate <0,0,-abacus_space_HX>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame_right_side =
  object{abacus_frame_short_side
    scale <1,1,-1>
    translate <0,0,+abacus_space_HX>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame_separator_X = 
  - abacus_space_HX + 6 * abacus_slider_scaled_width + abacus_frame_width/2;

#declare abacus_frame_separator =
  box{
    < -abacus_space_HZ, 
      -abacus_frame_HY, 
      -abacus_frame_width/2
    >, 
    < +abacus_space_HZ,
      +abacus_frame_HY, 
      +abacus_frame_width/2
    >
    translate <0,0,abacus_frame_separator_X>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame =
  merge{
    object{abacus_frame_bottom_side}
    object{abacus_frame_top_side}
    object{abacus_frame_left_side}
    object{abacus_frame_right_side}
    object{abacus_frame_separator}
  }
  
#declare abacus_stick =
  cylinder{
    <-(abacus_space_HX + abacus_frame_width/2), 0, 0>,
    <+(abacus_space_HX + abacus_frame_width/2), 0, 0>
    abacus_stick_R
    texture{abacus_stick_texture}
  }
  
#declare abacus_stick_spacing = abacus_space_height / 4.5;
  
#declare abacus =
  union{
    object{abacus_frame}
    // Top stick:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 1.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 1.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      translate <0, 0.5*abacus_frame_HY, +1.5*abacus_stick_spacing>
    }
    // Second stick from top:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      translate <0, 0.5*abacus_frame_HY, +0.5*abacus_stick_spacing>
    }
    // Third stick from top:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      translate <0, 0.5*abacus_frame_HY, -0.5*abacus_stick_spacing>
    }
    // Bottom stick:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_unit_slider_texture } interior{ abacus_unit_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_five_slider_texture } interior{ abacus_five_slider_interior }
      }
      translate <0, 0.5*abacus_frame_HY, -1.5*abacus_stick_spacing>
    }
  }
