#include <iostream>
#include <fstream>
#include <filesystem>

#include <boost/program_options.hpp>

#include <cdt2d/polygon2d.hpp>
#include <cdt2d/iterator.hpp>

namespace opt = boost::program_options;
namespace fs  = std::filesystem;

struct options{
	std::optional<fs::path> input_path;
	std::optional<fs::path> output_path;
	bool origin_centralize = false;
	bool flip_horizontally = false;
	bool flip_vertically = false;
}options;

std::string const& usage(std::string const& argv0 = "program-name")
{
	static std::string usage_msg(
		argv0 + " [plygn-options] <polygon-path>"
	);

	return usage_msg;
}

std::istream& operator>>(std::istream& in, std::optional<fs::path>& op)
{
	fs::path tmp;
	in >> tmp;
	op = tmp;
	return in;
}

void parse_arguments(int const argc, char const* const argv[])
try{
	opt::options_description plygn_options("plygn options");
	opt::options_description visible_options;

	visible_options.add_options()
		("help,h", "Show this message.")
		("origin-centralize,c", "Centralize the polygon to the origin.")
		(
			"flip-horizontally,-H",
			opt::value<bool>(&options.flip_horizontally),
			"Flips the polygon horizontally"
		)
		(
			"flip-vertically,-V",
			opt::value<bool>(&options.flip_vertically),
			"Flips the polygon vertically"
		)
		(
			"output,o",
			opt::value<decltype(options.output_path)>(
				&options.output_path
			),
			"Sets the output path. (default: stdout)"
		);

	plygn_options.add(visible_options);
	plygn_options.add_options()
	(
		"polygon-path",
		opt::value<decltype(options.input_path)>(&options.input_path),
		"path to the polygon"
	);

	opt::positional_options_description positional;
	positional.add("polygon-path", 1);

	auto parsed = opt::command_line_parser(argc, argv)
		.options(plygn_options)
		.positional(positional)
		.run();

	opt::variables_map vm;
	opt::store(parsed, vm);

	if(vm.count("help")){
		std::cout << visible_options << '\n';
		std::exit(EXIT_SUCCESS);
	}

	opt::notify(vm);

	if(vm.count("origin-centralize"))
		options.origin_centralize = true;

}catch(boost::program_options::error& e){
	std::cerr << e.what() << '\n';
	std::exit(EXIT_FAILURE);
}

std::istream& get_input_reference(std::ifstream& file)
{
	if(options.input_path.has_value()
		&& fs::exists(options.input_path.value())){

		file.open(options.input_path.value());
		return file;
	}

	return std::cin;
}

std::ostream& get_output_reference(std::ofstream& file)
{
	if(options.output_path.has_value()){
		file.open(options.output_path.value());
		return file;
	}

	return std::cout;
}

auto compute_middle_of_bounding_box(cdt::polygon2d const& polygon)
{
	auto b = polygon.bbox();

	return cdt::point2d{
		(b.xmin() + b.xmax())/2.,
		(b.ymin() + b.ymax())/2.
	};
}

/**
  * compute the bounding box of the `polygon` and centralize
  * the center of the bounding box in the origin
  */
void origin_centralize(cdt::polygon2d& polygon)
{
	auto m = compute_middle_of_bounding_box(polygon);

	for(auto& v : cdt::vertices(polygon.outer_boundary())){
		v = cdt::point2d{
			v.x() - m.x(),
			v.y() - m.y()
		};
	}

	for(auto& hole : cdt::holes(polygon)){
		for(auto& v : cdt::vertices(hole)){
			v = cdt::point2d{
				v.x() - m.x(),
				v.y() - m.y()
			};
		}
	}

}

void flip(cdt::polygon2d const& polygon)
{
	auto m = compute_middle_of_bounding_box(polygon);

	auto flip_v = [&](cdt::point2d const& v)
	{
		auto new_x = v.x() - m.x();
		auto new_y = v.y() - m.y();

		if(options.flip_horizontally)
			new_x *= -1.;

		if(options.flip_vertically)
			new_y *= -1.;

		return cdt::point2d{
			new_x + m.x(),
			new_y + m.y()
		};
	};

	for(auto& v : cdt::vertices(polygon.outer_boundary()))
		v = flip_v(v);

	for(auto& hole : cdt::holes(polygon))
		for(auto& v : cdt::vertices(hole))
			v = flip_v(v);
}

void dispatch_operations(cdt::polygon2d& polygon)
{
	if(options.origin_centralize)
		origin_centralize(polygon);
	if(options.flip_vertically || options.flip_horizontally)
		flip(polygon);

}

int main(int const argc, char const* const argv[])
{

	parse_arguments(argc, argv);

	std::ifstream input_file;
	auto& input_ref = get_input_reference(input_file);

	std::ofstream output_file;
	auto& output_ref = get_output_reference(output_file);

	cdt::polygon2d polygon;
	input_ref >> polygon;

	dispatch_operations(polygon);

	output_ref << polygon;

	return EXIT_SUCCESS;
}
