/** @file shapetor-options.hpp
  * Shapetor options definition and command line parsing
  */

#pragma once
#include <cstdint>
#include <filesystem>
#include <stdexcept>

#include <cdt2d/cdt2d.hpp>

#include <boost/program_options.hpp>

/**
  * Representation of the shapetor-app options. Encapsulates the parsing from
  * command line and the storage in memory of the options.
  */
class shapetor_options{
public:

	/**
	  * This class is used as an exception when 
	  * an user input error was detected.
	  */
	class usage_error : public std::runtime_error{
	public:
		usage_error() = delete;
		explicit usage_error(std::string const& msg);
		virtual ~usage_error() = default;
	};

	using paths = std::vector<std::filesystem::path>;
	using options_description = boost::program_options::options_description;

	using delta_type	= cdt::delta_type;
	using k_type		= cdt::k_type;
	using resolution_rep	= double;
	using resolution_type	= ruler::ppcm;
	using margin_type	= cdt::margin_type;

	/**
	  * enum class of the supported commands
	  */
	enum class cmd : uint8_t {
		NONE,
		CONVERT,
		UNION,
		MINUS,
		INTERSECTION
	};


	/**
	  * supported representation types for convertion
	  */
	enum class shape_representation : uint8_t {
		NONE,
		BINARY,
		POLYGON,
		CDT2D
	};

	/**
	  * represents the state of the options parsing
	  */
	enum class parsing_state : uint8_t {
		NONE,
		HELP,
		VERSION,
		OPTIONS_READY
	};

	shapetor_options() = delete;
	virtual ~shapetor_options() = default;

	/**
	  * @return the usage string of shapetor
	  */
	std::string usage() const;
	static std::string usage(std::string const& argv0);

	/**
	  * Shapetor options constructor. Parses and store the options
	  * given from command line.
	  *
	  * @param argc	number of arguments
	  * @param argv	array of size `argc' of arguments
	  */
	explicit shapetor_options(int const argc, char const* const* argv);

	/**
	  * Getter of the parsed command.
	  *
	  * @see cmd
	  */
	cmd command() const;

	/**
	  * @return representation of the object to be converted in the convert
	  * command
	  * @see shape_representation
	  */
	shape_representation from_representation() const;

	/**
	  * @return representation of the object after the conversion
	  * @see shape_representation
	  */
	shape_representation to_representation() const;

	/**
	  * @return the list of the input paths passed by parameter
	  */
	paths const& input_paths() const;

	/**
	  * @return delta for cdt
	  */
	delta_type delta() const;

	/** k is the proportion of the reducing of the binary image
	  * to cdt
	  * @return k for cdt. 
	  */
	k_type k() const;

	/**
	  * @return the resolution to convert the boundary objects
	  */
	resolution_type resolution() const;

	/**
	  * @return the bouding box to consider around the object
	  */
	margin_type margin() const;

	/**
	  * @return the current state of the program
	  */
	parsing_state state() const;

	/**
	  * @return true if the verbose mode is enable and false otherwise
	  */
	bool verbose() const;

	/**
	  * @return true if the user want to overwrite the output file
	  */
	bool force() const;

	/**
	  * @return all the options descriptions of shapetor program
	  */
	options_description all_options_descriptions() const;

	/**
	  * @return the arguments used to build this instance
	  */
	std::vector<std::string> const& args() const;

	friend std::ostream& operator<<(std::ostream&, parsing_state);
	friend std::ostream& operator<<(std::ostream&, cmd);
	friend std::ostream& operator<<(std::ostream&, shape_representation);
	friend std::ostream& operator<<(std::ostream&, shapetor_options const&);

protected:

	/*
	 * Helper protected functions
	 * --------------------------
	 */

	/**
	  * helper function to parse the suboptions of convert cmd
	  */
	void parse_convert_cmd(std::vector<std::string> const&);

	/**
	  * assert convert command semantics and try to deduce
	  * options (e.g. deduce the shape representation by file
	  * extension)
	  */
	void assert_convert_semantics(
		boost::program_options::variables_map const&
	);

	/*
	 * Attributes
	 * ----------
	 */

	std::vector<std::string> m_args;
	parsing_state m_state = parsing_state::NONE;

	bool m_verbose = false;
	bool m_force = false;

	cmd m_command = cmd::NONE;

	shape_representation m_from = shape_representation::NONE;
	shape_representation m_to = shape_representation::NONE;

	delta_type	m_delta{0};
	k_type		m_k{0};
	resolution_type	m_resolution{0};
	margin_type	m_margin{0};

	paths		m_input_paths;

};
