#! /usr/bin/python3
# Implementation of module {paper_figures_B}
# Last edited on 2021-10-18 04:41:48 by stolfi

import paper_figures_B
import raster_example_paper_B
import move
import move_example
import move_parms
import contact
import path
import path_example
import rootray_shape
import rootray
import raster
import block
import hacks
import pyx
import rn
import sys
from math import sqrt, hypot, sin, cos, atan2, floor, ceil, inf, nan, pi

def plot_figure(fname, fig,subfig):
  
  title = "### figure %s sub-figure %d ###" % (fig,subfig)
  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write(title + "#"*(80-len(title)) + "\n")
  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write("\n")

  mp_cont, mp_fill, mp_jump = make_move_parms();
  sys.stderr.write("printer parameters:\n")
  move_parms.show(sys.stderr, "contours = { ", mp_cont, " }\n")
  move_parms.show(sys.stderr, "filling =  { ", mp_fill, " }\n")
  move_parms.show(sys.stderr, "jumps =    { ", mp_jump, " }\n")
  
  mp_link = mp_fill # In plots they are shown thinner, but once included in paths they are like rasters.
  
  # Create the input data {} and define the :
  OCRS = []; OPHS = []; OLKS = []; CTS = []; VGS = []; EGS = [] # Just in case...
  if fig == "paths":
    # Get the simple move examples:
    OPHS = raster_example_paper_B.make_simple_path(mp_fill, mp_jump)
  elif fig == "moves":
    # Get the simple path example:
    OPHS = raster_example_paper_B.make_simple_moves(mp_fill, mp_jump)
  else:
    # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
    OCRS,OPHS,OLKS,CTS,VGS,EGS = raster_example_paper_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
  
  style = make_style_dict(mp_cont, mp_fill)
  color = make_color_dict()

  if fig == "paths":
    c = plot_figure_paths(subfig, OPHS, style,color)
  elif fig == "moves":
    c = plot_figure_moves(subfig, OPHS, style,color)
  elif fig == "input":
    c = plot_figure_input(subfig, OCRS, OPHS, OLKS, CTS, VGS, EGS, style,color)
  elif fig == "zigzag":
     c = plot_figure_scanline(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color)
  elif fig == "cold":
     c = plot_figure_cold(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color)
  elif fig == "rivers":
     c = plot_figure_rivers(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color)
  elif fig == "canon":
     c = plot_figure_canon(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color)
  elif fig == "blocks":
     c = plot_figure_blocks(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color)
  else:
     sys.stderr.write("!! figure '%s ' not implemented.\n" % fig)

  if c != None:
    hacks.write_plot(c, fname)
  else:
    sys.stderr.write("!! figure '%s subfigure %d' not plotted.\n" % (fig,subfig))

  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write("\n")

  return
  # ----------------------------------------------------------------------

def plot_figure_paths(subfig, OPHS, style,color):
  # Plots the figure that shows a smple path {OPHS[0]} with labels "pini", etc.
  # and its reverse.
  
  assert len(OPHS) == 1
  oph = OPHS[0]
  nmv = path.nelems(oph) 

  B = path.bbox([oph,]) # Bounding box of all move endpoints.

  Xstep = B[1][0] - B[0][0] + 8.0  # Displacement betwwen the two versions of the path.
  
  # Compute the figure's bounding box {Bfig}:
  Bfig = (B[0], rn.add(B[1], (Xstep,0))) # Bounding box for both versions of the path.

  # Add space at bottom and top for labels:
  ymrg0 = 1.8 # Extra margin for labels at bottom
  ymrg1 = 2.4 # Extra margin for labels at top.
  Bfig = rn.box_expand(Bfig, (0, ymrg0), (0, ymrg1))

  # Widen {Bfig} symmetrically to standard "math figure" widtdh:
  Bfig = widen_box_for_math_figure(Bfig, style)
 
  c = make_figure_canvas(Bfig, style,color)

  for rev in False, True:
    dp = (int(rev)*Xstep, 0)
    
    ophr = path.rev(oph) if rev else oph

    # Plot the path:
    rwdf = style['rwd_path']
    wd_axes = style['wd_axes']
    clr = color['fill']
    axes = True
    dots = True
    arrows = True
    matter = False
    path.plot_standard(c, [ophr,], dp, None, [clr,], rwdf,wd_axes, axes, dots, arrows, matter)
    
    # Plot the endpoints of the paths in black and larger:
    plot_path_endpoints(c, [ophr,], dp, style,color)
    
    # Labeled points ({xxxP}) and label displacements ({xxxD}) on the unreversed path:
    piniP = path.pini(oph); piniD = (-1.2, +0.8)
    pfinP = path.pfin(oph); pfinD = (-1.0, +0.8)
    pmovP = [ rn.mix(0.5,move.pini(mvk),0.5,move.pfin(mvk)) for k in range(nmv) for mvk in (path.elem(oph,k),) ]
    pmovD = [ 
      (-3.0, -0.9), # P[0]
      (-1.0, -1.8), # P[1]
      (-1.6, -1.8), # P[2]
      (+0.8, -0.9), # P[3]
    ]
    pmidP = rn.mix(0.5,piniP, 0.5,pfinP); pmidD = (-0.3, +1.0)


    txphr = r"{\mkern-0.7\thinmuskip\overleftarrow{P}\mkern-0.7\thinmuskip}" if rev else r"P"
    
    txpini = r"$\mathop{\mathrm{pini(%s)}}$" % txphr
    txpfin = r"$\mathop{\mathrm{pfin(%s)}}$" % txphr
    txpmid = r"$%s$" % txphr
    txmovS = [ (r"$%s[%d]$" % (txphr,k)) for k in range(nmv) ]

    plot_math_label(c, dp, pmidP, pmidD, txpmid, style)
    if not rev:
      plot_math_label(c, dp, piniP, piniD, txpini, style)
      plot_math_label(c, dp, pfinP, pfinD, txpfin, style)
      for k in range(nmv):
        plot_math_label(c, dp, pmovP[k], pmovD[k], txmovS[k], style)
    else:
      plot_math_label(c, dp, piniP, piniD, txpfin, style)
      plot_math_label(c, dp, pfinP, pfinD, txpini, style)
      for k in range(nmv):
        plot_math_label(c, dp, pmovP[k], pmovD[k], txmovS[nmv-1-k], style)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_moves(subfig, OPHS, style,color):
  # If {subfig} is zero, plots a figure that shows a simple move {OPHS[0]} and a simple jump {OPHS[1]}, with labels "pini", etc.
  # If {subfig} is 1, plots the reversed moves instead.
  
  assert len(OPHS) == 2

  B = path.bbox(OPHS) # Bounding box of endpoints of both moves (undisplaced).
  Xstep = B[1][0] - B[0][0] + 11.0  # Displacement betwwen the two versions of the path.
  
  # Compute the figure's bounding box {Bfig}:
  Bfig = (B[0], rn.add(B[1], (Xstep,0))) # Bounding box for both versions of the path.
  
  # Add space for labels at top and bottom:
  ymrg0 = 1.0 # Extra margin for labels at bottom
  ymrg1 = 1.5 # Extra margin for labels at top.
  Bfig = rn.box_expand(Bfig, (0,ymrg0), (0,ymrg1))
  
  # Widen to standars "math figure" width:
  Bfig = widen_box_for_math_figure(Bfig, style)
  
  c = make_figure_canvas(Bfig, style,color)
    
  rev = (subfig == 1)

  for iph in range(2):
    dp = (iph*Xstep, 0)
    
    oph = OPHS[iph]
    assert path.nelems(oph) == 1
    omv = path.elem(oph,0)
    omvr = move.rev(omv) if rev else omv

    # Plot the move:
    rwdf = style['rwd_path']
    wd_axes = style['wd_axes']
    clr = color['fill']
    axes = True
    dots = True  
    arrows = True
    matter = True
    move.plot_standard(c, [omvr,], dp, None, [clr,], rwdf,wd_axes, axes, dots, arrows, matter)
    
    # Labeled points ({xxxP}) and label displacements ({xxxD}) on the unreversed moves:
    piniP = move.pini(omv); piniD = (-4.0 + 0.4*iph - 0.4*subfig, -0.3)
    pfinP = move.pfin(omv); pfinD = (+0.8 - 0.2*iph, +0.1)
    pmidP = rn.mix(0.5,piniP, 0.5,pfinP); pmidD = (-0.5, +1.0 - 0.3*iph)

    txmvr = r"{\mkern-0.7\thinmuskip\overleftarrow{r}\mkern-0.7\thinmuskip}" if rev else r"r"
    
    txpini = r"$\mathop{\mathrm{pini(%s)}}$" % txmvr
    txpfin = r"$\mathop{\mathrm{pfin(%s)}}$" % txmvr
    txpmid = r"$%s$" % txmvr

    plot_math_label(c, dp, pmidP, pmidD, txpmid, style)
    if not rev:
      plot_math_label(c, dp, piniP, piniD, txpini, style)
      plot_math_label(c, dp, pfinP, pfinD, txpfin, style)
    else:
      plot_math_label(c, dp, piniP, piniD, txpfin, style)
      plot_math_label(c, dp, pfinP, pfinD, txpini, style)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_input(subfig, OCRS, OPHS, OLKS, CTS, VGS, EGS, style,color):
  # Plots the figures with individual rasters, as slected by {subfig}: 
  #
  #   subfig = 0 rasters and contacts.
  #   subfig = 1 links.
  #   subfig = 2 contact graph.
  #
  # Returns the canvas {c} with the figure.
  
  assert len(OPHS) > 0
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  c = make_figure_canvas(Bfig, style,color)

  xdir = (1,0)
  ydir = (0,1)
  ystep,yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir,ydir, ystep,yphase)

  dp = (0,0)
  
  if subfig == 0:

    # RASTERS AND CONTACTS

    plot_trace_matter(c, OCRS, OPHS, OLKS, dp, style,color)

    plot_contours(c, OCRS, dp, style, color['cont'])

    # Plot the filling path(s) with strong color:
    clr = color['fill']
    rwdf = style['rwd_fill']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = False
    matter = False
    path.plot_standard(c, OPHS, dp, None, [clr,], rwdf, wd_axes, axes, dots, arrows, matter)

    # Plot contacts.
    interbc = False
    plot_contacts(c, CTS, dp, interbc, color, wd_axes)

  elif subfig == 1 and len(OLKS) > 0:

    # LINKS

    plot_contours(c, OCRS, dp, style, color['ghost'])

    # Plot the filling path(s) with weak color:
    clr = color['ghost']
    rwdf = style['rwd_fill']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = False
    matter = False
    path.plot_standard(c, OPHS, dp, None, [color['ghost'],], rwdf, wd_axes, axes, dots, arrows, matter)

    # Plot individual links with various colors: 
    rwdl = style['rwd_link']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = False
    matter = False
    for isc in range(len(SCS)):
      SCSi = SCS[isc] # Rasters on scaline {isc}
      for irs in SCSi:
        ors = OPHS[irs] # A raster element on that scanline.
        ysc = path.pini(ors)[1] # Y coordinate of scanline.
        for olk in path.get_links(ors) + path.get_links(path.rev(ors)):
          # Orient the link to go up:
          if path.pini(olk)[1] > path.pfin(olk)[1]: olk = path.rev(olk)
          # Plot only links that go up from the current scanline:
          if path.pini(olk)[1] > ysc - 0.001*wd_axes:
            clk = (color['link0'], color['link1'])[isc%2]
            path.plot_standard(c, [olk,], dp, None, [clk,], rwdl, wd_axes, axes, dots, arrows, matter)

    # Plot dots at start and end of links: 
    plot_path_endpoints(c, OLKS, dp, style,color)

  elif subfig == 2:

    # CONTACT GRAPH

    plot_contours(c, OCRS, dp, style, color['ghost'])

    # Plot the filling path(s) with weak color:
    rwdf = style['rwd_fill']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = False
    matter = False
    path.plot_standard(c, OPHS, dp, None, [color['ghost'],], rwdf, wd_axes, axes, dots, arrows, matter)

    plot_contact_graph(c, VGS, EGS, style,color)
    
  else:
    assert False, ("invalid subfigure number %d.\n" % subfig)

  return c
  # ----------------------------------------------------------------------

def plot_figure_scanline(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color):
  # Plots the figures with scanline tool-path using the rasters {OPHS}: 
  #
  #   subfig = 0 non-alternating
  #
  #   subfig = 1 alternating
  #
  # Returns the canvas {c} with the figure.
  
  assert subfig == 0 or subfig == 1, ("invalid subfigure number %d.\n" % subfig)
  alt = (subfig == 1) # True for alternating path.
  
  dp = (0,0)
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  c = make_figure_canvas(Bfig, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Reverse alpernate scanlines and get list {CTSsel} of selected contacts to plot:
  if alt:
    # Reverse order and orientation of raster elements of {OPHS} on alternate scanlines:
    PFSnew = [] # Reaster paths, reversed if needed.
    rev = False # True if next scan-line is to be reversed.
    for Si in SCS:
      # Now {Si} is a list of indices of rasters in one scanline.
      if rev:
        PFSi = [ path.rev(OPHS[jrs]) for jrs in Si ]
        PFSi.reverse()
      else:
        PFSi = [ OPHS[jrs] for jrs in Si ]
      rev = not rev
      PFSnew += PFSi
    OPHS = PFSnew

  # Assemble the path:
  use_links = alt
  ph = path.concat(OPHS, use_links, mp_jump)
  
  # Find the coldest contact and its cooling time:
  ncold = 1 # Number of coldest contacts to show.
  CTScold = find_coldest_contacts(ph, CTS, ncold)

  plot_contours(c, OCRS, dp, style, color['ghost'])
  
  # Plot path:
  clr = color['fill']
  rwdf = style['rwd_fill']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = True
  matter = False
  path.plot_standard(c, [ph,], dp, None, [clr,], rwdf, wd_axes, axes, dots, arrows, matter)

  # Plot the path endpoints:
  wd_edots = style['wd_edots']
  plot_path_endpoints(c, [ph,], dp, style,color)

  # Plot selected contact(s).
  interbc = False
  plot_contacts(c, CTScold, dp, interbc, color, wd_axes)
    
  # Label path endpoints:
  tdA = (-1.8, -1.6)
  tdB = (-1.6, +0.5) if alt else (+0.8, +0.8)
  label_path_endpoints(c, ph, "A", tdA, "B", tdB, style,color)

  return c
  # ----------------------------------------------------------------------

def get_cold_paths(OPHS):
  # Retruns a list {CRSS} that describes a typical tool-path produced by
  # {RP3} or {slic3r}. Each element of the list {CRSS} specifies the
  # rasters that comprise a snake sub-path ("continuous raster sequence)
  # of that tool-path. It is a list of pairs {(isc,jrs)}, meaning raster {jrs}
  # of scanline {isc}.
  
  CRSS = ( 
    ( (0,0), (1,0), (2,0), (3,0), (4,0), (5,0), (6,0), (7,0), (8,0), 
      (9,0), (10,0), (11,0), (12,0), (13,0), (14,0),
      (15,0), (16,0), (17,0),
    ),
    ( (14,1), (13,1), (12,1), (11,1), (10,1), (9,1), (8,1), (7,1), (6,1), (5,1), (4,1), (3,1), ),
    ( (0,1), (1,1), (2,1), (3,3), (4,3), (5,3), (6,3), (7,2), (8,2), 
      (9,3), (10,3), (11,3), (12,3), (13,3), (14,3), (15,1),
    ),
    ( (14,2), (13,2), ),
    ( (12,2), (11,2), (10,2), (9,2), ),
    ( (6,2), (5,2), (4,2), (3,2), ),
  )
  return CRSS
  # ----------------------------------------------------------------------
  
def plot_figure_cold(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color):
  # Plots the figures with scanline path, alternating or not according to {alt}: 
  #
  #   subfig = 0 the path, with sected contact(s)
  #
  # Returns the canvas {c} with the figure.
  
  assert subfig == 0
  
  dp = (0,0)
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  c = make_figure_canvas(Bfig, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Collect the rasters in the guessed {RP3} or {slic3r} order.
  CRSS = get_cold_paths(OPHS)
  
  def choose_direction(p, ij):
    # Returns true if the raster identified by the index pair {ij}
    # should be reversed, based on which end is closer to {p}.
    isc,jrs = ij
    oph = OPHS[SCS[isc][jrs]]
    d0 = rn.dist(p, path.pini(oph))
    d1 = rn.dist(p, path.pfin(oph))
    return d0 > d1
    # ....................................................................

  # Join each Alternate directions between adjacent scanlines.
  p_prev = (15,0) # Last position of the nozzle.
  EFS = []
  for CRS in CRSS:
    # Decide orientation {rev} of first raster in {CRS}
    rev = choose_direction(p_prev, CRS[0])

    # Now collect the raster fill paths specified by {CRS}, alternating directions:
    for isc,jrs in CRS:
      oph = OPHS[SCS[isc][jrs]]
      if rev: oph = path.rev(oph)
      EFS.append(oph)
      p_prev = path.pfin(oph)
      rev = not rev
  
  # Assemble the path:
  use_links = True
  ph = path.concat(EFS, use_links, mp_jump)
      
  # Find the coldest contact and its cooling time:
  ncold = 4 # Number of coldest contacts to show.
  CTScold = find_coldest_contacts(ph, CTS, ncold)
  
  # SPlit at jumps for coloring:
  OCRS, JMPS = path.split_at_jumps(ph)
  CLRS = hacks.trace_colors(len(OCRS), None)

  # Plot trace components:
  rwdf = style['rwd_fill']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  path.plot_standard(c, OCRS, dp, None, CLRS, rwdf, wd_axes, axes, dots, arrows, matter)

  # Plot the jumps:
  rwdf = style['rwd_fill']
  wd_axes = style['wd_axes']
  axes = True
  dots = True
  arrows = True
  matter = False
  move.plot_standard(c, JMPS, dp, 3, [color['jump'],], rwdf, wd_axes, axes, dots, arrows, matter) 
  
  # Plot the path endpoints:
  plot_path_endpoints(c, [ph,], dp, style,color)

  # Plot selected contact(s).
  interbc = False
  plot_contacts(c, CTScold, dp, interbc, color, wd_axes)
  
  # Label path endpoints:
  tdA = (+0.8, -1.4)
  tdB = (-2.0, -0.4)
  label_path_endpoints(c, ph, "A", tdA, "B", tdB, style,color)
 
  return c
  # ----------------------------------------------------------------------

def get_rivers(OPHS, CTS):
  # Retruns a list {GROUPS} that describes the rivers in the filling raster set {OPHS} 
  # defined by the contacts {CTS}. Each element of the list {GROUPS} specifies the
  # rasters that comprise a river. It is a list of pairs {(isc,jrs)}, meaning raster {jrs}
  # of scanline {isc}.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  GROUPS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (3,0), (4,0), (5,0), (6,0), (7,0), (8,0), (9,0), (10,0), (11,0), (12,0), (13,0), (14,0), ),
    ( (3,1), (4,1), (5,1), (6,1), (7,1), (8,1), (9,1), (10,1), (11,1), (12,1), (13,1), (14,1), ),
    ( (15,0), (16,0), (17,0), ),
    ( (14,2), (13,2), ),
    ( (13,3), (14,3), (15,1), ),
    ( (9,2), (10,2), (11,2), (12,2), ),
    ( (9,3), (10,3), (11,3), (12,3), ),
    ( (7,2), (8,2), ),
    ( (3,3), (4,3), (5,3), (6,3), ),
    ( (3,2), (4,2), (5,2), (6,2), ),
    ( (0,1), (1,1), (2,1), ),
  )
  return GROUPS
  # ----------------------------------------------------------------------

def get_sub_rivers(OPHS, CTS):
  # Retruns a list {GROUPS} that describes the sub-rivers in the filling raster set {OPHS} 
  # defined by the contacts {CTS}.
  #
  # Each element of the list {GROUPS} specifies the rasters that comprise
  # a sub-river. It is a list of pairs {(isc,jrs)}, meaning raster {jrs} of
  # scanline {isc}.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  GROUPS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (0,1), (1,1), (2,1), ),

    ( (3,0), (4,0), (5,0), (6,0), ), 
    ( (3,1), (4,1), (5,1), (6,1), ),
    ( (3,2), (4,2), (5,2), (6,2), ),
    ( (3,3), (4,3), (5,3), (6,3), ),

    ( (7,0), (8,0), ),
    ( (7,1), (8,1), ),
    ( (7,2), (8,2), ),

    ( (9,0), (10,0), (11,0), (12,0), ),
    ( (9,1), (10,1), (11,1), (12,1), ),
    ( (9,2), (10,2), (11,2), (12,2), ),
    ( (9,3), (10,3), (11,3), (12,3), ),

    ( (13,0), (14,0), ),
    ( (13,1), (14,1), ),
    ( (13,2), (14,2), ),
    ( (13,3), (14,3), ),

    ( (15,0), ),
    ( (15,1), ),

    ( (16,0), (17,0), ),
  )
 
  return GROUPS
  # ----------------------------------------------------------------------
 
def get_essential_cut_lines(OPHS, CTS):
  # Returns a list {LNS} of essential cut-lines.
  # Each element of {LNS} is a pair {(icu, t)} where {icu} is the index of the scan-line just above 
  # the cut-line and {t} is a code (2 for essential, 1 for non-essential, 0 for not relevant). 

  LNS = (
    (0,2),
    (3,2),
    (7,2),
    (9,2),
    (13,2),
    (15,2),
    (16,2),
    (18,2),
  )

  return LNS
  # ----------------------------------------------------------------------

def plot_figure_rivers(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color):
  # Plots the figure with rasters grouped into rivers.
  #
  # Returns the canvas {c} with the figure.
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  dp = (0,0)
  c = make_figure_canvas(Bfig, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Assign group indices to rivers.
  # Each element of the list {GROUPS} below is a river. 
  # Each river is a list of pairs {(isc,jrs)}, meaning raster {jrs} of scanline {isc}.
  
  if subfig == 0:
    GROUPS = get_rivers(OPHS, CTS)
  elif subfig == 1:
    GROUPS = get_sub_rivers(OPHS, CTS)
  else:
    assert False, "invalid subfig"
  
  ngr = len(GROUPS)
  igr = 0
  for RIV in GROUPS:
    # Assign group index {igr} to rasters in {RIV}:
    for isc,jrs in RIV:
      oph = OPHS[SCS[isc][jrs]]
      assert path.nelems(oph) == 1
      path.set_group(oph, igr)
    igr += 1
  assert igr == ngr
  
  plot_contours(c, OCRS, dp, style, color['ghost'])

  # Make a color list, one for each river:
  CLRS = hacks.trace_colors(ngr, None)

  # Plot the filling path(s) with color based on group:
  rwdf = style['rwd_fill']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  for oph in OPHS:
    igr = path.get_group(oph)
    assert igr != None, "group was not assigned"
    clr = CLRS[igr]
    path.plot_standard(c, [oph,], dp, None, [clr,], rwdf, wd_axes, axes, dots, arrows, matter)

  if subfig == 0:
    interbc = True # Only inter-block contacts please
    plot_contacts(c, CTS, dp, interbc, color, wd_axes)
  elif subfig == 1:
    LNS = get_essential_cut_lines(OPHS, CTS);
    plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, style,color)

  return c
  # ----------------------------------------------------------------------
 
def get_canon_cut_lines(OPHS, CTS, icumax):
  # Returns a list {LNS} of essential and non-essential cut-lines to plot on the canonical path figure.
  # Each element of {LNS} is a pair {(icu, t)} where {icu} is the index of the scan-line just above 
  # the cut-line and {t} is a code (2 for essential, 1 for non-essential, 0 for not relevant). 

  LNS = get_essential_cut_lines(OPHS, CTS)
  LNS += (
      (5,1),
      (12,1),
    )

  # Remove excess cut lines:
  LNS = [ (icu, t) for icu, t in LNS if icu <= icumax ]

  return LNS
  # ----------------------------------------------------------------------

def get_canon_blocks(OPHS,SCS,icumax,full,mp_jump):
  # Returns a list of lists {BCSS} of the canonical snake blocks
  # for the canonical path figure, and a list {RSrest} with the rasters
  # of {OPHS} not used in the snakes.
  #
  # The parameter {OPHS} should be a list of all the filling raster elements.
  # each element oriented from left to right.
  #
  # The parameter {SCS} should be a list of lists of indices. The raster
  # element {jrs} (from 0, left to right) on scanline {isc}, as returned by
  # {raster.separate_by_scanline}.
  # 
  # The returned snake blocks are separated according to the canonical bands
  # defined by the cut-lines of {get_canon_cut_lines}, up to the cut-line {icumax}.
  # Each element {BCSS[ibd]} of the result is a list of the blocks that
  # appear in one of those bands. The returned blocks will use all the rasters
  # up to the cut-line {icumax}.  The rasters of {OPHS} above that cut-line 
  # are returned in {RSrest}.
  # 
  # Each block will be constructed from a canonical sub-river -- a
  # subset of of raster paths in {OPHS} that lie on adjacent scan-lines,
  # with exactly one contact between conscutive rasters, and span the
  # band in quation. The rasters will be connected by the link paths
  # attached to those rasters, if any.
  # 
  # If {full} is true, the choices of each block will be all the snake
  # paths that can be built from those rasters (either 2 or 4, snake
  # paths, depending on the number of scan-lines that it spans)
  #
  # If {full} is false, each block will have only one choice -- the one 
  # that starts with the bottom-most raster, oriented from left to right.
  # 
  # The links from the input rasters are copied to the block choices
  # when applicable.
  #
  # No new {Contact} objects are created, but the attachments between
  # the existing contacts and paths of {OPHS} that are used in blocks,
  # as given by {path.get_contacst} and {contact.get_side_paths}, are
  # broken and the contacts are attached to the choices of the blocks
  # instead.
  #
  # The procedure also sets group index (as returned by
  # {path.get_group}) of all original raster paths in {OPHS} to the
  # sequential index of the block in which they were used; or to {None}
  # if they were not used in any block.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  
  # {SNIXSSS} is a list of list of list of pairs. If
  # {SNIXSSS[ibd][isn][irs]} is {(isc,jrs)}, it means that raster {irs}
  # (from bottom) of snake {isn} (from left) on band {ibd} (from bottom)
  # is raster {jrs} of scanline {isc}.
  SNIXSSS = ( 
    # band (0,3):
    ( ( (0,0), (1,0), (2,0), ),
      ( (0,1), (1,1), (2,1), ),
    ),

    # Band (3,5):
    ( ( (3,0), (4,0), ), 
      ( (3,1), (4,1), ),
      ( (3,2), (4,2), ),
      ( (3,3), (4,3), ),
    ),

    # Band (5,7):
    ( ( (5,0), (6,0), ),
      ( (5,1), (6,1), ),
      ( (5,2), (6,2), ),
      ( (5,3), (6,3), ),
    ),

    # Band (7,9):
    ( ( (7,0), (8,0), ),
      ( (7,1), (8,1), ),
      ( (7,2), (8,2), ),
    ),

    # Band (9,12):
    ( ( (9,0), (10,0), (11,0), ),
      ( (9,1), (10,1), (11,1), ),
      ( (9,2), (10,2), (11,2), ),
      ( (9,3), (10,3), (11,3), ),
    ),
    
    # Band (12,13):
    ( ( (12,0), ),
      ( (12,1), ),
      ( (12,2), ),
      ( (12,3), ),
    ),
    
    # Band (13,15):
    ( ( (13,0), (14,0), ),
      ( (13,1), (14,1), ),
      ( (13,2), (14,2), ),
      ( (13,3), (14,3), ),
    ),

    # Band (15,16):
    ( ( (15,0), ),
      ( (15,1), ),
    ),
    
    # Band (16,18):
    ( ( (16,0), (17,0), ), 
    ),
  )
  
  # Clear the goup index of all raster paths in {OPHS}:
  for oph in OPHS: path.set_group(oph, None)

  nbc = 0 # Number of snake blocks created.
  
  BCSS = [] # List of lists of canonical snake blocks.
  RSrest = []
  for SNIXSSk in SNIXSSS:
    # SNIXSSk is a list of lists of rasters (represented as index pairs) in the band with index {k},
    # separated by snake. Create snake blocks of that band:
    BCSk = [] # List of canonical snake blocks in band.
    for SNIXSkj in SNIXSSk:
      # SNIXSkj is a list of index pairs of the rasters in one canonical snake block of that band.
      # Create the list of blocks {BCSk}
      OPBSj = collect_and_tag_snake_rasters(SNIXSkj, SCS, OPHS, icumax, nbc, RSrest)
      bckj = make_snake_block_from_rasters(OPBSj, full, nbc, mp_jump)
      if bckj != None: 
        BCSk.append(bckj) 
        nbc += 1
    if len(BCSk) != 0: BCSS.append(BCSk)
  
  sys.stderr.write("got %d bands of blocks and %d leftover rasters\n" % (len(BCSS), len(RSrest)))
  for ibd in range(len(BCSS)):
    SNB = BCSS[ibd]
    sys.stderr.write("  band %d has %d blocks\n" % (ibd,len(SNB)));

  return BCSS, RSrest
  # ----------------------------------------------------------------------

def get_generic_blocks(OPHS,SCS,full,mp_jump):
  # Returns a collection of snake blocks built from the raster paths in
  # {OPHS}, as could be the input to
  # {BestHotPath}. 
  #
  # For compatibility with {get_canon_blocks}, the procedure returns a
  # list {BCSS} where each element is not a block but a list that
  # contains a single block. It also also returns an (empty) list
  # {RSrest} of unused rasters, for the same reason.
  #
  # The parameter {OPHS} should be a list of all the filling raster elements.
  # each element oriented from left to right.
  #
  # The parameter {SCS} should be a list of lists of indices. The raster
  # element {jrs} (from 0, left to right) on scanline {isc}, as returned by
  # {raster.separate_by_scanline}.
  # 
  # Each block will be constructed from a subset of rasters from {OPHS}
  # that lie on adjacent scan-lines, with exactly one contact between
  # conscutive rasters. The rasters will be connected by the link paths
  # attached to those rasters, if any. Apart from that constraint, the
  # partition of rasters into blocks is arbitrarily chosen by the
  # procedure
  # 
  # If {full} is true, the choices of each block will be all the snake
  # paths that can be built from those rasters (either 2 or 4, snake
  # paths, depending on the number of scan-lines that it spans)
  #
  # If {full} is false, each block will have only one choice,
  # chosen arbitrarily.
  # 
  # The links from the input rasters are copied to the block choices
  # when applicable.
  #
  # No new {Contact} objects are created, but the attachments between
  # the existing contacts and paths of {OPHS} that are used in blocks,
  # as given by {path.get_contacst} and {contact.get_side_paths}, are
  # broken and the contacts are attached to the choices of the blocks
  # instead.
  #
  # The procedure also sets group index (as returned by
  # {path.get_group}) of all original raster paths in {OPHS} to the
  # sequential index of the block in which they were used; or to {None}
  # if they were not used in any block.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  
  # {SNIXSS} is a list of lists of pairs.  If {SNIXSS[ibc][irs]} is {(isc,jrs)}, 
  # it means that raster {irs} (from bottom) of block {ibc} 
  # is raster {jrs} of scanline {isc}.
  SNIXSS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (0,1), (1,1), (2,1), ),

    ( (3,0), (4,0), ),
    ( (5,0), (6,0), ), 
    ( (7,0), (8,0), ),
    ( (9,0), (10,0), ),
    ( (11,0), (12,0), ),
    ( (13,0), (14,0), ),
    
    ( (3,1), (4,1), (5,1), (6,1), ),
    ( (7,1), (8,1), (9,1), (10,1), ),
    ( (11,1), (12,1), (13,1), (14,1), ),

    ( (3,2), (4,2), (5,2), (6,2), ),

    ( (7,2), (8,2), ),

    ( (9,2), (10,2), ),
    ( (11,2), (12,2), ),
    ( (13,2), (14,2), ),

    ( (3,3), (4,3), (5,3), (6,3), ),
    ( (9,3), (10,3), (11,3), (12,3), ),

    ( (13,3), (14,3), (15,1), ),

    ( (15,0), ),
    ( (16,0), (17,0), ), 
  )
  
  # Clear the goup index of all raster paths in {OPHS}:
  for oph in OPHS: path.set_group(oph, None)
  
  nsc = len(SCS) # Number of scan-lines.
  nbc = 0 # Number of blocks created.
  
  BCSS = [] # List of lists of canonical snake blocks.
  RSrest = []
  for SNIXSj in SNIXSS:
    # SNIXSj is a list of index pairs, specifying the rasters of one block.
    # Create the block {bcj}
    icumax = nsc # Cut-line above last scan-line.
    OPBSj = collect_and_tag_snake_rasters(SNIXSj, SCS, OPHS, icumax, nbc, RSrest)
    bcj = make_snake_block_from_rasters(OPBSj, full, nbc, mp_jump)
    assert bcj != None
    BCSS.append([bcj,]) 
    nbc += 1

  # All rasters must have been used:
  assert len(RSrest) == 0
  
  sys.stderr.write("got %d blocks\n" % len(BCSS))
  for jbc in range(len(BCSS)):
    bcj = BCSS[jbc][0]
    if full:
      assert isinstance(bcj, block.Block)
      nchj = block.nchoices(bcj)
      nrsj = path.nelems(block.choice(bcj,0))
      sys.stderr.write("  block %d has %d choices of %d rasters\n" % (jbc,nchj,nrsj))
    else:
      assert isinstance(bcj, path.Path)
      nrsj = path.nelems(bcj)
      sys.stderr.write("  path %d has %d rasters\n" % (jbc,nrsj));

  RSrest = []
  return BCSS, RSrest
  # ----------------------------------------------------------------------

def collect_and_tag_snake_rasters(SNIXS, SCS, OPHS, icumax, ibc, RSrest):
  # Parameters {SCS}, {OPHS}, {icumax} are as in {get_canon_blocks}.
  # Parameter {SNIXS} is a list of pairs {(isc,jrs)} that identify the rasters of 
  # one canonical snake block.  Returns the list {OPKS} of those raster paths,
  # in the same orientation as they are in {OPHS}.
  #
  # However, only rasters below the cut-line {icumax} are used in the snake block.
  # Rasters above that cut-line are instead appended to the list {RSrest} 
  # with no change.  If all rasters specified by {SNIXS} are above that cut-line,
  # returns the empty list.
  #
  # Also sets the group index of all the selected raster paths to {ibc}.

  OPKS = [] # Raster elements in this snake.
  for isc, jrs in SNIXS:
    ors = OPHS[SCS[isc][jrs]]
    if isc >= icumax:
      RSrest.append(ors)
    else:
      path.set_group(ors, ibc)
      OPKS.append(ors)
  return OPKS
  # ----------------------------------------------------------------------

def make_snake_block_from_rasters(OPBS, full, ibc, mp_jump):
  # Builds and returns a snake block from the raster paths in the list {OPBS}. They should be in consecutive scan-lines
  # and all oriented from left to right.  
  #
  # The raster paths in {OPBS} should all have group index {ibc}, which
  # should be distinct from the groups of al other original raster paths
  # and of all choices of previously created blocks.
  #
  # If {full} is true, returns a block whose choices are all snake paths
  # that can be built from the rasters on {OPBS}; usually 2 if {OPBS}
  # has a single raster, and 4 if it has two or more. If {full} is
  # false, returns a block with only one choice, specifically the snake
  # path that begins with the bottom-most raster oriented from left to
  # right.
  #
  # However, if the list {OPBS} is empty, returns {None}.
  #
  # Also copies the applicable link paths from the raster paths in {OPBS}
  # to the choices of the new block.
  #
  # Also modifies the contact-path attachments of to refer to the block
  # choices instead of those raster paths. Specifically, any contact
  # {ct} that is attached to a raster path {ors} of {OPBS} and to some
  # path not in {OPBS} (original raster or choice of a previosuly
  # created block) will be detached from {ors} and attached to all the
  # choices of the new block.

  # Collect raster paths {OPHS0,OPHS1} of the snake and (if {full}) of the "mirror" snake:
  OPHS0 = [] # Raster elements in this snake.
  OPHS1 = [] # Reversed raster elemens in this snake (if {full}).
  rev = False # Should reverse the next raster?
  for opb in OPBS:
    assert path.get_group(opb) == ibc
    if rev: opb = path.rev(opb)
    OPHS0.append(opb)
    if full: OPHS1.append(path.rev(opb))
    rev = not rev

  # Did we get any rasters at all?
  if len(OPHS0) == 0: return None
  
  # Now join the rasters in each of {OPHS0} and {OPHS1} into a snake block
  use_links = True
  ph0 = path.concat(OPHS0, use_links, mp_jump) # The snake path.
  if not full: 
    bc = block.from_paths([ph0,])
  else:
    ph1 = path.concat(OPHS1, use_links, mp_jump) # The other snake path.
    if len(OPHS0) == 1:
      bc = block.from_paths([ph0, ph1])
    else:
      bc = block.from_paths([ph0, path.rev(ph0), ph1, path.rev(ph1)])

  reattach_snake_block_contacts(OPHS0, bc, ibc)
 
  for ich in range(block.nchoices(bc)): path.set_group(block.choice(bc, ich), ibc)

  return bc
  # ----------------------------------------------------------------------
 
def reattach_snake_block_contacts(OPHS, bc, ibc):
  # Given a list of raster-fill paths {OPHS} that were used to make one
  # snake block. detaches them from their attached contacts and attaches
  # the latter to the choices of the block {bc}.
  #
  # On input, all the raster paths in {OPHS} should have group index
  # {ibc}, and all other paths (original raster paths or choices of
  # previoulsy created blocks) must have group index {None} or strictly
  # less than {ibc}.  The group index of the choices of {bc} is not used.

  debug = False

  # Get all the contacts {CTS} whose sides were used in this block:
  CTS = set()
  for ophi in OPHS:
    for isd in range(2):
      CTS = set.union(CTS, path.get_contacts(ophi, isd))
  
  # Fix the attachments of all those contacts:
  for ct in CTS:
    if debug: contact.show(sys.stderr, "  ct = ", ct, "\n", 0)
    for ksd in range(2):
      # Get the paths that are attached to side {ksd} of {ct}.
      # These are either a single old raster path, possibly used in this block, 
      # or one or more choices of a previously conctructed block.
      SDPSk = contact.get_side_paths(ct, ksd)
      assert len(SDPSk) > 0 # Since the contact must originally have had have both sides on raster paths of {OPHSD}.
      
      # Set {ibck} to the index of the block that contains side {k} of {ct} (which may be {ibc})
      # or to {None} if that side is still not in any block:
      # sys.stderr.write("ibc = %d contact %s side %d SPDSk = %s:\n" % (ibc, contact.get_name(ct),isd, str(SDPSk)))
      igrk = None
      for phkj, drkj, imvkj in SDPSk:
        igrkj = path.get_group(phkj)
        if debug: path.show(sys.stderr, ("    side %d @ " % isd), phkj, (" igr = %s\n" % str(igrkj)), True, 0,0)
        # sys.stderr.write("  igrkj = %s igrk = %s\n" % (str(igrkj),str(igrk)))
        if igrkj == ibc or igrkj == None:
          # This side of the contact is in one of original rasters, either used by this
          # block or not yet used by any block:
          assert igrk == None # Since there should be only one such raster.
          assert len(SDPSk) == 1 # Since there should be only one such raster.
          assert imvkj == 0 # Since those original rasters paths have length 1.
        else:
          # This side of the contact is in a choice of a previously constructed block.
          assert igrkj < ibc # Assuming blocks are numbered consecutively by creation.
          assert igrk == None or igrk == igrkj # Since all side paths must be in the same block.
        igrk = igrkj 
        
      if igrk == ibc:
        # Path attachments on side {ksd} of {ct} must be changed from old raster to choices of new block:
        contact.clear_side_paths(ct, ksd)
        for phkj, drkj, imvkj in SDPSk: path.clear_contacts(phkj, ksd)
          
        # Attach side {ksd} of {ct} to the choices of {bc}
        mvk = contact.side_move(ct, ksd)
        for ich in range(block.nchoices(bc)):
          ochi = block.choice(bc, ich)
          jmvki = path.find_move(ochi, mvk)
          if jmvki != None:
            path.add_contact(ophi, ksd, ct)
            contact.add_side_path(ct, ksd, ochi, jmvki)
  return
  # ----------------------------------------------------------------------

def plot_figure_canon(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color):
  # Plots the figure with canonical paths and canonical bandpath.
  #
  # Returns the canvas {c} with the figure.
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  c = make_figure_canvas(Bfig, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Build a list of list of blocks {BCSS} with one choice each, the canonical snake,
  # up to some cut-line {icumax} below the top one, separated into bands:
  
  dp = None
  icumax = 12
  full = False  
  BCSS, RSrest = get_canon_blocks(OPHS,SCS,icumax,full,mp_jump)

  # Plot the relevant cut-lines that separate the chosen bands:
  LNS = get_canon_cut_lines(OPHS, CTS, icumax);
  plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, style,color)
  
  # Concatenate the canonical snake paths in {BCSS} into a {(k,i)} canonical path
  # {cph}, up to the next-to-lasr band, and the following {(i,j)} canonical
  # band-path {bph}:
  nbd = len(BCSS)
  CPS = [] # List of canonical snake paths that make up {cph}
  BPS = [] # List of canonical snake paths that make up {bph}
  for ibd in range(nbd):
    for bc in BCSS[ibd]: # Blocks in band {ibd}
      assert block.nchoices(bc) == 1 # Since {full} was false.
      och = block.choice(bc, 0)
      (BPS if ibd == nbd-1 else CPS).append(och)
  use_links = True
  cph = path.concat(CPS, use_links, mp_jump)
  bph = path.concat(BPS, use_links, mp_jump)

  plot_contours(c, OCRS, dp, style, color['ghost'])
  
  # Plot the canonical path and the canonical sub-path:
  rwdf = style['rwd_fill']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  ccph = color['fill'] # Color for the canonical {(k,i)} path.
  path.plot_standard(c, [cph,], dp, None, [ccph,], rwdf, wd_axes, axes, dots, arrows, matter)

  cbph = color['hifi'] # Color for the canonical {(i,j)} band-path.
  path.plot_standard(c, [bph,], dp, None, [cbph,], rwdf, wd_axes, axes, dots, arrows, matter)
  
  # Plot dots at the path endpoints:
  wd_edots = style['wd_edots']
  plot_path_endpoints(c, [cph,], dp, style,color)
  plot_path_endpoints(c, [bph,], dp, style,color)

  # Plot the unused rasters:
  for oph in RSrest:
    path.plot_standard(c, [oph,], dp, None, [color['ghost'],], rwdf, wd_axes, axes, dots, arrows, matter)
    
  # Plot the path endpoint labels:
  tdA = (-1.8, -1.6)
  tdB = (-1.8, -0.4)
  label_path_endpoints(c, cph, "A", tdA, "B", tdB, style,color)
  tdC = (-2.0, -0.1)
  tdD = (+0.7, -1.0)
  label_path_endpoints(c, bph, "C", tdC, "D", tdD, style,color)
  
  # ??? Should plot the contacts between {cph} and {bph} ???

  return c
  # ----------------------------------------------------------------------

def plot_figure_blocks(subfig, OCRS, OPHS, OLKS, CTS, mp_jump, style,color):
  # If {subfig} is 0, plots the figure with canonical blocks as could be selected by {HotPath}, with cut-lines
  # If {subfig} is 1, plots the figure with the generic blocks, including links and contacts. 
  # If {subfig} is 2, plots the four alternatives of a selected block from the generic blocks figure. 
  # If {subfig} is 3, plots the contact graphs of the generic blocks. 
  #
  # Returns the canvas {c} with the figure.

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  nsc = len(SCS) # Number of scan-lines.
  
  # Create the blocks to be plotted, as a list {BCSS} of lists of
  # blocks. For the sub-figures with canonical blocks, each element of
  # {BCSS} is a list of the blocks of one of the bands defined by the
  # cut-lines returned by {get_canon_cut_lines}. For the sub-figures
  # with generic blocks, each element of {BCSS} is a list with a single
  # block.

  # Get the blocks, canonical or generic, possibly with links and contacts or cut-lines,
  # comprising all rasters; and pick a block {bcX} to be highlighted or expanded;

  dp = (0,0)

  full = True  # Generate blocks with all possible choices.
  if subfig == 0:
    # Get the canonical blocks, in all the bands up to the top:
    icumax = nsc # Go all the way to the top.
    BCSS, RSrest = get_canon_blocks(OPHS,SCS,icumax,full,mp_jump)
    assert len(RSrest) == 0
    # No selected block:
    ibcX = None
  elif subfig == 1 or subfig == 2 or subfig == 3:
    # Generic blocks without cut-lines
    BCSS, RSrest = get_generic_blocks(OPHS,SCS,full,mp_jump)
    # Select a block to highlight in the generic blocks figure:
    ibcX = 16
  else:
    assert False, ("invalid subfig = %d" % subfig)

  # Make a flat list {BCS} of all blocks:
  BCS = [ bc for BCSi in BCSS for bc in BCSi ]
  nbc = len(BCS) # Number of blocks.
  bcX = None if ibcX == None else BCS[ibcX]
  
  # Paranoia:
  for ibc in range(len(BCS)):
    bc = BCS[ibc]
    for ich in range(block.nchoices(bc)):
      och = block.choice(bc,ich)
      assert path.get_group(och) == ibc
  
  # Determine the figure's bounding box {Bfig} and the cut-line X range {cuxlo,cuxhi}:
  if subfig == 2:
    B = block.bbox([bcX,], True, False)
    Xstep = B[1][0] - B[0][0] + 3.0 # # Displacement between choices.
    Bfig = (B[0], rn.add(B[1], (3*Xstep, 0)))
    wdf = style['wd_fill']
    mrg = (1.0*wdf, 1.0*wdf) # To account for sausage overflow and links.
    Bfig = rn.box_expand(Bfig, mrg, mrg)
    cuxlo = None; cuxhi = None
  else:
    B = path.bbox(OCRS+OPHS+OLKS)
    Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)

  c = make_figure_canvas(Bfig, style,color)

  if subfig == 0 or subfig == 1 or subfig == 3:

    # Plot either the cut-lines for the canonical blocks figure,
    # or the link paths for the generic blocks figure:
    if subfig == 0:
      # Plot the cut-lines used to define the canonical blocks:
      LNS = get_canon_cut_lines(OPHS, CTS, icumax)
      plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, style,color)
    elif subfig == 1:
      # Plot the links beneath the blocks:
      for bc in BCS:
        plot_block_links(c, [bc,], dp, ystep,yphase, style,color)
    elif subfig == 3:
      # Neither cut-lines nor links:
      pass
    else:
      assert False

    plot_contours(c, OCRS, dp, style, color['ghost'])

    # Plot the blocks, highlighting the selected one:
    CLRS = hacks.trace_colors(nbc, 0.450)
    if ibcX != None: CLRS[ibcX] = color['hifi']
    if subfig == 3:
      CLRS = ghostify_colors(CLRS, color['ghost'])
    rwdf = style['rwd_fill']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = False
    matter = False
    for ibc in range(nbc):
      bc = BCS[ibc]
      clr =  CLRS[ibc]
      assert isinstance(bc, block.Block)
      nch = block.nchoices(bc)
      for ich in range(nch):
        oph = block.choice(bc, ich)
        assert path.get_group(oph) == ibc
        path.plot_standard(c, [oph,], dp, None, [clr,], rwdf, wd_axes, axes, dots, arrows, matter)

    if subfig == 1:
      # Plot the inter-block contacts:
      interbc = True # Only inter-block contacts.
      plot_contacts(c, CTS, dp, interbc, color, wd_axes)
    elif subfig == 3:
      # Compute the vertices of the contact graph as the barycenters of blocks.
      VGS = []
      for bc in BCS:
        pbar = block.barycenter(bc)
        VGS.append(pbar)
      # Compute the edges of the graph:
      EGS = []
      for ct in CTS:
        igr = contact_side_groups(ct)
        assert igr[0] != None and igr[1] != None, "contact side not in any block"
        if igr[0] != igr[1]:
          EGS.append(igr)
      # Plot the graph:
      plot_contact_graph(c, VGS, EGS, style,color) 
        
  elif subfig == 2:
  
    # Figure shows the alternatives of the selected block from the generic blocks figure, with attached links:
    
    assert ibcX != None
    bcX = BCS[ibcX]
    assert block.nchoices(bcX) == 4
    
    clr = color['hifi']
    rwdf = style['rwd_fill']
    wd_axes = style['wd_axes']
    axes = False
    dots = False
    arrows = True
    matter = False
    nch = block.nchoices(bcX)
    for ich in range(nch):
      dp = ( ich*Xstep, 0.0 )
      oph = block.choice(bcX, ich)
      plot_path_links(c, [oph,], dp, ystep,yphase, style,color)
      path.plot_standard(c, [oph,], dp, None, [clr,], rwdf, wd_axes, axes, dots, arrows, matter)
      # ??? Should show the attached contacts too ???
  
  else:
    assert False, "invalid subfig %d\n" % subfig

  return c
  # ----------------------------------------------------------------------

# GENERAL UTILITIES

def plot_path_endpoints(c, OPHS, dp, style,color):
  # Plots the endpoints of the paths in {OPHS}.
  
  clr = color['dots']
  wd_edots = style['wd_edots']
  for oph in OPHS:
    for p in path.endpoints(oph):
      q = p if dp == None else rn.add(p, dp)
      hacks.plot_line(c, clr, wd_edots, None, q, q)
  return
  # ----------------------------------------------------------------------
  
def contact_side_groups(ct):
  # Returns the group indices of the paths that contain the two sides of {ct}.
  # Uses {contact.get_side_paths} and {path.get_group}. 
  # Each side must have at least one attached path, and all groups attached 
  # to the same side must have the same group index.
  
  debug = False
  if debug: contact.show(sys.stderr, "  ct = ", ct, "\n", 0)
  igr = [-1, -1] # Group indices on the two sides of contact:
  for isd in range(2):
    for phij, drij, imvij in contact.get_side_paths(ct, isd):
      igrij = path.get_group(phij)
      assert igrij == None or igrij >= 0
      if debug: path.show(sys.stderr, ("    side %d @ " % isd), phij, (" igr = %s\n" % str(igrij)), True, 0,0)
      if igr[isd] == -1:
        igr[isd] = igrij
      else:
        assert igr[isd] == igrij, ("side %d of contact on different groups" % isd)
    assert igr[isd] != -1, "contact has no side paths"
  return tuple(igr)
  # ----------------------------------------------------------------------

def plot_contacts(c, CTS, dp, interbc, color, wd_axes):
  # Plots the contacts in {CTS} with the style used in the paper, with color {color['ctac']}.
  # If {interbc} is true, plots only contacts between paths that have different group indices.
  
  wd_ctac = 1.7*wd_axes
  dashpat = (1.0*wd_ctac, 2.0*wd_ctac)
  ext = 0
  sz_tics = 0
  arrows_ct = False
  ncp = 0 # Interblock contacts plotted.
  for ct in CTS:
    igr = contact_side_groups(ct)
    plotit = True if not interbc else igr[0] != igr[1]
    if plotit:
      contact.plot_single(c, ct, dp, color['ctac'], dashpat, ext, wd=wd_ctac, sz_tic=sz_tics, arrow=arrows_ct)
      ncp += 1
      
  sys.stderr.write("plotted %d out of %d contacts\n" % (ncp,len(CTS)))
  return
  # ----------------------------------------------------------------------

def plot_path_links(c, OPHS, dp, ystep,yphase, style,color):
  # Plots onto {c} the link paths of the paths in {OPHS}.  The color is chosen
  # depending on the parity of the scan-line index of the lowest endpoint of the link.
  
  rwd_link = style['rwd_link']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  for oph in OPHS:
    for ophr in oph, path.rev(oph):
      OLKS = path.get_links(ophr)
      for ilk in range(len(OLKS)):
        olki = OLKS[ilk]
        # Make sure the link is directed up:
        if path.pini(olki)[1] > path.pfin(olki)[1]: olki = path.rev(olki)
        # Determine the scan-line index:
        isc = raster.point_scanline_index(path.pini(olki), (1,0), (0,1), ystep, yphase)
        clr = color['link0'] if isc % 2 == 0 else color['link1']
        path.plot_standard(c, [olki,], dp, None, [clr,], rwd_link, wd_axes, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

def plot_block_links(c, BCS, dp, ystep,yphase, style,color):
  # Plots onto {c} the link paths of the choices of the blocks in {BCS}.  The color is chosen
  # depending on the parity of the scan-line index of the lowest endpoint of the link.
  
  for bc in BCS:
    nch = block.nchoices(bc)
    for ich in range(nch):
      oph = block.choice(bc, ich)
      plot_path_links(c, [oph,], dp, ystep,yphase, style,color)
  return
  # ----------------------------------------------------------------------
    
def find_coldest_contacts(oph, CTS, n):
  # Finds the {n} contacts in {CTS} that are closed by the path {oph}
  # and have the maximum cooling time.
  WORST = [] # List of pairs {(ict, tc)} where {ict} is index into {CTS} and {tc} is its cooling time.
  for ict in range(len(CTS)):
    ct = CTS[ict] 
    tc = contact.tcool(oph, ct)
    if tc != None:
      WORST.append((ict, tc))
  WORST.sort(key = lambda x: -x[1])
  if len(WORST) > n: WORST = WORST[0:n]
  CTScold = []
  if len(WORST) > 0:
    sys.stderr.write("coldest contacts (%d):\n" % len(WORST))
    for ict, tc in WORST:
      ct = CTS[ict]
      sys.stderr.write("  %10.6f s %s\n" % (tc, contact.get_name(ct)))
      CTScold.append(ct)
  else:
    assert False, "no contacts are closed by the path."
  return CTScold
  # ----------------------------------------------------------------------

def label_point(c, p, lab, td, style,color):
  # Places label {lab} next with the lower left corner at the point {p}
  # displaced by {td}.  Also plots a white shadow all around.
  
  fsize = style['tfsize'] # Font size
  pt = rn.add(p, td)
  tad = 0.002*fsize
  for kx in range(5):
    for ky in range(5):
      dx = kx - 2; dy = ky - 2;
      if dx != 0 or dy != 0:
        ptk = rn.add(pt, (dx*tad, dy*tad))
        txk = r"\textbf{\textsf{%s}}" % lab
        hacks.plot_text(c, fsize, 'white', ptk, txk)
  tx = r"\textbf{\textsf{%s}}" % lab
  hacks.plot_text(c, fsize, None, pt, tx)
  return
  # ----------------------------------------------------------------------

def label_path_endpoints(c, oph, labA, tdA, labB, tdB, style,color):
  # Places labels {labA} and {labB} next to the start and end of 
  # path {oph}, displaced by {tdA} and {tdB}, respectively.
  
  label_point(c, path.pini(oph), labA, tdA, style,color)
  label_point(c, path.pfin(oph), labB, tdB, style,color)
  return
  # ----------------------------------------------------------------------

def plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, style,color):
  # The parameter {LNS} must be a list of pairs {(icu, t)}, as in the 
  # output of {get_essential_cut_lines}.
  # Plots the essential ({t=2}) and non-essential ({t=1}) cut-lines listed in {LNS}.
  # Ignores cut-lines with {t=0}
  #
  wd_cuts = style['wd_cuts'] 
  for icu, t in LNS:
    if t != 0:
      clr = color['cuess'] if t == 2 else color['cunon']
      dashpat = [ 0.100, 0.250 ] if t == 1 else None
      y = ystep*(icu-0.5) + yphase
      p = (cuxlo, y)
      q = (cuxhi, y)
      hacks.plot_line(c, clr, wd_cuts, dashpat, p, q)

      # Label the cut-line:
      fsize = 40
      td = (1,-0.3)
      pt = rn.add(q, td)
      tx = r"\texttt{%d}" % icu
      hacks.plot_text(c, fsize, None, pt, tx)

  return
  # ----------------------------------------------------------------------

def make_move_parms():
  # Returns the {MoveParms} records to use. They should be the same as 
  # used in the tests section of the paper, exccept that the trace width 
  # will be 1.0 (instead of 0.4) to match the raster spacing in the figures.
  
  ac = 3000      # Acceleration/deceleration for moves and jumps (mm/s^2).
  sp_trace = 40  # Cruise speed for traces (mm/s).
  sp_jump = 130  # Cruise speed for jumps (mm/s).
  ud = 0.05      # Trace/jump transition penalty (s).
  
  wd_cont = 0.75  # Contour trace width (mm).
  wd_fill = 1.00  # Fill trace width (mm).
  
  mp_cont = move_parms.make(wd_cont, ac, sp_trace, 0.0)
  mp_fill = move_parms.make(wd_fill, ac, sp_trace, 0.0)
  mp_jump = move_parms.make(0.0,     ac, sp_jump,  0.0)

  return mp_cont, mp_fill, mp_jump
  # ----------------------------------------------------------------------

def make_style_dict(mp_cont, mp_fill):
  # Returns a Python dict with the plot dimension parameters ({wd_fill}, {wd_axes}, etc) to be used 
  # in figures of the paper.
  
  wd_cont = move_parms.width(mp_cont)
  wd_fill = move_parms.width(mp_fill)
  wd_min = min(wd_fill,wd_cont)
  wd_axes = 0.15*wd_min

  rwd_path = 0.80 # Plot trace sausages with relative width {rwd_path} on isolated path plots.
  rwd_fill = 0.50 # Plot fill trace sausages with relative width {rwd_fill} on turkey plots.

  wd_fisa = rwd_fill*wd_fill # Width of plotted sausages in fillin.

  style = {
    'mfsize':     36,            # Font size for {hacks.plot_text} (math).
    'tfsize':     48,            # Font size for {hacks.plot_text} (points in turkey figures).
    
    'cuext':      2.0,           # Extra left and right overhang of cut-lines, excl. label (mm).
    'culab':      2.5,           # Estimated width of cut-line labels, including space (mm).

    'wd_cont':    wd_cont,       # Nominal width of contour traces.
    'wd_fill':    wd_fill,       # Nonimal width of fill traces.

    'wd_ctac':    wd_axes,       # Width of contact lines.

    'wd_axes':    wd_axes,       # Width of jumps and axis lines.
    'wd_cuts':    wd_axes,       # Width of cut-lines.
    'wd_edots':   0.75*wd_fisa,  # Diameter of black dots at end of paths.
    'wd_edges':   1.5*wd_axes,   # Width of graph edges.
    'wd_verts':   6.0*wd_axes,   # Diameter of graph vertices.
    
    'rwd_path':   rwd_path,       # Relative width of fill trace sausages in isolated path plots.
    'rwd_fill':   rwd_fill,       # Relative width of fill trace sausages in turkey plots.
    'rwd_cont':   rwd_fill,       # Relative width of contour trace sausages in turkey plots.
    'rwd_link':   0.67*rwd_fill,  # Relative width of link trace sausages in turkey plots.
    'rwd_matter': 1.13,           # Relative width of material footprint in plots.

    'wd_mfig':    34.0,           # Standard width of figures with math formulas.
  }
  return style
  # ----------------------------------------------------------------------
  
def make_color_dict():
  # Returns a Python dict with the standard color scheme for paper figures.
  color = {
    'white':  pyx.color.rgb( 1.000, 1.000, 1.000 ),  # Color for invisible frame and label shadow.
    'black':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Default color.
    'matter': pyx.color.rgb( 0.900, 0.870, 0.850 ),  # Color for estimated material footprints.
    'fill':   pyx.color.rgb( 0.000, 0.700, 0.050 ),  # Color for relevant fill traces.
    'hifi':   pyx.color.rgb( 0.000, 0.700, 1.000 ),  # Color for highlighted fill traces.
    'cont':   pyx.color.rgb( 0.600, 0.700, 0.800 ),  # Color for contours, when somewhat relevant.
    'ghost':  pyx.color.rgb( 0.850, 0.850, 0.850 ),  # Color for non-relevant traces.
    'ctac':   pyx.color.rgb( 1.000, 0.200, 0.000 ),  # Color for contacts.
    'dots':   pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of dots (maybe not used).
    'link0':  pyx.color.rgb( 1.000, 0.500, 0.000 ),  # Color of links above even scan-lines.
    'link1':  pyx.color.rgb( 0.000, 0.833, 1.000 ),  # Color of links above odd scan-line.
    'edges':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color or edges of contact graph
    'verts':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of vertices of contact graph.
    'jump':   pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of jumps (maybe not used).
    'cuess':  pyx.color.rgb( 1.000, 0.200, 1.000 ),  # Color of essential cut-lines.
    'cunon':  pyx.color.rgb( 0.400, 0.400, 1.000 ),  # Color of non-essential cut-lines.
  }
  return color
  # ----------------------------------------------------------------------
  
def make_figure_canvas(Bfig, style,color):
  # Creates the canvas for a figure whose contents has bounding box {Bfig}.
  
  # Extra extra margin:
  mrg = (0.2,0.2)
  Bplot = rn.box_expand(Bfig, mrg, mrg)

  dp = None
  c, szx, szy = hacks.make_canvas(Bplot, dp, False, False, 1, 1)

  # Plot an invisible frame to keep the figure sizes uniform:
  wd_frame = 0.5*style['wd_axes']
  clr_frame = color['white'] # Invisible.
  # clr_frame = color['black'] # Just to check.
  hacks.plot_frame(c, clr_frame, wd_frame, dp, Bfig, wd_frame/2)
  
  return c
  # ----------------------------------------------------------------------
  
def get_turkey_figure_bbox(OCRS, OPHS, OLKS, style):
  # Given the lists of contours {OCRS}, raster paths {OPHS}, and link
  # paths {OLKS} of the "turkey" slice, returns a bounding box suitable
  # for {make_canvas}.
  #
  # The box includes all the move endpoints in those paths, plus some extra space on the sides for 
  # the cut-lines that may be inserted in it, and some extra space to account for matter and sausage width.
  #
  # Also returns the abscissas {cuxlo} and {cuxhi} of the endpoints of cut-lines (excluding the labels).

  # Get the bounding box {B} of all contents:
  B = path.bbox(OPHS)
  Bsize = rn.sub(B[1], B[0])
  sys.stderr.write("bounding box (excl. contours) = %6.2f x %6.2f mm\n" % (Bsize[0], Bsize[1]))
  if len(OCRS) != 0: B = rn.box_join(B, path.bbox(OCRS))
  if len(OLKS) != 0: B = rn.box_join(B, path.bbox(OLKS))
  # Assume that the contacts are automatically included in the box.

  # Compute X-range of cut-lines:
  cuxlo = B[0][0] - style['cuext']
  cuxhi = B[1][0] + style['cuext']

  # Expand box for cut-lines and their labels:
  dxcut = style['cuext'] + style['culab']
  Bfig = B
  Bfig = rn.box_expand(Bfig, (dxcut,0.3), (dxcut,0.3))

  # Extra margin for sausage widths:
  wdf = style['wd_fill']
  mrg = (wdf, wdf)
  Bfig = rn.box_expand(Bfig, mrg, mrg)

  return Bfig, cuxlo, cuxhi
  # ----------------------------------------------------------------------
    
def plot_trace_matter(c, OCRS, OPHS, OLKS, dp, style,color):
  # Plot all the matter traces:
  clr = color['matter']
  jmp = False     # Plot traces only.
  wd_matter = 0   # Extra width of material footprint.
  rwd_matter = style['rwd_matter']
  dashed = False
  wd_dots = 0
  sz_arrows = 0
  for ph in OCRS + OPHS + OLKS: 
    path.plot_layer(c, ph, dp, jmp, clr, rwd_matter, wd_matter, dashed, wd_dots, sz_arrows)
  return
  # ----------------------------------------------------------------------
  
def plot_contours(c, OCRS, dp, style, clr):
  # Plot the contour paths {OCRS} with color {clr}:

  rwd_cont = style['rwd_cont']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  path.plot_standard(c, OCRS, dp, None, [clr,], rwd_cont, wd_axes, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

def plot_math_label(c, dp, qP, qD, tx, style):
  # Plots the text {tx} at point {dp+qP+qD} with the standard font size from {style}.
  # If {dp} is {None}, assumes {(0,0)}
  
  q = rn.add(qP, qD)
  if dp != None: q = rn.add(q, dp)
  fsize = style['mfsize']
  hacks.plot_text(c, fsize, None, q, tx)
  return 
  # ----------------------------------------------------------------------
    
def plot_contact_graph(c, VGS, EGS, style,color):
  # Plots the contact graph given the vertices {VGS} (a list of points) and the edges {EGS}
  # (a list of pairs of indices into {VGS}).

  # Plot edges:
  wd_edges = style['wd_edges']
  clr_edges = color['edges']
  for e in EGS:
    p = VGS[e[0]]
    q = VGS[e[1]]
    hacks.plot_line(c, clr_edges, wd_edges, None, p, q)

  # Plot vertices:
  wd_verts = style['wd_verts'] # Diameter of vertices.
  clr_verts = color['verts']
  for p in VGS:
    hacks.plot_line(c, clr_verts, wd_verts, None, p, p)
  return
  # ----------------------------------------------------------------------

def ghostify_colors(CLRS, cgh):
  # Mixes a lot of {cgh} into each color of {CLRS}.
  CLRS_new = []
  f = 0.3
  for clr in CLRS:
    R = f*clr.r + (1-f)*cgh.r
    G = f*clr.g + (1-f)*cgh.g
    B = f*clr.b + (1-f)*cgh.b
    clr_new = pyx.color.rgb(R,G,B)
    CLRS_new.append(clr_new)
  return CLRS_new
  # ----------------------------------------------------------------------
  
def widen_box_for_math_figure(B, style):
  wd_mfig = style['wd_mfig']
  xmrg = (wd_mfig - (B[1][0] - B[0][0]))/2
  assert xmrg >= 0, "math-containing box is wider than standard"
  B = rn.box_expand(B, (xmrg,0), (xmrg,0))
  return B
  # ----------------------------------------------------------------------
 
