/* Last edited on 2008-05-24 12:20:24 by stolfi */ 

/*
 * spr2D.c
 * 29/11/2006
 *
 * Demonstrates sparse decompositions based on iterated 
 * interpolation in 2D   Gaussian
 * Available geometries: empty domain, closed box, open horn.
 */

/* We need to define _GNU_SOURCE to get {asprintf}. */
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <values.h>

#include "int2d.h"
#include "mwplot.h"

/* The maximum number of samples that we can have: */
#define NPx 100
#define NPy 100

#define SIGMA_MAX (1e7)
  /* Nominal conductivity {sigma} of metals. */

/* NON-HOMOGENEOUS MEDIUM

  Non-homogeneous media are simulated by setting 
  the appropriate elements of the {sigma} field. */
  
void setup_initial_state
  ( int geom, 
    int nx, 
    int ny, 
    double dx, 
    double dy, 
    double **sigma, 
    double **hz, 
    double **ex, 
    double **ey
  );
  /* Sets the conductivity map {sigma} according to the
    code {geom} that describes the domain's geometry.  Currently:
    
      0 -> homogeneous non-conductive medium (vacuum).
      1 -> closed square conductive box.
      2 -> conductive microwave horn.
      
    Also sets the initial state of the magnetic field {hz} and of
    the electric field {ex,ey} to values appropriate to that 
    geometry. 
  */

void setup_geometry_box(int nx, int ny, double **sigma);
  /* Draws into {sigma[0..nx][0..ny]} the outline of a microwave horn,
    with open mouth and highly conductive sides. */

void setup_geometry_horn(int nx, int ny, double **sigma);
  /* Draws into {sigma[0..nx][0..ny]} a closed square box, with empty
    interior and highly conductive walls. */

/* DATA PLOTTING */

void show_field
  ( mwplot_t *wp, 
    int nx, 
    int ny, 
    double dx, 
    double dy,
    double **A, 
    double vMag,
    char *zLabel,
    double time
  );
  /* generates on {wp} a 3D graph of the field {A}, assumed to be an
    array of samples with {nx} rows and {ny} columns, with step sizes
    {dx} and {dy}, respectively. The vertical scale is set assuming
    the values of {A} are in {[-vMag _ +vMag]}.
    
    The Z axis will be labeled with {zLabel}, and the plot will have a
    title showing the specified {time}, assumed to be in nanoseconds.
    
    If the plotter {wp} is configured to write to disk instead of the
    screen, the plot file will be called "{zLabel}-{NNNNNN}.{ext}",
    where {NNNNNN} is the {time} value rounded to integer, and {ext}
    is the appropriate extension ("eps", "png", etc.) */
  
void show_grid
  ( mwplot_t *wp, 
    int nx, 
    int ny, 
    double dx, 
    double dy,
    double **A, 
    double vMag,
    char *zLabel,
    double time
  );
  /* Generates on {wp} a 2D plot of the matrix {A}, assumed to be an
    array of samples with {nx} rows and {ny} columns, with step sizes
    {dx} and {dy}, respectively. The color scale is set assuming the
    values of {A} are in {[-vMag _ +vMag]}.
    
    The plot will be a grid of colore squares. It will have a title
    showing the specified {time}, assumed to be in nanoseconds.
    
    If the plotter {wp} is configured to write to disk instead of the
    screen, the plot file will be called "{zLabel}-{NNNNNN}.{ext}",
    where {NNNNNN} is the {time} value rounded to integer, and {ext}
    is the appropriate extension ("eps", "png", etc.) */
  
void get_grid_structure(int nx, int ny, double **F, double **A);
  /* Sets {A[0..nx][0..ny]} to a map showing the occupied elements
    of the sparse arrray {F[0..nx][0..ny]}.  Specifically.
    each {A[i][j]} is set to 0.0 if {F[i][j]} is NAN, and
    to 1 otherwise. The array {A} can then be displayed with
    {show_grid}. */

void get_field_range(int nx, int ny, double **F, double *vLo, double *vHi, double *vMag);
  /* Stores in {*vLo} and {*vHi}, respectively, the minimum and
    maximum values seen in {F[0..nx][0..ny]} Also stores in {*vHi}
    the maximum absolute value among all elements -- that is,
    {max{fabs(vHi),fabs(vLo)}}. */

/* UTILITY ROUTINES */

void fatal(char* s);

double** dvector(int nl, int nc);
  /* Allocates an array with {nl} rows and {nc} columns. */

void wait_for_user(void);
  /* Writes "Hit return to continue" and waits for user to do so. */

double get_double_from_user(char *prompt, double defval);
int get_int_from_user(char *prompt, int defval);
  /*  Writes a message "Enter {promt} [{defval}]:" to {stderr}; then
    waits for the user to type a number ({double} or {int},
    rspectively) + 'ENTER', and returns that number. Returns {defval}
    if the user just types 'ENTER'. */

void print_grid_statistics(int nx, int ny, double **A);
  /* Counts the number of valid (not NaN) entries in {A[0..nx][0..ny]},
    compares with total number of entries {(nx+1)*(ny+1)}. */

/* MAIN PROGRAM */

int main()
  { /* Fundamental constants */
    double cc = 2.99792458e8;          /* speed of light in free space */
    double muz = 4.0 * M_PI * 1.0e-7;  /* permeability of free space */
    double epsz =1.0 / (cc*cc*muz);    /* permittivity of free space */
    /* double aimp = sqrt(muz/epsz);  */      /* wave impedance in free space */

    int i, j, k;

    /* Starts the gnuplot process for 3D field plots: */
    mwplot_t *wpf = mwplot_start_gnuplot(640,480);
    mwplot_set_terminal(wpf, /*x11*/ 0);

    /* Starts the gnuplot process for 2D grid plots */
    mwplot_t *wpg = mwplot_start_gnuplot(640,480); /* 0,0 for debug output */
    mwplot_set_terminal(wpg, /*x11*/ 0);
    mwplot_set_gridmap_palette(wpg, /*yellow--red*/ 0);

    fprintf(stderr, "Max samples in coarsest level = %d\n", NPx);
    int n0 = get_int_from_user("the number of samples in the coarsest level", 8);
    
    int ell = get_int_from_user("the number of decompositions levels", 4); 
    
    int ell_atual = ell; /* actual number of decomp levels with significant wavelet coeffs */
    int level_max; /* number of decomposition levels */
    int interp_points; /* 2 or 4 point interpolation (linear or cubic) */

    double eps = get_double_from_user("the error/threshold for sparse representation", 1e-6);
    
    double eps1 = eps/10;  /* error/threshold for the sparse representation. */

    interp_points = get_int_from_user("the number of interpolation points", 4);

    int n = n0 * (1<<ell); /* Number of samples in the finest level. */

    double **ex=dvector(n+1, n+1);     /* X component of the elecric field. */
    double **ey=dvector(n+1, n+1);     /* Y component of the electric field. */
    double **hz=dvector(n+1, n+1);     /* Z component of the magnetic field. */
    
    double **dyex=dvector(n+1, n+1);   /* dex/dy - sparse form. */
    double **dxey=dvector(n+1, n+1);   /* dey/dx - sparse form. */
    double **dyhz=dvector(n+1, n+1);   /* dhz/dx - sparse form. */
    double **dxhz=dvector(n+1, n+1);   /* dhz/dy - sparse form. */
    
    double **az=dvector(n+1, n+1);     /* Copy of hz on full grid */
    double **sigma=dvector(n+1, n+1);  /* Conductivity of medium. */
    /*double **coef=dvector(n+1, n+1);  */         /* coeficiente */

    double dx = 1.0/n;              /* space increment in x-direction */
    double dy = 1.0/n;              /* space increment in y-direction */
    double dt = 0.5*dx/cc;          /* time step */
    int nmax =(int)(8.0e-9 / dt);   /* total number of time steps */

    /* Define the conductivity field {sigma} and the position and shape of initial pulse: */
    int geometry = get_int_from_user("the conductivity map (domain geometry) code", 0);
    setup_initial_state(geometry, n, n, dx, dy, sigma, hz, ex, ey);
    
    /* Show the {sigma} field: */
    show_grid(wpg, n, n, dx, dy, sigma, SIGMA_MAX, "SIGMA", 0.0);
    wait_for_user();

    /* Find the maximum {hz} value: */
    double hzLo, hzHi, hzMag;
    get_field_range(n, n, hz, &hzLo, &hzHi, &hzMag);
    
    /* Show the initial {hz} field, allow user to choose the style: */
    int pal = 0; /* User-selected palette code. */
    int sty = 0; /* User-selected plot style code. */
    while (1)
      { /* Set the selected palette and style: */
        fprintf(stderr, "using 3D graph palette %d\n", pal);
        mwplot_set_3dgraph_palette(wpf, pal);
        fprintf(stderr, "using 3D graph style %d\n", sty);
        /* Display the field: */
        mwplot_set_3dgraph_style(wpf, sty);
        show_field(wpf, n, n, dx, dy, hz, hzMag, "Hz", 0.0);
        /* Ask user for a new palette and style: */
        pal = get_int_from_user("a new palette code, or -1 to go on", -1);
        if (pal == -1) { break; }
        sty = get_int_from_user("a new style code", sty);
      }

    /* Convert the initial field {hz} to sparse representation: */
    spr_data_to_sparseH(hz, ell, n0, interp_points, eps1);
    spr_refineH(hz, ell, n0,  interp_points);

    /* Show the initial sparse grid for {hz}, allow user to choose style: */
    get_grid_structure(n, n, hz, az);
    while (1)
      { /* Set the selected palette and style: */
        fprintf(stderr, "using grid map palette %d\n", pal);
        mwplot_set_gridmap_palette(wpf, pal);
        fprintf(stderr, "using grid map style %d\n", sty);
        /* Display the field: */
        mwplot_set_gridmap_style(wpf, sty);
        show_grid(wpg, n, n, dx, dy, az, 1.0, "GRID", 0.0);
        /* Ask user for a new palette and style: */
        pal = get_int_from_user("a new palette code, or -1 to go on", -1);
        if (pal == -1) { break; }
        sty = get_int_from_user("a new style code", sty);
      }

     /* Clear out the field derivatives {dyex,dxey,dxhz,dyhz}: */
    for (i=0; i <= n; i++)
      for (j=0; j <= n; j++)
        { dyex[i][j] = dxey[i][j] = 0.0;
          dxhz[i][j] = dyhz[i][j] = 0.0;
          /*coef[i][j]=(1.0 -(dt*sigma[i][j])/2.0*epsz)/(1.0 +(dt*sigma[i][j])/2.0*epsz);*/
        }
 
    /* BEGIN TIME-STEPPING LOOP */
    for (k=1; k<= nmax; k++)
      {
        /* Calculo de Hz*/
        spr_data_to_sparseEx(ex, ell, n0, interp_points, eps);
        spr_data_to_sparseEy(ey, ell, n0, interp_points, eps);

        spr_refineEx(ex, ell, n0,  interp_points);
        spr_refineEy(ey, ell, n0,  interp_points);

        /* Makes sure that {ex,ey,hz} are all NAN or all valid: */
        for (i=0; i <=n; i++)
          for (j=0; j <=n; j++)
            { if ((isnan(ex[i][j])) && ((!isnan(ey[i][j]))||(!isnan(hz[i][j]))))
                ex[i][j]=interpEx(ex, ell, n0, interp_points, i, j); 

              if ((isnan(ey[i][j])) && ((!isnan(hz[i][j]))||(!isnan(ex[i][j]))))
                ey[i][j]=interpEy(ey, ell, n0, interp_points, i, j); 

              if ((isnan(hz[i][j])) && ((!isnan(ey[i][j]))||(!isnan(ex[i][j]))))
                hz[i][j]=interpH(hz, ell, n0, interp_points, i, j); 
            }

        /* Comparando o numero de pontos usados*/ 
        print_grid_statistics(n, n, hz);

        derivYEx(ex, ell, n0, interp_points, dx, dyex, hz);
        derivXEy(ey, ell, n0, interp_points, dx, dxey, hz);

        /*fazendo o update do hz*/
        for (i=0; i <=n; i++)
          for (j=0; j <=n; j++)
            { if (!isnan(hz[i][j]))
                { hz[i][j] = hz[i][j]+dt*(dyex[i][j]-dxey[i][j])/muz; }
            }
        spr_data_to_sparseH(hz, ell, n0, interp_points, eps1); 
        spr_refineH(hz, ell, n0,  interp_points);

        /* derivando o campo {hz} */
        derivYH(hz, ell, n0, interp_points, dx, dyhz, ex);
        derivXH(hz, ell, n0, interp_points, dx, dxhz, ey);

        /* fazendo o update do ex */
        for (i=0; i <=n; i++)
          for (j=0; j <=n; j++)
            { if (!isnan(ex[i][j]))
                { double sij = sigma[i][j];
                  double C = dt*sij/(2.0*epsz);
                  ex[i][j] = ((1-C) * ex[i][j] + (dt/epsz) * dyhz[i][j])/(1+C);
                }
            }

        /*fazendo o update do ey*/
        for (i=0; i <=n; i++)
          for (j=0; j <=n; j++)
            { if (!isnan(ey[i][j]))
                { double sij = sigma[i][j];
                  double C = dt*sij/(2.0*epsz);
                  ey[i][j] = ((1-C) * ey[i][j] - (dt/epsz) * dxhz[i][j])/(1+C);
                }
            }

        level_max = find_level_max(ey, ell, n0, ell_atual);
        fprintf(stderr, "level_max = %d\n", level_max);

        /* Copy {hz} to {az}, remove NaN, plot {az}: */
        for (i=0; i <= n; i++)
          for (j=0; j <= n; j++)
            az[i][j]= hz[i][j];
        spr_sparse_to_dataH(az, ell, n0, interp_points); 
        show_field(wpf, n, n, dx, dy, az, NAN, "Hz", k*dt*1e9);
        /* wait_for_user(); */ 

        /* Set {az} to the grid structure map of {hz}, display it: */
        get_grid_structure(n, n, hz, az);
        show_grid(wpg, n, n, dx, dy, az, 1.0, "GRID", k*dt*1e9);
        
        if (k == 1) { wait_for_user(); }

      }/* fim da evolucao no tempo */ 

    mwplot_stop_gnuplot(wpf);
    mwplot_stop_gnuplot(wpg);
    return 0;
  }

/* IMPLEMENTATIONS */
    
void setup_initial_state
  ( int geom, 
    int nx, 
    int ny, 
    double dx, 
    double dy, 
    double **sigma, 
    double **hz, 
    double **ex, 
    double **ey
  )
  {
    double xpulse, ypulse;   /* Position of initial pulse. */
    double wxpulse, wypulse; /* X-width and Y-width of initial pulse. */

    /* Clear all elements ({malloc} does not do it!): */
    int i, j;
    for (i = 0; i < nx; i++)
      for (j = 0; j < ny; j++)
        { sigma[i][j] = 0; }

    switch(geom)
      {
        case 1:
          fprintf(stderr, "geometry: closed conductive box\n");
          setup_geometry_box(nx, ny, sigma);
          xpulse = 0.500; wxpulse = 1.0/50.0;
          ypulse = 0.500; wypulse = 1.0/50.0;

          break;

        case 2:
          fprintf(stderr, "geometry: conductive horn\n");
          setup_geometry_horn(nx, ny, sigma);
          xpulse = 0.375; wxpulse = 1.0/50.0;
          ypulse = 0.500; wypulse = 1.0/50.0;
          break;

        default:
          fprintf(stderr, "invalid geometry code %d, assuming 0\n", geom);
          geom = 0;

        case 0:
          fprintf(stderr, "geometry: homogeneous non-conductive\n");
          xpulse = 0.500; wxpulse = sqrt(1.0/150.0);
          ypulse = 0.500; wypulse = sqrt(1.0/150.0);
          break;
      }

    /* initialize {hz} with a Gaussian pulse, except where {sigma} is nonzero. */
    for (i = 0; i <= nx; i++)
      for (j = 0; j <= ny; j++)
        { double vx = (i*dx - xpulse)/wxpulse;
          double vy = (j*dy - ypulse)/wypulse;
          hz[i][j]= (sigma[i][j] == 0 ? exp(-vx*vx)*exp(-vy*vy) : 0.0);
          ex[i][j] = ey[i][j] = 0.0;
        }
  }
  
void setup_geometry_box(int nx, int ny, double **sigma)
  { 

    /* The following code assumes a square grid, so: */
    assert(nx == ny);
    int n = nx;
    
    int nn4 = n/4;
    int i;
    
    /* Draw the box: */
    double sigma1 = SIGMA_MAX; /* Conductivity of box walls. */
    for (i = nn4; i <= n-nn4; i++)
      { sigma[nn4][i]= sigma1;
        sigma[n-nn4][i]= sigma1;
        sigma[i][nn4]=sigma1;
        sigma[i][n-nn4]=sigma1;
      }
    for (i= nn4+1; i <= n-nn4-1; i++)
      { sigma[nn4+1][i]= sigma1;
        sigma[n-nn4-1][i]= sigma1;
        sigma[i][nn4+1]=sigma1;
        sigma[i][n-nn4-1]=sigma1;
      }
    for (i = nn4+2; i <= n-nn4-2; i++)
      { sigma[nn4+2][i]= sigma1;
        sigma[n-nn4-2][i]= sigma1;
        sigma[i][nn4+2]=sigma1;
        sigma[i][n-nn4-2]=sigma1;
      }
  }

void setup_geometry_horn(int nx, int ny, double **sigma)
  { double sigma1 = SIGMA_MAX; /* Conductivity of horn walls */

    /* The following code assumes a square grid, so: */
    assert(nx == ny);
    int n = nx;
    
    /* The horn stops at x = 3/4. */
    int nn4 = n/4;
    int nn = 3*n/8;         
    int kn = n/16;   /* Num steps across width w*dx; k=n/4*w, w=4. */

    int i, ver, horn, cont;

    /* Draw the horn: */
    for (i = nn; i <= n-nn; i++)  /* Bottom of horn */
      { sigma[nn4][i]= sigma1;
        sigma[nn4-1][i]= sigma1;
        sigma[nn4+1][i]= sigma1;
      }
    for (i = nn4-1; i <= n/2; i++)  /* Horn's parallel sides. */
      { sigma[i][nn]= sigma1;    /* Bottom side (3 layers) */
        sigma[i][nn-1]= sigma1;
        sigma[i][nn+1]= sigma1;
        sigma[i][n-nn]= sigma1;  /* Top side (3 layers) */
        sigma[i][n-nn+1]= sigma1;
        sigma[i][n-nn-1]= sigma1;
      }
    horn=n/2; /* Start of horn's opening */
    ver=nn-1;
    for (cont = 1; cont <= kn; cont++)  /* Counting the steps. */
      { for (i = horn; i <= horn+4; i++)
          { sigma[i][ver]= sigma1;
            sigma[i][ver-1]= sigma1;
            sigma[i][ver+1]= sigma1;
            sigma[i][n-ver]= sigma1;
            sigma[i][n-ver+1]= sigma1;
            sigma[i][n-ver-1]= sigma1;
          }
        ver=ver-1;  /* Descending one step */
        horn=horn+4;            
      }           
  }

void show_field
  ( mwplot_t *wp, 
    int nx, 
    int ny, 
    double dx, 
    double dy,
    double **A, 
    double vMag,
    char *zLabel,
    double time
  )
  { mwplot_set_3dgraph_style(wp, /*surface with palette*/ 0);
    mwplot_set_axis_labels(wp, "X", "Y", zLabel);
    mwplot_set_ranges(wp, -0.5*dx, (nx+0.5)*dx, -0.5*dy, (ny+0.5)*dy, -vMag, vMag);
    /* Make up a {title} string: */
    char *title = NULL;
    asprintf(&title, "Time = %g ns", time);
    /* Make up a file name, just in case: */
    char *fileName = NULL;
    asprintf(&fileName, "%s-%06d", zLabel, (int)floor(1000*time + 0.5));
    mwplot_plot_3dgraph(wp, fileName, nx, ny, dx, dy, A, title);
    free(title);
    free(fileName);
  }

void show_grid
  ( mwplot_t *wp, 
    int nx, 
    int ny, 
    double dx, 
    double dy,
    double **A, 
    double vMag,
    char *zLabel,
    double time
  )
  { mwplot_set_gridmap_style(wp, /*default*/ 0);
    mwplot_set_axis_labels(wp, "X", "Y", NULL);
    mwplot_set_ranges(wp, 0, nx+1, 0, ny+1, -vMag, vMag);
    /* Make up a {title} string: */
    char *title = NULL;
    asprintf(&title, "Time = %g ns", time);
    /* Make up a file name, just in case: */
    char *fileName = NULL;
    asprintf(&fileName, "%s-%06d", zLabel, (int)floor(1000*time + 0.5));
    mwplot_plot_gridmap(wp, fileName, nx, ny, A, title);
    free(title);
    free(fileName);
  }

void get_field_range(int nx, int ny, double **F, double *vLo, double *vHi, double *vMag)
  {
    (*vLo) = +INFINITY;
    (*vHi) = -INFINITY;
    int i, j;
    for (i = 0; i <= nx; i++)
      for (j = 0; j <= ny; j++)
        { double Fij = F[i][j];
          if (! isnan(Fij))
            { if (Fij < (*vLo)) { (*vLo) = Fij; }
              if (Fij > (*vHi)) { (*vHi) = Fij; }
            }
        }
        
    double aLo = fabs(*vLo), aHi = fabs(*vHi);
    (*vMag) = (aLo > aHi ? aLo : aHi);
  }

void get_grid_structure(int nx, int ny, double **F, double **A)
  { /* Copy {hz} to {az}, changing NaNs to zeros: */
    int i, j;
    for (i = 0; i <= nx; i++)
      for (j = 0; j <= ny; j++)
        { A[i][j] = (isnan(F[i][j]) ? 0.0 : 1.0); }

    /* DEBUG -- add some {-1} marks to debug plot ranges: */

    auto void mark(int im, int jm);

    void mark(int im, int jm)
      { if (A[im][jm] == 0) { A[im][jm] = -1; } }

    for (i = 0; i < 5; i++)
      { mark(i,0); mark(i,i); mark(0,i);
        mark(nx-i,ny); mark(nx-i,ny-i); mark(nx,ny-i);
      }
  }

void print_grid_statistics(int nx, int ny, double **A)
  {
    int i, j;
    int nvalid = 0;
    for (i = 0; i <= nx; i++)
      for (j = 0; j <= ny; j++)
        { if (!isnan(A[i][j])) { nvalid = nvalid+1; } }
    fprintf(stderr,  "sparse grid: %7d points\n", nvalid);
    int ntotal=(nx+1)*(ny+1);
    fprintf(stderr,  "full grid:   %7d points\n", ntotal);
    fprintf(stderr,  "fill ratio:  %7.5f\n", ((double)nvalid)/((double)ntotal));
    fprintf(stderr, "\n");
  }

void wait_for_user(void)
  { char buf[10];
    fputs("\nHit return to continue", stderr);
    fgets(buf, sizeof(buf), stdin);
  }

#define BUFSZ 1000

double get_double_from_user(char *prompt, double defval)
  {
    char buf[BUFSZ];
    fprintf(stderr,"Enter %s", prompt);
    if (! isnan(defval)) { fprintf(stderr," [%g]", defval); }
    fprintf(stderr,": ");
    fgets(buf, BUFSZ, stdin);
    char *data = buf;
    while (((*data) == ' ') || ((*data) == '\011') || ((*data) == '\015')) { data++; }
    if (((*data) != '\n') && ((*data) != 0))
      { /* Parse input as integer: */
        char *rest = NULL;
        double dval = strtod(data, &rest); 
        /* Check number syntax: */
        while(((*rest) == ' ') || ((*rest) == '\011') || ((*rest) == '\015')) { rest++; }
        if ((*rest) != '\n')
          { fprintf(stderr, "** invalid character '%c' = '\\%03o'\n", (*rest), (*rest));
            exit(-1);
          }
        return dval;
      }
    else
      { /* Empty input: */
        return defval;
      }
  }

int get_int_from_user(char *prompt, int defval)
  {
    /* All 32-bit ints are exactly representable as {double}s, so: */
    double dval = get_double_from_user(prompt, (double)defval); 
    /* Check whether value is an exact integer: */
    int ival = (int)dval;
    if (dval != ((double)ival)) 
      { fprintf(stderr, "** value '%g' must be an integer\n", dval);
        exit(-1);
      }
    return ival;
  }

void fatal(char* s)
  { fprintf(stderr, "ERROR: %s\n", s);
    perror(s);
    exit(-1);
  }

double** dvector(int nl, int nc)
  { int i;
    double** dv = (double**) malloc(nc*sizeof(double*));
    if (dv == NULL)
      { fprintf(stderr, "ERROR: allocation failure in dvector(%d, %d)\n", nl, nc);
        exit(-1);
      }
    for( i=0; i<nc; i++)
      { dv[i] = (double*)malloc(nl*sizeof(double));
        if (dv[i] == NULL)
          { fprintf(stderr, "ERROR: allocation failure in dvector(%d, %d)\n", nl, nc);
            exit(-1);
          }
      }
    return dv;
  }

