# Generic Makefile for one or more executable programs
# Last edited on 2008-08-17 09:43:27 by stolfi

######################################################################
# To be included by the top-level Makefile in the program's directory.
# The caller must define ${PROG} or ${PROGS}, and possibly also:
#   
#   ${IGNORE}         list of ".h" or ".c" files that should not be compiled or installed.
#   ${JS_LIBS}        list of Stolfi's libraries to bind with (WITHOUT directories) 
#   ${USE_GL}         define as `YES' to bind with the GL headers and libraries
#   ${USE_X11}        define as `YES' to bind with the X11 headers and libraries
#   ${OTHER_LIBS}     list of additional libraries (WITH directories)
#   ${OTHER_C_FLAGS}  additional flags to compile C source files
#   ${OTHER_LD_FLAGS} additional flags for linking 
#   
# All library names must include the extension (".a", ".so", etc.)
# The client must define the "all" target.
# The client may extend the "clean" target with double-colon rules.

ifeq "/${PROGS}" "/"
PROGS := ${PROG}
endif

ifneq "/${PROGS}" "/"

SHELL := /bin/sh

# ----------------------------------------------------------------------
# Package components and directories

# Get list of all local ".h" files, and their derived objects:
HFILES := ${filter-out ${IGNORE}, ${wildcard *.h}}
HOFILES := ${subst .h,.ho,${HFILES}}

# Get list of all local ".c" files (including the main sources of all ${PROGS}):
CFILES := ${filter-out ${IGNORE}, ${wildcard *.c}}
OFILES := ${subst .c,.o,${CFILES}}

# Get list of ".c" files of good main programs and their objects:
PROG_CFILES := ${addsuffix .c,${PROGS}}
PROG_OFILES := ${subst .c,.o,${PROG_CFILES}}

# Get list of non-main ".c" files and objects:
LIB_CFILES := ${filter-out ${PROG_CFILES} ${IGNORE}, ${CFILES}}
LIB_OFILES := ${subst .c,.o,${LIB_CFILES}}

# Directory with test runs:
TEST_DIR := tests

# ----------------------------------------------------------------------
# Installation directories

# Where to install the program:
INST_PUB_DIR := ${HOME}/Projeto/Programas
INST_INC_DIR := ${INST_PUB_DIR}/include
INST_LIB_DIR := ${INST_PUB_DIR}/lib
#INST_BIN_DIR := ${INST_PUB_DIR}/bin

#@if [ "${PLATAFORMA}" -eq "i486" ]; then \
#	INST_BIN_DIR = ${INST_PUB_DIR}/bin32 \
#elif [ "${PLATAFORMA}" -eq "x86_64" ]; then \
#	INST_BIN_DIR = ${INST_PUB_DIR}/bin64 \
#fi

ifeq (${PLATAFORMA},i486)
	INST_BIN_DIR := ${INST_PUB_DIR}/bin32
endif

ifeq (${PLATAFORMA},x86_64)
	INST_BIN_DIR := ${INST_PUB_DIR}/bin64
endif

# ----------------------------------------------------------------------
# Locations of imported packages

# Location of Stolfi's headers and libraries:
JS_PUB_DIR := ${HOME}/Projeto/Programas
JS_LIB_DIR := ${JS_PUB_DIR}/lib
JS_INC_DIR := ${JS_PUB_DIR}/include

# Location of the GL headers and libraries
GL_INC_DIR := /usr/include
GL_LIB_DIR := /usr/lib

# Location of X11 headers and libraries (including ICE):
X11_INC_DIR := /usr/include/X11
X11_LIB_DIR := /usr/lib

# ----------------------------------------------------------------------
# Optional libraries and include directories

# Get OpenGL libraries and headers if requested:
ifeq "/${USE_GL}" "/YES" 
  GL_LIBRARIES := \
    ${JS_LIB_DIR}/libglut.so \
    ${GL_LIB_DIR}/libGLU.so \
    ${GL_LIB_DIR}/libGL.so
  GL_I_FLAGS := 
else
  GL_LIBRARIES :=
  GL_I_FLAGS := 
endif

# Get X11 libraries and headers if requested:
ifeq "/${USE_X11}" "/YES" 
  X11_LIBRARIES := \
    ${X11_LIB_DIR}/libX11.so \
    ${X11_LIB_DIR}/libXext.so \
    ${X11_LIB_DIR}/libXmu.so \
    ${X11_LIB_DIR}/libXt.so \
    ${X11_LIB_DIR}/libXi.so \
    ${X11_LIB_DIR}/libSM.so \
    ${X11_LIB_DIR}/libICE.so
  X11_I_FLAGS := -I${X11_INC_DIR}
else
  X11_LIBRARIES := 
  X11_I_FLAGS := 
endif

#---------------------------------------------------------------------
# Phony "make" targets:

.PHONY:: \
  default-action \
  all actions \
  depend \
  uninstall uninstall-progs \
  build build-progs do-build-progs check \
  install install-progs \
  clean clean-progs clean-tests \
  tar-export tar-save

#---------------------------------------------------------------------
# Default target: 

#default-action: uninstall build install

all: uninstall build install

#---------------------------------------------------------------------
# "make actions" performs the actions listed in ${ACTIONS}, if any

actions:
ifeq "/${ACTIONS}" "/"
	@echo '$${ACTIONS} is empty'
else
	${MAKE} ${ACTIONS}
endif

#---------------------------------------------------------------------
# Directories to search for #include files:

I_FLAGS := \
  -I. \
  -I${JS_INC_DIR} \
  ${GL_I_FLAGS} \
  ${X11_I_FLAGS} \
  ${OTHER_I_FLAGS}

#---------------------------------------------------------------------
# "make depend" recreates the source dependency file ${DEPFILE}.

DEPFILE := Deps.make

depend: ${DEPFILE}

${DEPFILE}: extract-ho-deps ${CFILES} ${HFILES}
	/bin/rm -f ${DEPFILE}
	./extract-ho-deps ${I_FLAGS} -I/usr/include \
            ${HFILES} ${CFILES} \
          | egrep -v ': /usr/include' \
          > ${DEPFILE}

# ----------------------------------------------------------------------
# Create a local symlink for {extract-ho-deps}, for {tar-export}'s sake

#extract-ho-deps: ${HOME}/Projeto/Programas/bin/extract-ho-deps
#	ln -s ${HOME}/Projeto/Programas/bin/extract-ho-deps

extract-ho-deps: ${INST_BIN_DIR}/extract-ho-deps
	ln -s ${INST_BIN_DIR}/extract-ho-deps

#---------------------------------------------------------------------
# "make build" assumes that the dependencies in ${DEPFILE}
# are up-to-date.

build: build-progs
  
build-progs: ${DEPFILE}
	@echo "-- make do-build-progs ------------------------------"
	${MAKE} do-build-progs

do-build-progs: ${PROGS}

LIBRARIES := \
  ${addprefix ${JS_LIB_DIR}/, ${JS_LIBS}} \
  ${GL_LIBRARIES} \
  ${X11_LIBRARIES} \
  ${OTHER_LIBS}

CC := /usr/bin/gcc

C_FLAGS := \
  -Wall -Wundef \
  -Wpointer-arith \
  -Wmissing-prototypes \
  -ggdb \
  -ffloat-store \
  -frounding-math \
  -fpcc-struct-return
  
  
LD_FLAGS := \
  -ggdb

%.o: %.c
	${CC} -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} $*.c -pg
   
%.ho: %.h
	${CC} -o $*.ho -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} -x c $*.h \
          || /bin/rm -f $*.ho -pg
  
${PROGS}: ${PROG_OFILES} ${LIB_OFILES} ${LIBRARIES}
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'
	@echo building $@ ...
	-rm -f $@
	${CC} -o $@ ${LD_FLAGS} ${OTHER_LD_FLAGS} $@.o \
          ${LIB_OFILES} \
          ${LIBRARIES} -lm -lrt -pg
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'

ifneq "/${wildcard ${DEPFILE}}" "/"
  # Include specific dependencies extracted by "make depend"
  include ${DEPFILE}
endif

#---------------------------------------------------------------------
# "make install" copies program and makefile to the public dir.

install: install-progs

install-progs: ${PROGS}
	@echo "-- make install-progs ------------------------------"
	cp -p ${PROGS} ${INST_BIN_DIR}/

#---------------------------------------------------------------------
# "make uninstall" deletes the exported program and makefile:

uninstall: uninstall-progs

uninstall-progs:
	@echo "-- make uninstall-progs ------------------------------"
	( cd ${INST_BIN_DIR}/. && rm -f ${PROGS} )
 

#---------------------------------------------------------------------
# "make tar-export" creates a tarball of sources, and executables.

THIS_DIR := ${notdir ${shell pwd}}
NOW := ${shell date '+%Y-%m-%d-%H%M%S'}
TARBALL_FILE := ${THIS_DIR}-${NOW}.tgz
  
tar-export: ${PROGS} extract-ho-deps
	@echo "archiving to ${TARBALL_FILE} ..."
	tar -cvzf ${TARBALL_FILE} \
          00-README Makefile \
            extract-ho-deps \
            ${PROGS} \
            ${HFILES} ${CFILES}

#---------------------------------------------------------------------
# "make tar-save" creates a date-stamped tarball of all files except
# edit backups and derived files.

SAVEDIR := SAVE
SAVETARFILE := ${LIBNAME}-${NOW}-save.tgz
  
tar-save: clean
	tar -cvzf ${SAVEDIR}/${SAVETARFILE} \
          `find ./ -type f -print | egrep -v -e '[~]$'`

#---------------------------------------------------------------------
# "make check" runs the tests in the "tests" subdirectory.

TEST_DIR := tests

check: ${TEST_DIR}
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR}/. && ${MAKE} all ) fi

#---------------------------------------------------------------------
# "make clean" deletes all derived files.
# It is a double-colon rule so that clients may add more actions.
# Also recreates the ".deps" files.

clean:: clean-progs clean-tests

clean-progs:
	-/bin/rm -f *.o *.ho *.a core ${PROGS} *.deps ${DEPFILE}
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && ${MAKE} clean ) fi
	touch --date='Jan 01 1970' ${DEPFILE}
	${MAKE} depend

clean-tests:
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && make clean ) fi

endif
# End of ${PROG} or ${PROGS} section.
######################################################################

