/* filefmt.h -- version-checked file headers, footers, and comments. */
/* Last edited on 2007-02-03 07:04:50 by stolfi */

#ifndef filefmt_H
#define filefmt_H

#include <stdio.h>
#include <affirm.h>

/* FILE HEADERS AND FOOTERS */

/* These procedures help maintaining structured files with typed 
  and versioned headers and footers, like this:
  
     begin myimage (format of 1996-11-29)
     [...stuff...]
     end myimage
     
  The strings "myimage" and "1996-11-29" are client-specified,
  and checked on input.  These begin/end pairs may be nested, etc. */

void filefmt_write_header(FILE *wr, char *type, char *version);
void filefmt_write_footer(FILE *wr, char *type);
  /* These procedures write a header or footer line to {wr}, 
    complete with the final newline. */

void filefmt_read_header(FILE *rd, char *type, char *version);
void filefmt_read_footer(FILE *rd, char *type);
  /* These procedures parse a header or footer line from the given file.
    They skip any formatting characters (SPACE, TAB, NUL, CR, LF, and page breaks),
    before the first token, and any number of spaces (SPACE, TAB, NUL)
    after each token; but require the entire header or footer to be
    contained in one line, and terminated by a newline (which is consumed too).
    They abort the program on any error. */

char *filefmt_make_header(char *type, char *version);
char *filefmt_make_footer(char *type);
  /* These procedures construct a header or footer line as a string, 
    including the terminating newline. */
      
/* COMMENTS: */

/* These routines write and parse a comment text, consisting of zero or
  more lines marked by a given {prefix} character:
  
    | Blah blah blah
    |   blah blah
    | and more blah.
  
  The {prefix} must be the first character of the line, and is
  normally followed by a blank. */

void filefmt_write_comment(FILE *wr, char *cmt, char prefix);
  /* Writes the given {cmt} text to {wr}, with a {prefix} character
    and a blank in front of every line.  If {cmt} is {NULL}
    or an empty string, writes nothing. Supplies a final '\n' if 
    the text is non-empty but does not end with newline. */

char *filefmt_read_comment(FILE *rd, char prefix);
  /* Reads zero or more lines from {rd} that begin with the {prefix}
    character; strips the leading {prefix} and the following blank (if
    present) from each line; and returns all those lines as a single
    newly allocated string, where each line is terminated by a
    newline. */

#endif
