/* Coordinate systems for images. */
/* Last edited on 2008-12-11 17:20:21 by stolfi */

#ifndef image_coords_H
#define image_coords_H

#include <bool.h>
#include <r2.h>
#include <r3x3.h>
#include <hr2.h>
#include <argparser.h>

hr2_pmap_t imgc_coord_sys_map
  ( bool_t xRev, 
    bool_t yRev, 
    bool_t center, 
    r2_t *org, 
    int cols, 
    int rows
  );
  /* Returns a projective map {M} that converts point coordinates from
    the {float_image.h} native coordinate system (FS) to the user's
    coordinate system (US).
    
    The FS has the X axis pointing to the right, the Y axis pointing
    up, and the origin at the bottom left corner of the image's
    domain.
    
    The US has the X axis pointing to the right if {xRev} is FALSE,
    and to the left if {xRev} is TRUE; and the Y axis pointing up if {yRev}
    is FALSE, and down if {yRev} is true. The US origin is located at
    the center of the image's domain if {center} is TRUE, or at the
    position {org} if {center} is FALSE.  In the second case, each
    coordinate of {org} is measured from the edge of the image's
    domain that has lowest coordinate (that is, the left edge if
    {xRev} is FALSE, the right edge if {xRev} is true; the bottom edge
    if {yRev} is FALSE, the top edge if {yRev} is TRUE).
    
    In any case, all coordinates are measured in pixels. The image's
    domain is assumed to be a rectangle of width {cols} and height
    {rows}. This is equivalent to assuming that a pixel with indices
    {[ix,iy]} is a square, displaced {ix} units to the right and {iy}
    units up from the domain's lower left corner. */

/* COMMAND LINE PARSING */

/* General help about image coordinate systems */

#define imgc_axes_INFO \
  "Points in image domains" \
  " are expressed in terms of a coordinate system" \
  " specific to each image, input or output.  In both systems," \
  " the directions of the axes are determined by" \
  " the \"-xAxis\" and \"-yAxis\" command line arguments.  These" \
  " arguments also affect the default origin of the coordinate" \
  " system.  In the default coordinate systems, the origin lies" \
  " at the low corner of the domain (which depends on the axis directions)."
  
#define imgc_pixel_centers_INFO \
  "The unit of length is the pixel size; in the default coordinate systems," \
  " pixel *corners* have integer coordinates, and pixel *centers* have half-integer coordinates."
  
/* Horizontal axis direction */

void imgc_parse_x_axis(argparser_t *pp, bool_t *xleft);
  /*  Parses the "-xAxis" command line option, according to
    {imgc_parse_x_axis_HELP}. Sets {*xleft} to TRUE if "-xAxis left" is
    present, to FALSE if "-xAxis right" is present. If the option is
    not present, leaves {*xleft} unchanged. */

#define imgc_parse_x_axis_HELP \
  "[ -xAxis { right | left } ]"

#define imgc_parse_x_axis_HELP_INFO \
  "  -xAxis { right | left }\n" \
  "    This optional argument specifies the direction of the" \
  " horizontal axis of image coordinate systems.  It also defines the" \
  " default horizontal position of the origin: at the left edge" \
  " of the image if \"right\", or at the right edge" \
  " if \"left\"."
  
#define imgc_parse_x_axis_pbm_default_INFO \
  "The default is \"-xAxis right\" as in most image processing tools."
  
#define imgc_parse_x_axis_math_default_INFO \
  "The default is \"-xAxis right\" as in mathematical tradition."
  
/* Vertical axis direction */

void imgc_parse_y_axis(argparser_t *pp, bool_t *yDown);
  /*  Parses the "-yAxis" command line option, according to
    {imgc_parse_y_axis_HELP}. Sets {*ydown} to TRUE if the "-yAxis
    down" is present, to FALSE if "-yAxis up" is present. If the
    option is not present, leaves {*ydown} unchanged. */

#define imgc_parse_y_axis_HELP_INFO \
  "  -yAxis { down | up }\n" \
  "    This optional argument specifies the direction of" \
  " the vertical axis of image coordinate systems.  It also defines the" \
  " default vertical position of the origin: at the top edge of the" \
  " image if \"down\", or at the bottom edge if \"up\"."
  
#define imgc_parse_y_axis_pbm_default_INFO \
  "The default is \"-yAxis down\" as in most image processing tools." \

#define imgc_parse_y_axis_math_default_INFO \
  "The default is \"-yAxis up\", in the mathematical tradition." \

#define imgc_parse_y_axis_HELP \
  "[ -yAxis { down | up } ]"
  
/* Input image origin */

void imgc_parse_input_center_org(argparser_t *pp, bool_t *iCenter, r2_t *iOrg);
  /* Parses the mutually exclusive arguments "-iCenter" and "-iOrg",
    that define the coordinate origin for input images,
    according to the syntax specs {imgc_parse_input_center_org_HELP}. If
    "-iCenter"is preent, sets {*iCenter} to TRUE and leaves {*iOrg}
    unchanged. If "-iOrg" is present, sets {*iCenter} to FALSE, and sets
    {*iOrg} to the given point. If neither is present, leaves {*iCenter}
    and {*iOrg} unchanged. */

#define imgc_parse_input_center_org_HELP \
  "[ -iCenter | -iOrg {CX_IN} {CY_IN} ]" \

#define imgc_input_origin_INFO \
  "The default coordinate system origin for input images" \
  " can be overriden by the \"-iOrg\" or \"-iCenter\" argument."  

#define imgc_parse_input_center_org_HELP_INFO \
  "  -iCenter\n" \
  "  -iOrg {CX_IN} {CY_IN}\n" \
  "    These mutually exclusive optional arguments specify" \
  " the origin of the coordinate system for input" \
  " images.  The \"-iCenter\" option sets the origin" \
  " at the center of the image domain.  The \"-iOrg\" option sets" \
  " the origin at the point {(CX_IN,CY_IN)} relative to the" \
  " default origin.  If this argument is not specified, the program" \
  " assumes \"-iOrg 0 0\" (which means the default origin)."

/* Output image origin */

void imgc_parse_output_center_org(argparser_t *pp, bool_t *oCenter, r2_t *oOrg);
  /* Analogous to {imgc_parse_center_org}, but applying to output
    images, according to {imgc_parse_output_center_org_HELP_INFO}. 
    Looks for the keywords "-oCenter" and "-oOrg". */

#define imgc_parse_output_center_org_HELP \
  "[ -oCenter | -oOrg {CX_OUT} {CY_OUT} ]" \

#define imgc_output_origin_INFO \
  "The default coordinate system origin for output images" \
  " can be overriden by the \"-oOrg\" or \"-oCenter\" argument."

#define imgc_parse_output_center_org_HELP_INFO \
  "  -oCenter\n" \
  "  -oOrg {CX_OUT} {CY_OUT}\n" \
  "    These mutually exclusive optional arguments specify" \
  " the origin of the coordinate system for output" \
  " images.  The \"-oCenter\" option sets the origin" \
  " at the center of the image domain.  The \"-oOrg\" option sets" \
  " the origin at the point {(CX_OUT,CY_OUT)} relative to the" \
  " default origin.  If this argument is not specified, the program" \
  " assumes \"-oOrg 0 0\" (which means the default origin).\n"

/* Output image size: */

void imgc_parse_output_size(argparser_t *pp, int *oCols, int *oRows, int max_size);
  /* Parses the output image size option "-oSize", according to {imgc_parse_output_size_HELP}. If
    "-oSize" is present, sets {*oCols} and {*oRows} to the next two
    arguments. Otherwise leaves {*oCols} and {*oRows} unchanged.
    Fails with error if either dimension exceeds {max_size}. */
    
#define imgc_parse_output_size_HELP \
  "[ -oSize {NX} {NY} ]"

#define imgc_parse_output_size_HELP_INFO \
  "  -oSize {NX} {NY}\n" \
  "    Specifies the horizontal and vertical size of the output image."

#endif
