/* See fget.h */
/* Last edited on 2008-07-24 17:49:31 by stolfi */

#include <affirm.h>
#include <fget.h>
#include <jsstring.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

/* INTERNAL PROTOTYPES */

void fget_skip_to_non_blank(FILE *f);
  /* Skips spaces and requires that the next character is not 
   a line or page break, or end-of-file. */

int fget_digit(FILE *f, bool_t alpha);
  /* Tries to read the next character from {f}, expecting it to be a
    digit in '0'..'9', or, if {alpha} is true, also a letter in
    'a'..'z' or 'A'..'Z'. If it succeeds, returns the numeric value of
    that character as an integer in 0..9 (if digit) or 10..35 (if
    letter). If the next character is anything else (including EOF),
    puts it back and returns -1. */

/* IMPLEMENTATIONS */

void fget_skip_formatting_chars(FILE *f)
  { int c;
    do 
      { c = fgetc(f);
        if (c == EOF) { return; }
      }
    while ((c=='\000')||(c==' ')||((c>='\011')&&(c<='\015')));
    ungetc(c, f); 
    return;
  }

void fget_skip_spaces(FILE *f)
  { int c;
    do 
      { c = fgetc(f);
        if (c == EOF) { return; }
      }
    while ((c=='\000')||(c==' ')||(c=='\011'));
    ungetc(c, f); return;
  }

void fget_match(FILE *f, char *t)
  { while ((*t) != '\000')
      { int c = fgetc(f);
        if (c != (unsigned char)(*t))
          { fprintf(stderr, "next char = '%c'\n", c);
            demand(FALSE, txtcat("cannot find \"", txtcat(t, "\"")));
          }
        t++;
      }
  }

void fget_eol(FILE *f)
  { int c;
    fget_skip_spaces(f);
    c = fgetc(f);
    demand(c == '\n', "extraneous data on input line");
  }

void fget_skip_to_non_blank(FILE *f)
  { int c;
    fget_skip_spaces(f);
    c = fgetc(f);
    demand((c != EOF)&&((c<'\012')||(c>'\015')), "item not found");
    ungetc(c, f);
  }

char fget_char(FILE *f)
  { int c;
    fget_skip_to_non_blank(f); 
    c = fgetc(f);
    demand(c != EOF, "expecting nonblank char, found end of file");
    return c;
  }

bool_t fget_bool(FILE *f)
  { int c;
    fget_skip_to_non_blank(f); 
    c = fgetc(f);
    if ((c=='t')||(c=='T')||(c=='1' )) 
      { return TRUE; }
    else if ((c=='f')||(c=='F')||(c=='0' ))
      { return FALSE; }
    else
      { ungetc(c, f);
        demand(FALSE, "missing or invalid bool_t value");
        return FALSE;
      }
  }

#define INITEXTLENGTH 1024

char *fget_string(FILE *f)
  { char buf[INITEXTLENGTH+1];
    char *pbuf = &(buf[0]);
    int bufsz = INITEXTLENGTH+1;
    int i = 0, c;
    fget_skip_to_non_blank(f);
    c = fgetc(f);
    while ((c != EOF)&&(c!='\000')&&(c!=' ')&&(c!='\011')&&((c<'\012')||(c>'\015')))
      { if (i >= bufsz-1)
          { int tmpsz = 2*bufsz;
            char *tmp = (char *)notnull(malloc(tmpsz), "out of mem for buf");
            pbuf[i] = '\000';
            strcpy(tmp, pbuf);
            if (pbuf != &(buf[0])) { free(pbuf); }
            pbuf = tmp; bufsz = tmpsz;
          }
        pbuf[i] = c; i++;
        c = fgetc(f);
      }
    if (c != EOF) { ungetc(c, f); }
    pbuf[i] = '\000'; i++; 
    if ((i < bufsz)||(pbuf==&(buf[0])))
      { char *tmp = (char *)notnull(malloc(i), "out of mem for result");
        strcpy(tmp, pbuf);
        if (pbuf != &(buf[0])) { free(pbuf); }
        pbuf = tmp;
      }
    return pbuf;
  }

int fget_digit(FILE *f, bool_t alpha)
  { int c = fgetc(f);
    if ((c >= '0')&&(c <= '9'))
      { return c - '0'; }
    else if (alpha)
      { if ((c >= 'a')&&(c <= 'z'))
          { return 10 + (c - 'a'); }
        else if ((c >= 'A')&&(c <= 'Z'))
          { return 10 + (c - 'A'); }
      }
    /* No digit found: */
    if (c != EOF) { ungetc(c, f); } 
    return -1;
  }

int fget_int(FILE *f)
  { int64_t x = fget_int64(f);
    demand((x >= INT_MIN)&&(x < INT_MAX), "integer does not fit in {int} type");
    return (int)x;
  }

int fget_int32(FILE *f)
  { int64_t x = fget_int64(f);
    demand((x >= INT32_MIN)&&(x < INT32_MAX), "integer does not fit in {int32_t} type");
    return (int)x;
  }

int64_t fget_int64(FILE *f)
  { int c;
    bool_t positive = TRUE;
    fget_skip_to_non_blank(f);
    c = fgetc(f);
    if (c=='+') 
      { c = fgetc(f); }
    else if (c=='-') 
      { positive = FALSE; c = fgetc(f); }
    if (c != EOF) { ungetc(c, f); }
    uint64_t x = fget_uint64(f, 10);
    if (positive) 
      { demand (x <= ((uint64_t)INT64_MAX), "integer does not fit in {int64_t} type");
        return (int64_t)x;
      }
    else
      { demand (x <= ((uint64_t)INT64_MAX) + 1, "integer does not fit in {int64_t} type");
        return (int64_t)(0 - x);
      }
  }

unsigned int fget_uint(FILE *f, int base)
  { uint64_t x = fget_uint64(f, base);
    demand(x < UINT_MAX, "integer does not fit in {unsigned int} type");
    return (unsigned int)x;
  }

unsigned int fget_uint32(FILE *f, int base)
  { uint64_t x = fget_uint64(f, base);
    demand(x < UINT32_MAX, "integer does not fit in {uint32_t} type");
    return (unsigned int)x;
  }

uint64_t fget_uint64(FILE *f, int base)
  { demand((base >= 2)&&(base <= 36), "invalid base");
    uint64_t x = 0;
    fget_skip_to_non_blank(f);
    int d = fget_digit(f, (base > 10));
    demand(d >= 0, "number not found"); 
    do
      { demand(d < base, "invalid digit in number"); 
        demand (x <= (UINT64_MAX - d)/base, "number does not fit in 64 bits");
        x = base*x + d;
        d = fget_digit(f, (base > 10));
      }
    while (d >= 0);
    return x;
  }

#define MAXNUMLENGTH 1024
  /* Maximum length of a "double" item. */ 

double fget_double(FILE *f)
  { char buf[MAXNUMLENGTH+1];
    int i, c;
    double x;
    char *rest;
    fget_skip_to_non_blank(f);
    i = 0;
    c = fgetc(f);
    if ((c=='+')||(c=='-'))
      { buf[i] = c; i++; c = fgetc(f); }
    while ((i < MAXNUMLENGTH)&&(c != EOF)&&(c >= '0')&&(c <= '9'))
      { buf[i] = c; i++; c = fgetc(f); }
    if ((i < MAXNUMLENGTH)&&(c=='.'))
      { buf[i] = c; i++; c = fgetc(f); }
    while ((i < MAXNUMLENGTH)&&(c >= '0')&&(c <= '9'))
      { buf[i] = c; i++; c = fgetc(f); }
    if ((i < MAXNUMLENGTH)&&((c=='e')||(c=='E')||(c=='d')||(c=='D')))
      { buf[i] = c; i++; c = fgetc(f);
        if ((i < MAXNUMLENGTH)&&((c=='+')||(c=='-')))
          { buf[i] = c; i++; c = fgetc(f); }
        while ((i < MAXNUMLENGTH)&&(c >= '0')&&(c <= '9'))
          { buf[i] = c; i++; c = fgetc(f); }
      }
    if (c != EOF) { ungetc(c, f); }
    buf[i] = '\000'; 
    x = strtod(&(buf[0]), &rest);
    demand((*rest) == '\000', txtcat("invalid number", &(buf[0])));
    return x;
  }

bool_t fget_test_char(FILE *f, char c)
  { int r;
    fget_skip_spaces(f); 
    r = fgetc(f);
    if (r == (unsigned char)c)
      { return TRUE; }
    else if (r == EOF)
      { return FALSE; }
    else
      { ungetc(r, f); return FALSE; }
  }

void fget_skip_and_match(FILE *f, char *t)
  { fget_skip_spaces(f); 
    fget_match(f, t); 
  }

bool_t fget_skip_and_test_char(FILE *f, char c)
  { fget_skip_spaces(f); 
    return fget_test_char(f, c); 
  }

/* Created by J. Stolfi, Unicamp, Dec/2002. */
