/* See float_image.h */
/* Last edited on 2009-06-03 11:36:05 by stolfi */ 

#define _GNU_SOURCE
#include <limits.h>
#include <float.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <sample_conv.h>
#include <frgb.h>
#include <frgb_ops.h>
#include <float_image_color.h>
#include <indexing.h>
#include <filefmt.h>
#include <nget.h>
#include <fget.h>
#include <affirm.h>
#include <bool.h>

#include <float_image.h>

/* IMPLEMENTATIONS */

float float_image_get_sample(float_image_t *A, int c, int x, int y)
  { assert((c >= 0) && (c < A->sz[0]));
    assert((x >= 0) && (x < A->sz[1]));
    assert((y >= 0) && (y < A->sz[2]));
    ix_index_t ix[3] = {c, x, y};
    ix_pos_t p = ix_position(3, ix, A->bp, A->st);
    return A->sample[p];
  }

float *float_image_get_sample_address(float_image_t *A, int c, int x, int y)
  { assert((c >= 0) && (c < A->sz[0]));
    assert((x >= 0) && (x < A->sz[1]));
    assert((y >= 0) && (y < A->sz[2]));
    ix_index_t ix[3] = {c, x, y};
    ix_pos_t p = ix_position(3, ix, A->bp, A->st);
    return &(A->sample[p]);
  }

void float_image_set_sample(float_image_t *A, int c, int x, int y, float v)
  { assert((c >= 0) && (c < A->sz[0]));
    assert((x >= 0) && (x < A->sz[1]));
    assert((y >= 0) && (y < A->sz[2]));
    ix_index_t ix[3] = {c, x, y};
    ix_pos_t p = ix_position(3, ix, A->bp, A->st);
    A->sample[p] = v;
  }

void float_image_get_pixel(float_image_t *A, int x, int y, float v[])
  { float *sp = float_image_get_sample_address(A, 0, x, y);
    int NC = A->sz[0];
    int c;
    for (c = 0; c < NC; c++) { v[c] = (*sp); sp += A->st[0]; }
  }

void float_image_set_pixel(float_image_t *A, int x, int y, float v[])
  { float *sp = float_image_get_sample_address(A, 0, x, y);
    int NC = A->sz[0];
    int c;
    for (c = 0; c < NC; c++) { (*sp) = v[c]; sp += A->st[0]; }
  }

void float_image_fill_pixel(float_image_t *A, int x, int y, float v)
  { float *sp = float_image_get_sample_address(A, 0, x, y);
    int NC = A->sz[0];
    int c;
    for (c = 0; c < NC; c++) { (*sp) = v; sp += A->st[0]; }
  }

void float_image_get_sample_row(float_image_t *A, int c, int y, float v[])
  { float *sp = float_image_get_sample_address(A, c, 0, y);
    int NX = A->sz[1];
    int x;
    for (x = 0; x < NX; x++) { v[x] = (*sp); sp += A->st[1]; }
  }

void float_image_set_sample_row(float_image_t *A, int c, int y, float v[])
  { float *sp = float_image_get_sample_address(A, c, 0, y);
    int NX = A->sz[1];
    int x;
    for (x = 0; x < NX; x++) { (*sp) = v[x]; sp += A->st[1]; }
  }

void float_image_fill_sample_row(float_image_t *A, int c, int y, float v)
  { float *sp = float_image_get_sample_address(A, c, 0, y);
    int NX = A->sz[1];
    int x;
    for (x = 0; x < NX; x++) { (*sp) = v; sp += A->st[1]; }
  }


void float_image_get_pixel_row(float_image_t *A, int y, float v[])
  { float *pxp = float_image_get_sample_address(A, 0, 0, y);
    int NC = A->sz[0];
    int NX = A->sz[1];
    int c, x; int k = 0;
    for (x = 0; x < NX; x++)
      { float *sp = pxp;
        for (c = 0; c < NC; c++) 
          { v[k] = (*sp); sp += A->st[0]; k++; }
        pxp += A->st[1];
      }
  }

void float_image_set_pixel_row(float_image_t *A, int y, float v[])
  { float *pxp = float_image_get_sample_address(A, 0, 0, y);
    int NC = A->sz[0];
    int NX = A->sz[1];
    int c, x; int k = 0;
    for (x = 0; x < NX; x++)
      { float *sp = pxp;
        for (c = 0; c < NC; c++) 
          { (*sp) = v[k]; sp += A->st[0]; k++; }
        pxp += A->st[1];
      }
  }

void float_image_fill_pixel_row(float_image_t *A, int y, float v)
  { float *pxp = float_image_get_sample_address(A, 0, 0, y);
    int NC = A->sz[0];
    int NX = A->sz[1];
    int c, x;
    for (x = 0; x < NX; x++)
      { float *sp = pxp;
        for (c = 0; c < NC; c++) 
          { (*sp) = v; sp += A->st[0]; }
        pxp += A->st[1];
      }
  }

void float_image_fill_channel(float_image_t *A, int c, float v)
  { int NC = A->sz[0];
    int NX = A->sz[1]; 
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float_image_set_sample(A, c, x, y, v); }
      }
  }

void float_image_set_channel(float_image_t *A, int cA, float_image_t *V, int cV)
  { int NCV = V->sz[0];
    demand((cV >= 0) && (cV < NCV), "invalid {V} channel");
    int NCA = A->sz[0];
    demand((cA >= 0) && (cA < NCA), "invalid {A} channel");
    int NX = V->sz[1]; 
    int NY = V->sz[2];
    float_image_check_size(A, -1, NX, NY);
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float v = float_image_get_sample(V, cV, x, y);
            float_image_set_sample(A, cA, x, y, v);
          }
      }
  }

void float_image_mix_channels
  ( double sA,
    float_image_t *A,
    int cA, 
    double sB,
    float_image_t *B,
    int cB, 
    float_image_t *R,
    int cR
  )
  { int NCR = R->sz[0];
    demand((cR >= 0) && (cR < NCR), "invalid {R} channel");
    int NCA = A->sz[0];
    demand((cA >= 0) && (cA < NCA), "invalid {A} channel");
    int NCB = B->sz[0];
    demand((cB >= 0) && (cB < NCB), "invalid {B} channel");
    int NX = R->sz[1]; 
    int NY = R->sz[2];
    float_image_check_size(A, -1, NX, NY);
    float_image_check_size(B, -1, NX, NY);
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float vA = float_image_get_sample(A, cA, x, y);
            float vB = float_image_get_sample(B, cB, x, y);
            float_image_set_sample(R, cR, x, y, sA*vA + sB*vB);
          }
      }
  }

void float_image_fill(float_image_t *A, float v)
  { int NC = A->sz[0];
    int c;
    for (c = 0; c < NC; c++) { float_image_fill_channel(A, c, v); }
  }

void float_image_fill_pixels(float_image_t *A, float v[])
  { int NC = A->sz[0];
    int c;
    for (c = 0; c < NC; c++) { float_image_fill_channel(A, c, v[c]); }
  }

void float_image_assign(float_image_t *A, float_image_t *V)
  { int NC = A->sz[0];
    demand(V->sz[0] == NC, "incompatible channel counts");
    int c;
    for (c = 0; c < NC; c++) { float_image_set_channel(A, c, V, c); }
  }

void float_image_get_gradient_sobel
  ( float_image_t *A, 
    int c, 
    int x, 
    int y,
    double *fxP, 
    double *fyP
  )
  {
    int NX = A->sz[1]; 
    int NY = A->sz[2];
    /* Compute the horizontal and vertical derivatives in channel {c}: */
    double sum_wy_fx = 0.0;
    double sum_wy = 0.0;
    double sum_wx_fy = 0.0;
    double sum_wx = 0.0;
    int d;
    for (d = -1; d <= +1; d++)
      { /* Sobel weight for slice {d}: */
        double wd = (d == 0 ? 2.0 : 1.0);
        if ((x-1 >= 0) && (x+1 < NX))
          { /* Accumulate the horizontal derivative: */
            int yd = y + d;
            double wy = ((yd < 0) || (yd >= NY) ? 0.0 : wd);
            if (wy > 0.0)
              { double fxp = float_image_get_sample(A, c, x+1, yd);
                double fxm = float_image_get_sample(A, c, x-1, yd);
                double fx = (fxp - fxm)/2;
                sum_wy_fx += wy*fx;
                sum_wy += wy;
              }
          }

        if ((y-1 >= 0) && (y+1 < NY))
          { /* Accumulate the vertical derivative: */
            int xd = x + d;
            double wx = ((xd < 0) || (xd >= NX) ? 0.0 : wd);
            if (wx > 0.0)
              { double fyp = float_image_get_sample(A, c, xd, y+1);
                double fym = float_image_get_sample(A, c, xd, y-1);
                double fy = (fyp - fym)/2;
                sum_wx_fy += wx*fy;
                sum_wx += wx;
              }
          }
      }

    /* Compute the derivatives for pixel {x,y}: */
    (*fxP) = (sum_wy == 0 ? 0.0 : sum_wy_fx/sum_wy);
    (*fyP) = (sum_wx == 0 ? 0.0 : sum_wx_fy/sum_wx);
  }

void float_image_get_local_avg_var
  ( float_image_t *A, 
    int c, 
    int x, 
    int y,
    int hw,
    double wt[],
    double *avgP, 
    double *varP
  )
  {
    int NX = A->sz[1]; 
    int NY = A->sz[2];
    double sum_w = 0.0;
    double sum_w_f = 0.0;
    double sum_w_f2 = 0.0;
    int dx, dy;
    for (dx = -hw; dx <= +hw; dx++)
      { int xs = x + dx;
        double wx = ((xs < 0) || (xs >= NX) ? 0 : wt[dx + hw]); 
        for (dy = -hw; dy <= +hw; dy++)
          { int ys = y + dy;
            double wy = ((ys < 0) || (ys >= NY) ? 0 : wt[dy + hw]); 
            double w = wx*wy;
            if (w > 0)
              { double f = float_image_get_sample(A, c, xs, ys);
                sum_w += w;
                sum_w_f += w*f;
                sum_w_f2 += w*f*f;
              }
          }
      }
    /* Compute the local variance: */
    double avg = (sum_w == 0 ? 0.0 : sum_w_f/sum_w);
    double var = (sum_w == 0 ? 0.0 : sum_w_f2/sum_w - avg*avg);
    (*avgP) = avg;
    (*varP) = var;
  }

void float_image_local_avg_var
  ( float_image_t *A, 
    int cA, 
    int hw,
    double wt[],
    float_image_t *M, 
    int cM,
    float_image_t *V, 
    int cV
  )
  {
    int NCA = A->sz[0];
    demand((cA >= 0) && (cA < NCA), "invalid {A} channel");
    int NX = A->sz[1]; 
    int NY = A->sz[2];
    if (M != NULL)
      { int NCM = M->sz[0];
        demand((cM >= 0) && (cM < NCM), "invalid {M} channel");
        float_image_check_size(M, -1, NX, NY);
      }
    if (V != NULL)
      { int NCV = V->sz[0];
        demand((cV >= 0) && (cV < NCV), "invalid {V} channel");
        float_image_check_size(V, -1, NX, NY);
      }
    
    int x, y;
    for (x = 0; x < NX; x++)
      { for (y = 0; y < NY; y++)
          { /* Compute the local gradient squared {g2} in channel {c}: */
            /* Compute the horizontal and vertical derivatives in channel {c}: */
            double avg, var;
            float_image_get_local_avg_var(A, cA, x, y, hw, wt, &avg, &var);
            if (M != NULL) { float_image_set_sample(M, cM, x, y, avg); }
            if (V != NULL) { float_image_set_sample(V, cV, x, y, var); }
          }
      }
  }
  
double float_image_get_dilated
  ( float_image_t *A, 
    int c, 
    int x, 
    int y,
    int hw,
    double wt[]
  )
  {
    int NX = A->sz[1]; 
    int NY = A->sz[2];
    double max_w_a = -INF;
    int dx, dy;
    for (dx = -hw; dx <= +hw; dx++)
      { int xs = x + dx;
        double wx = ((xs < 0) || (xs >= NX) ? 0 : wt[dx + hw]); 
        for (dy = -hw; dy <= +hw; dy++)
          { int ys = y + dy;
            double wy = ((ys < 0) || (ys >= NY) ? 0 : wt[dy + hw]); 
            double w = wx*wy;
            if (w > 0)
              { double a = float_image_get_sample(A, c, xs, ys);
                double w_a = w*a;
                if (w_a > max_w_a) { max_w_a = w_a; }
              }
          }
      }
    return max_w_a;
  }

void float_image_make_grayscale(float_image_t *A)
  {
    int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    if (NC == 1) { return; }
    demand(NC == 3, "channel count must be 1 or 3");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { frgb_t pix = fic_get_frgb_pixel(A, 0, 1, 2, x, y);
            double value = frgb_Y(&pix);
            int c;
  	    for (c = 0; c < A->sz[0]; c++) { pix.c[c] = value; }
            fic_set_frgb_pixel(A, 0, 1, 2, x, y, &pix);
          }
      }
  }

void float_image_apply_gamma(float_image_t *A, int c, double gamma, double bias)
  { if (gamma == 1) { return; }
    demand(gamma > 0, "gamma must be positive"); 
    int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float *pA = float_image_get_sample_address(A, c, x, y);
            (*pA) = sample_conv_gamma((*pA), gamma, bias);
          }
      }
  }

void float_image_log_scale(float_image_t *A, int c, double vref, double base)
  { demand((isfinite(vref)) && (vref > 0), "invalid vref");
    demand((isfinite(base)) && (base != 1) && (base > 0), "invalid base");
    double logBase = (base == M_E ? 1.0 : log(base));
    int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float *pA = float_image_get_sample_address(A, c, x, y);
            (*pA) = sample_conv_log((*pA), vref, logBase);
          }
      }
  }

void float_image_rescale_samples(float_image_t *A, int c, float a0, float a1, float z0, float z1)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    double scale = (z1 - z0)/(a1 - a0);
    demand(! isnan(scale), "scale factor is NaN");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float *v = float_image_get_sample_address(A, c, x, y);
            (*v) = z0 + scale*((*v) - a0);
          }
      }
  }

void float_image_square_samples(float_image_t *A, int c)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { float *smp = float_image_get_sample_address(A, c, x, y);
            float v = *smp;
            if (!isnan(v)) { (*smp) = v*v; }
          }
      }
  }

double float_image_compute_total_energy(float_image_t *A, int c, double avg)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    if ((c < 0) || (c >= NC))
      { /* Invalid channel, all samples are zero (or there are no samples). */
        return 0.0;
      }

    /* Compute the sum of squares {sum2}: */
    double sum2 = 0;
    int x, y;
    for (y = 0; y < NY; y++)
      { for (x = 0; x < NX; x++)
          { double v = float_image_get_sample(A, c, x, y);
            if (isfinite(v))
              { /* Neither {INF} nor {NAN}: */ v -= avg; sum2 += v*v; }
          }
      }
    return sum2;
  }

void float_image_compute_sample_avg_dev(float_image_t *A, int c, double *avg, double *dev)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    if ((c < 0) || (c >= NC))
      { /* Invalid channel, all samples are zero (or there are no samples). */
        (*avg) = 0.0;
        (*dev) = 0.0;
      }
    else
      { /* Valid channel. */
        /* Compute the sample average {sa}: */
        double sum = 0;
        int x, y;
        int tot = 0;
         for (y = 0; y < NY; y++)
          { for (x = 0; x < NX; x++)
              { double v = float_image_get_sample(A, c, x, y);
                if (isfinite(v))
                  { /* Neither {INF} nor {NAN}: */ sum += v; tot++; }
              }
          }
        double sa = (tot <= 0 ? 0.0 : sum/tot);
        /* Compute the sample variance {sv}: */
        sum = 0;
        for (y = 0; y < NY; y++)
          { for (x = 0; x < NX; x++)
              { double v = float_image_get_sample(A, c, x, y);
                if (isfinite(v))
                  { /* Neither {INF} nor {NAN}: */ v -= sa; sum += v*v; }
              }
          }
        double sv = (tot <= 1 ? 0.0 : sum/(tot - 1));
        /* Return the results: */
        (*avg) = sa; (*dev) = sqrt(sv);
      }
  }

void float_image_update_sample_range(float_image_t *A, int c, float *vMin, float *vMax)
  { 
    if ((vMin == NULL) && (vMax == NULL)) { /* Nothing to do: */ return; }
    int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    if ((c < 0) || (c >= NC))
      { /* Invalid channel, all samples are zero. (But there may be no samples!) */
        if ((NX > 0) && (NY > 0))
          { float v = 0.0;
            if (vMin != NULL) { if (v < (*vMin)) { (*vMin) = v; } }
            if (vMax != NULL) { if (v > (*vMax)) { (*vMax) = v; } }
          }
      }
    else
      { /* Valid channel. */
        int x, y;
        for (y = 0; y < NY; y++)
          { for (x = 0; x < NX; x++)
              { float v = float_image_get_sample(A, c, x, y);
                if (isfinite(v))
                  { /* Neither {INF} nor {NAN}: */ 
                    if (vMin != NULL) { if (v < (*vMin)) { (*vMin) = v; } }
                    if (vMax != NULL) { if (v > (*vMax)) { (*vMax) = v; } }
                  }
              }
          }
      }
  }

void float_image_flip_x(float_image_t *A, int c, int ix)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    
    /* Reduce the index {ix} to the range {0..NX-1}: */
    ix = (ix % NX); if (ix < 0) { ix += NX; }
    
    /* Compute the first and last indices for the X loop: */
    int fst_x = ix / 2 + 1; 
    int lst_x = (ix + NX - 1) / 2;
    int x1, y;
    for (y = 0; y < NY; y++)
      { for (x1 = fst_x; x1 <= lst_x; x1++)
          { /* Compute the partner: */
            int x2 = ix - x1; if (x2 < 0) { x2 += NX; }
            assert(x2 != x1);
            assert(x2 < NX);
            float *smp1 = float_image_get_sample_address(A, c, x1, y);
            float *smp2 = float_image_get_sample_address(A, c, x2, y);
            float tmp = (*smp1); (*smp1) = (*smp2); (*smp2) = tmp;
          }
      }
  }
  
void float_image_flip_y(float_image_t *A, int c, int iy)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    
    /* Reduce the index {iy} to the range {0..NY-1}: */
    iy = (iy % NY); if (iy < 0) { iy += NY; }
    
    /* Compute the first and last indices for the Y loop: */
    int fst_y = iy / 2 + 1; 
    int lst_y = (iy + NY - 1) / 2;
    int y1, x;
    for (x = 0; x < NX; x++)
      { for (y1 = fst_y; y1 <= lst_y; y1++)
          { /* Compute the partner: */
            int y2 = iy - y1; if (y2 < 0) { y2 += NY; }
            assert(y2 < NY);
            assert(y2 != y1);
            float *smp1 = float_image_get_sample_address(A, c, x, y1);
            float *smp2 = float_image_get_sample_address(A, c, x, y2);
            float tmp = (*smp1); (*smp1) = (*smp2); (*smp2) = tmp;
          }
      }
  }

void float_image_shift(float_image_t *A, int c, int dx, int dy)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    demand((c >= 0) && (c < NC), "invalid channel index");
    
    /* Reduce the shifts to the ranges {0..NX-1}, {0..NY-1}: */
    dx = (dx % NX); if (dx < 0) { dx += NX; }
    dy = (dy % NY); if (dy < 0) { dy += NY; }
    
    /* Uses the two-flip method to obtain a shift. This may be faster
      than decomposition, depending on cache size. */
    
    if (dx != 0)
      { /* Apply two flips in X: */
        float_image_flip_x(A, c, 0);
        float_image_flip_x(A, c, dx);
      }
      
    if (dy != 0)
      { /* Apply two flips in Y: */
        float_image_flip_y(A, c, 0);
        float_image_flip_y(A, c, dy);
      }
  }

#define float_image_max_samples (1024u*1024u*1024u)
  /* Maximum total elements (2^30), for sanity checks. */

float_image_t *float_image_new (int NC, int NX, int NY)
  { /* Sanity checks: */
    demand((NC >= 0) && (NC < float_image_max_size), "too many channels");
    demand((NX >= 0) && (NX < float_image_max_size), "too many columns");
    demand((NY >= 0) && (NY < float_image_max_size), "too many rows");
    ix_pos_count_t NS = ((ix_pos_count_t)NC)*NX*NY;
    demand(NS < float_image_max_samples, "too many samples");
    /* Allocate header: */
    float_image_t *A = (float_image_t *)notnull(malloc(sizeof(float_image_t)), "no mem");
    A->sz[0] = NC; A->st[0] = (NC < 2 ? 0 : 1);
    A->sz[1] = NX; A->st[1] = (NX < 2 ? 0 : NC);
    A->sz[2] = NY; A->st[2] = (NY < 2 ? 0 : NC*NX);
    A->bp = 0;
    if ((NC == 0) || (NX == 0) || (NY == 0))
      { A->sample = (float *)NULL; }
    else
      { A->sample = (float *)notnull(malloc(NS*sizeof(float)), "no mem"); }
    (void)ix_parms_are_valid(3, A->sz, A->bp, A->st, /*die:*/ TRUE);
    return A;
  }
  
float_image_t *float_image_copy (float_image_t *A)
  { int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    float_image_t *B = float_image_new(NC, NX, NY);
    int c, x, y;
    for(y = 0; y < NY; y++)
      { for(x = 0; x < NX; x++)
          { for (c = 0; c < NC; c++) 
              { float v = float_image_get_sample(A, c, x, y);
                float_image_set_sample(B, c, x, y, v);
              }
          }
      }
    return B;
  }

float_image_t *float_image_crop
  ( float_image_t *A, 
    int cLo,
    int cHi,
    int xLo,
    int xHi,
    int yLo,
    int yHi,
    float bg
  )
  { /* Grab the dimensions of {A}: */
    int NCA, NXA, NYA;
    float_image_get_size(A, &NCA, &NXA, &NYA);
    /* Compute the dimensions of the result {R} and allocate it: */
    int NCR = fmax(0, cHi - cLo);
    int NXR = fmax(0, xHi - xLo);
    int NYR = fmax(0, yHi - yLo);
    float_image_t *R = float_image_new(NCR, NXR, NYR);
    /* Fill {R} as appropriate: */
    int c, x, y;
    for (c = 0; c < NCR; c++)
      { int cA = cLo + c;
        bool_t cOK = ((cA >= 0) && (cA < NCA));
        for (x = 0; x < NXR; x++)
          { int xA = xLo + x;
            bool_t xOK = ((xA >= 0) && (xA < NXA));
            for (y = 0; y < NYR; y++)
              { int yA = yLo + y;
                bool_t yOK = ((yA >= 0) && (yA < NYA));
                float v;
                if (cOK && xOK && yOK)
                  { v = float_image_get_sample(A, cA, xA, yA); }
                else
                  { v = bg; }
                float_image_set_sample(R, c, x, y, v);
              }
          }
      }
    return R;
  }

void float_image_free(float_image_t *A)
  { if (A == NULL) return;
    if (A->sample != NULL) { free(A->sample); }
    free(A);
  }
  
void float_image_get_size(float_image_t *A, int *NC, int *NX, int *NY)
  { (*NC) = A->sz[0]; (*NX) = A->sz[1]; (*NY) = A->sz[2]; }
  
void float_image_check_size(float_image_t *A, int NC, int NX, int NY)
  { if (NC >= 0) { demand(A->sz[0] == NC, "wrong number of channels"); }
    if (NX >= 0) { demand(A->sz[1] == NX, "wrong number of columns"); }
    if (NY >= 0) { demand(A->sz[2] == NY, "wrong number of rows"); }
  }

#define float_image_file_version "2006-03-25"

void float_image_write(FILE *wr, float_image_t *A)
  { 
    int NC = A->sz[0];
    int NX = A->sz[1];
    int NY = A->sz[2];
    filefmt_write_header(wr, "float_image_t", float_image_file_version);
    fprintf(wr, "NC = %d\n", NC);
    fprintf(wr, "NX = %d\n", NX);
    fprintf(wr, "NY = %d\n", NY);
    int c, x, y;
    for(y = 0; y < NY; y++)
      { if (y > 0) { fprintf(wr, "\n"); }
        for(x = 0; x < NX; x++)
          { fprintf(wr, "%5d %5d", x, y);
            for (c = 0; c < NC; c++) 
              { float v = float_image_get_sample(A, c, x, y);
                fprintf(wr, " %+14.7e", v);
              }
            fprintf(wr, "\n");
          }
      }
    filefmt_write_footer(wr, "float_image_t");
    fflush(wr);
  }

float_image_t *float_image_read(FILE *rd)
  {
    filefmt_read_header(rd, "float_image_t", float_image_file_version);
    int NC = nget_int(rd, "NC"); fget_eol(rd);
    int NX = nget_int(rd, "NX"); fget_eol(rd);
    int NY = nget_int(rd, "NY"); fget_eol(rd);
    float_image_t *A = float_image_new(NC, NX, NY);
    int c, x, y;
    for (y = 0; y < NY; y++)
      { if (y > 0) { fget_eol(rd); }
        for (x = 0; x < NX; x++)
          { int xr = fget_int(rd);
            int yr = fget_int(rd);
            demand((xr == x) && (yr == y), "bad pixel indices");
            /* Read channels of pixel {(x,y)}: */
            for (c = 0; c < NC; c++)
              { double v = fget_double(rd);
                demand(fabs(v) <= DBL_MAX, "bad sample value");
                float_image_set_sample(A, c, x, y, v);
              }
            fget_eol(rd);
          }
      }
    filefmt_read_footer(rd, "float_image_t");
    return A;
  }
