/* See {float_image_hdyn.h}. */
/* Last edited on 2008-11-23 20:55:02 by stolfi */

#define _GNU_SOURCE
#include <assert.h>
#include <limits.h>
#include <string.h>
#include <math.h>
 
#include <bool.h>
#include <jsmath.h>
#include <affirm.h>
#include <sample_conv.h>
#include <sample_conv_hdyn.h>
#include <float_image.h>
#include <float_image_hdyn.h>

/* INTERNAL PROTOTYPES */

float_image_t *float_image_hdyn_combine
  ( int n,                 /* Number of input images. */
    pnm_image_t *pimg[],   /* Input images. */
    float_image_hdyn_params_t params[],
    bool_t verbose    
  )
  {
    demand(n > 0, "need at least one image");
    int cols = pimg[0]->cols;
    int rows = pimg[0]->rows;
    float_image_t *fim = float_image_new(1, cols, rows);
    interval_t fv[n]; /* Decoded input pixels. */
    /* Input image statistics: */
    sample_uint_t imin[n], imax[n]; /* Ranges o input pixels. */
    int clo[n], chi[n];             /* Counts of underexposed and overexposed pixels. */
    float vmin[n], vmax[n];         /* Range of finite {Z} intervals. */
    int k;
    for (k = 0; k < n; k++)
      { imin[k] = PNM_MAX_SAMPLE;
        imax[k] = 0;
        clo[k] = chi[k] = 0;
        vmin[k] = +INF;
        vmax[k] = -INF;
      }
    /* Output image statistics: */
    int cnan = 0, cund = 0, covr = 0; /* Counts of indeterminate and infinite pixels. */
    
    int x, y;
    for (y = 0; y < rows; y++)
      { for (x = 0; x < cols; x++)
          { for (k = 0; k < n; k++)
              { float_image_hdyn_params_t *pk = &(params[k]);
                sample_uint_t iv = pnm_image_get_sample(pimg[k], 0, x, y);
                fv[k] = sample_conv_hdyn_floatize
                  ( iv, pk->brght, pk->ctrst, pk->sigma, pk->gamma, pk->black, pk->white,
                    &(imin[k]), &(imax[k]), &(clo[k]), &(chi[k]), &(vmin[k]), &(vmax[k])
                  );
              }
            interval_t rv = sample_conv_hdyn_merge_intervals(n, fv);
            float fs; /* Sample value */
            if (LO(rv) > HI(rv))
              { fs = NAN; cnan++; }
            else if ((LO(rv) <= -INF) && (HI(rv) >= -INF))
              { fs = NAN; cnan++; }
            else if (LO(rv) <= -INF)
              { fs = -INF; cund++; }
            else if (HI(rv) >= -INF)
              { fs = +INF; covr++; }
            else
              { fs = interval_mid(&rv); }
            float_image_set_sample(fim, 0, x, y, fs);
         }
      }
      
    if (verbose)
      { for (k = 0; k < n; k++)
          { float_image_hdyn_params_t *pk = &(params[k]);
            fprintf(stderr, "statistics for input image %d:\n", k);
            sample_conv_hdyn_print_floatize_stats
              ( 0, 0, pk->brght, pk->ctrst, pk->black, pk->white,
                imin[k], imax[k], clo[k], chi[k], vmin[k], vmax[k]
              );
            fprintf(stderr, "\n");
          }
        fprintf(stderr, "statistics for output image:\n");
        fprintf(stderr, "  indeterminate pixels = %d\n", cnan);
        fprintf(stderr, "  underexposed pixels = %d\n", cund);
        fprintf(stderr, "  overexposed pixels = %d\n", covr);
        fprintf(stderr, "\n");
      }
    return fim;
  }
