/* See {float_image_mismatch.h}. */
/* Last edited on 2008-05-25 03:23:37 by stolfi */

#include <float_image_mismatch.h>

#include <assert.h>
#include <limits.h>
#include <math.h>
 
#include <affirm.h>

/* INTERNAL PROTOTYPES */

void fimm_est_avg_var(int n, double v[], double q[], double *Mp, double *Qp);
  /* Estimates the mean {*Mp} and variance {*Qp} of the samples {v[0..n-1]},
    taking into account the noise variances {q[0..n-1]}.
    
    More precisely, assumes that the true value {t[i]} of sample
    number {i} is an unknown random variable, independently drawn from
    some distribution with unknown mean {M} and unknown variance {Q}.
    Assumes also also that the given value {v[i]} is actually
    {t[i]+e[i]}, where {e[i]} is another unknown random variable,
    independently drawn from a Gaussian distribution with mean 0 and
    variance {q[i]}. The procedure returns the best estimate for {M}
    and {Q}, using Bayesian statistics.  
    
    Returns {*Mp == NAN} and {*Qp == NAN} if all the {q[i]} are zero. */

/* IMPLEMENTATIONS */

double fimm_mismatch_var
  ( int n,             /* Number of images being compared. */
    fimm_eval_t *eval, /* Image evaluator. */
    r2_t rd,           /* Half-width of the comparison window. */
    i2_t np            /* Number of sampling points along each axis. */
  )  
  { 
    /* Corresponding sample values from images {0..ni-1}. */
    double v[n];   /* {v[i]} is the value of image {i} at the current point {p}. */
    double q[n];   /* {q[i]} is the noise variance of image {i} at the current point {p}. */
    double w;      /* {w} is the weight of the point {p}. */

    /* Sums over all pairs {(c,p)}: */
    double sum_wQ = 0; /* Sum of mass-weighted variances {Q(c,p)*M(c,p)}. */
    double sum_w = 0;  /* Sum of masses {M(c,p)}. */

    /* Enumerate sampling points: */
    int nx = np.c[0], ny = np.c[1];
    int rx, ry;
    r2_t p; /* Sampling point. */
    for (rx = 0; rx < nx; rx++)
      { p.c[0] = rd.c[0]*((2.0*rx + 1.0)/nx - 1.0);
        for (ry = 0; ry <= ny; ry++)
          { p.c[1] = rd.c[1]*((2.0*ry + 1.0)/ny - 1.0);
            /* Obtain values and weights of all channels of all images at {p}: */
            eval(&p, n, v, q, &w);
            double M, Q;
            fimm_est_avg_var(n, v, q, &M, &Q);
            sum_wQ += w*Q;
            sum_w += w;
          }
      }
    return (sum_w == 0 ? 0.0 : sum_wQ / sum_w);
  }

void fimm_est_avg_var(int n, double v[], double q[], double *Mp, double *Qp)
  {
    /* !!! Possibly incorrect/inefficient implementation !!! */
    /* Find min variance {qmin} among all {q[i]}, check for bad values: */
    double qmin = +INF;
    int i;
    for (i = 0; i < n; i++) 
      { double qi = q[i];
        demand((! isnan(qi)) && (qi >= 0), "invalid variance {q[i]}");
        if (qi < qmin) { qi = qmin; }
      }
    /* Now compute the avg {M} and the var {Q}, iteratively: */
    double Q = 0;   /* No guess about the variance {Q}. */
    double M = 0;   /* Average - to be computed. */
    int niter = 3;  /* For no good reason... */
    int k;
    for (k = 0; k < niter; k++)
      { /* Compute average {M} of all {v[i]}, with weight {1/(Q+q[i])}: */
        /* If any {Q+q[i]} is zero, only those points count, with weight 1. */
        double sum_wv = 0;
        double sum_w = 0;
        int i;
        for (i = 0; i < n; i++)
          { double qi = q[i] + Q; 
            double wi = ((qmin == 0.0) && (Q == 0.0) ? (qi == 0.0) : 1.0/qi); 
            sum_wv += wi*v[i];
            sum_w += wi;
          }
        M = (sum_w == 0 ? NAN : sum_wv/sum_w); 
        if (isnan(M)) { Q = NAN; break; }
        /* Recompute the variance {Q} given {M}, with same weights: */
        double sum_wd2 = 0;
        for (i = 0; i < n; i++)
          { double qi = q[i] + Q; 
            double wi = ((qmin == 0.0) && (Q == 0.0) ? (qi == 0.0) : 1.0/qi); 
            double di = v[i] - M; 
            sum_wd2 += wi*di*di;
          }
        Q = sum_wd2/sum_w;
        if (Q == 0.0) { break; }
      }
    (*Mp) = M;
    (*Qp) = Q;
  }
    
            
    
