#ifndef float_image_mismatch_H
#define float_image_mismatch_H

/* Tools for local or global comparison of two or more images. */
/* Last edited on 2007-09-06 09:46:18 by stolfi */ 

#include <r2.h>
#include <i2.h>

/* 
  MISMATCH FUNCTION
  
  A /mismatch function/ is a function that takes {n} images, and
  returns a single real number that measures the overall difference
  between them.
  
  The result of a mismatch function is always non-negative. It is zero
  when the images are identical, but may be zero also on other
  situations.
  
  A mismatch function is sensitive only to image /differences/, in the
  sense that it does not change when the same constant {d} gets added
  to all samples of all images. On the other hand, it depends
  quadratically on those differences, so that multiplying all image
  samples by a constant {r} multiplies the result by {r^2}.
  
  A mismatch function is zero when {n = 1}, and is undefined when {n = 0}.
  
  !!! What about the triangle inequality?!!!
  
  IMAGE MODEL

  The mismatch functions in this interface assume that each image is a
  real-valued function of two real variables. Thus, for each image
  index {i} in {0..n-1} and each point {p} of the plane, there is a
  real sample value {v(i,p)}.
  
  SAMPLE ERROR VARIANCE
  
  In many applications, the sample values {v(i,p)} are only an
  approximate, being contaminated by sampling and measurement errors.
  
  To accomodate those situations, the mismatch functions below also
  require the nominal /noise variance/ {q(i,p)} for the sampling and
  quantization errors inherent in each sample value {v(i,p)}. They
  assume that the true value of image {i} at point {p} is {v(i,p) +
  e(i,p)}, where the values {e(i,p)} are unknown variables,
  independently drawn from Gaussian distributions with zero mean and
  variance {q(i,p)}.
  
  SAMPLING POINT WEIGHT
  
  The client of the mismatch functions below is asked to specify also
  a /weight/ {w(p)} for any given point {p}. These weights are used when
  computing averages over several points {p}.

  SAMPLING WINDOW

  The mismatch functions below operate on a /sampling window/ {W}
  which is a origin-centered rectangle {[-rx _ +rx]  [-ry _ +ry]}
  of the plane.  The half-extents {rx} and {ry} are the coordinates
  of the {rd} vector parameter. 
  
  SAMPLING POINTS
  
  The mismatch is computed from values of the given images at a
  certain set of /sampling points/ within the sampling window {W}.
  They are a regular axis-aligned rectangular grid that uniformly
  covers {W}. The grid has {nx} columns and {ny} rows, where {nx} and
  {ny} are the coordinates of the {np} vector parameter.
  
  More precisely, imagine the window {W} partitioned into an array of
  {nx} by {ny} identical rectangular cells. The sampling points are
  the *centers* of those cells.
  
  SAMPLING FUNCTION
  
  The images to be compared are defined implicitly by a
  client-specified /sampling function/ {eval} of type {fimm_eval_t}.  */

typedef void fimm_eval_t(r2_t *p, int n, double v[], double q[], double *wp); 
  /* Type of a procedure that evaluates images {0..n-1} at the given
    point {p}. It should return the samples {v(i,p)} in {v[0..n-1]},
    the corresponding noise variances {q(i,p)} in {q[0..n-1]}, and the
    weight {w(p)} of the sampling point {p} in {*wp}. */

double fimm_mismatch_var
  ( int n,             /* Number of images being compared. */
    fimm_eval_t *eval, /* Image evaluator. */
    r2_t rd,           /* Half-width of the comparison window. */
    i2_t np            /* Number of sampling points along each axis. */
  );
  /* Returns a mismatch measure for images {0..n-1}, based on the
    mean squared differences of corresponding sample values
    This is essentially the weighted mean-square-difference mismatch,
    generalized to {n} images.
    
    For each sampling point {p}, the procedure obtains {n} sample
    values {v(i,p)}, for {i} in {0..n-1}, the corresponding noise
    variances {q(i,p)}, and the sampling point weight {w(p). The
    procedure then estimates the variance {Q(p)} of those {n} samples
    relative to their mean, taking into account the noise variances
    {q(i,p)}. The result of the mismatch function is the average of
    {Q(p)} over all sampling points {p}, each term weighted by {w(p)}.
    
    This mismatch measure is appropriate when all images have been
    captured using the same instrument with the same measurement
    scale. It is not adequate, for example, when each image is a
    photograph that has been independently adjusted for brightness and
    constrast. */

#endif
