/* See {float_image_transform.h}. */
/* Last edited on 2008-05-25 03:23:43 by stolfi */

#include <assert.h>
#include <limits.h>
#include <string.h>
#include <math.h>
 
#include <bool.h>
#include <r2.h>
#include <r2x2.h>
#include <r3.h>
#include <r3x3.h>
#include <affirm.h>
#include <float_image.h>
#include <float_image_transform.h>

/* INTERNAL PROTOTYPES */

void fitr_get_transformed_pixel
  ( float_image_t *img, 
    int col, 
    int row, 
    fitr_transform_t *map,
    float undef, 
    bool_t avg, 
    float *f,
    bool_t *debugp
  );
  /* Computes pixel on column{col} and row {row} of the transformed
    image. The result is an average (if {avg} is true) or partition
    integral (if {avg} is false) of image values in the neighborhood
    of the the pixel's center. Uses {map} to find the corresponding
    points in the input image. Returns the result in {f[k]}, for {k}
    in {{0..chns-1}} where {chns=img->sz[0]}.
    
    If {map} sets its {invalid} argument to TRUE, the procedure sets
    all {f} samples to {undef}.
    
    The {*debugp} variable is passed to {map}. (If {debugp} is NULL,
    the procedure provides the address of an internal temporary
    variable.) If {map} sets that variable to TRUE, the procedure
    prints some debugging info. */

void fitr_eval_around_point
  ( float_image_t *img, 
    r2_t *p, 
    r2x2_t *J,
    bool_t avg,
    float v[],
    bool_t debug
  );
  /* Evaluates the image {img} at the point {p = (x,y)} (relative to
    the image's lower left corner). The value is an average (if {avg}
    is true) or partition integral (if {avg} is false) of the values
    of {img} around {p}, using a pixel sampling kernel whose domain
    has been transformed by the Jacobian {J}. The result is returned
    in {v[0..chns-1]} where {chns=img->sz[0]}. */

void fitr_accum_pixel(int chns, float *vs, double wt, float *v, double *wtotP);
  /* Adds {wt*vs[k]} to {v[k]} for {k} in {0..chns-1}. Also adds {wt} to {*wtotP}. */

double fitr_sample_weight(double d, double sigma);
  /* Returns the relative weight of a sample that is displaced by 
    a distance {d} from the center sample, assuming that 
    the samples are to be smoothed by a Gaussian with 
    deviation {sigma}. */

void fitr_check_jacobian(r2_t *op, r2x2_t *J, fitr_transform_t *map, double eps);
  /* Checks the given Jacobian {J} against the numerically estimated 
    derivatives of {ip} relative to {op}, where {ip} is computed from
    {op} by the {map} procedure.   The estimates are obtained by central
    divided difference with step not exceeding {eps} in the output or 
    input domain. */

void fitr_fill_pixel(int chns, float *v, float und);
  /* Sets {v[0..chns-1]} to the value {und}. */

void fitr_scale_pixel(int chns, double s, float *v);
  /* Scales {v[0..chns-1]} by {s}. */

void fitr_debug_direction(char *label, r2_t *v, char *tail);
  /* Prints the vector {v} to {stderr}. */

void fitr_debug_eigen(char *label, r2_t *e, char *tail);
  /* Prints the eigenvalues {e[0..1]} to {stderr}. */

/* IMPLEMENTATIONS */

void fitr_transform
  ( float_image_t *iimg,   /* Input image. */
    fitr_transform_t *map, /* Output-to-input coordinate transformation. */
    float undef,           /* Sample value for undefined output pixels. */
    bool_t avg,            /* TRUE to average pixels, FALSE to add them. */
    float_image_t *oimg    /* Output image. */
  )
  { demand(iimg->sz[0] == oimg->sz[0], "images must have the same channels");
    int chns = oimg->sz[0];
    int ocols = oimg->sz[1];
    int orows = oimg->sz[2];
    float fo[chns];
    int row, col;
    for (row = 0; row < orows; ++row)
      { for (col = 0; col < ocols; ++col)
          { bool_t debug = FALSE;
            /* Get transformed pixel: */
            fitr_get_transformed_pixel(iimg, col, row, map, undef, avg, fo, &debug);
            if (debug) { fitr_debug_pixel("  fo", col+0.5, row+0.5, chns, fo, "\n"); }
            /* Store it in the image: */
            float_image_set_pixel(oimg, col, row, fo);
            if (debug) { fprintf(stderr, "\n"); }
          }
      }
  }    

#define MAX_TAYLOR_ERROR (+INF)  
  /* For now */

#define MAX_SAMPLING_RADIUS (1000.0)
  /* Max radius of output pixel after mapping to input. */

void fitr_get_transformed_pixel
  ( float_image_t *img, 
    int col, 
    int row, 
    fitr_transform_t *map, 
    float undef, 
    bool_t avg,
    float *f,
    bool_t *debugp
  )
  { 
    int chns = img->sz[0];
    bool_t debug = (debugp == NULL ? FALSE : (*debugp));

    /* To make the sampling integrals manageable, we replace the transform
      {map} is by its 1st degree Taylor expansion.  
      
      More precisely, for each output pixel center {op} we approximate
      {map(op+r)}, for small {r}, by {aff(r) = ip + r J}, where {ip}
      is {map(op) and {J} is the Jacobian matrix of {map} evaluated at
      {op}. This approximation assumes that the higher-order terms of
      the Taylor expansion is negligible as long as {r} lies within
      the sampling kernel's support (typically, a few output pixels).
      The integral then reduces to a sum of input pixel values times
      affinely-transformed images of the reconstruction kernel {h}. */

    /* Get Cartesian coordinates {xc,yc} of destination pixel's center: */
    r2_t op = (r2_t) {{ (double)col + 0.5, (double)row + 0.5 }};
    
    /* Compute corresp point {pt=(xt,yt)} in source image, and Jacobian: */
    r2x2_t J; /* Jacobian {{dxt/dxc, dxt/dyc}, {dyt/dxc, dyt/dyc}} */
    r2_t ip;
    bool_t invalid; /* Set to TRUE by {map_point} if point is invalid. */
    map(&op, &ip, &J, &invalid, &debug);
    
    if (debug) { fprintf(stderr, "  computing output pixel with indices (%d,%d):\n", col,row); }
    if (debug) { fitr_debug_point("    po", FALSE,   &op, NULL, "\n"); }
    if (debug) { fitr_debug_point("    pi", invalid, &ip, &J, "\n"); }
    if (debug & (!invalid)) { fitr_check_jacobian(&op, &J, map, 1.0e-5); }

   /* Get input image value at point {pt}: */
    if (invalid)
      { /* There is no image on the negative side of the two-sided plane: */
        fitr_fill_pixel(chns, f, undef);
      }
    else
      { /* demand(err <= MAX_TAYLOR_ERROR, "excessive warp, should use subsampling"); */
        /* Sample points of input image: */
        fitr_eval_around_point(img, &ip, &J, avg, f, debug);
      }
    
    /* Return the debug variable to the caller: */
    if (debugp != NULL) { (*debugp) = debug; }
  }

void fitr_apply_radial_map(r2_t *p, double R, r2x2_t *J, bool_t *invalidp)
  { 
    /* If {p} is already invalid, do nothing: */
    if ((invalidp != NULL) && (*invalidp)) { return; }
    
    /* If the radial map is not the identity, apply it: */
    if (R != 0)
      { /* Get given point coordinates {x,y}: */
        double pX = p->c[0];
        double pY = p->c[1];
        
        /* Compute relative distance squared {s}: */
        double s = (pX*pX + pY*pY)/(R*R);
        
        /* Compute scaling factor {f} and its derivative {df_ds}: */
        double f, df_ds;
        if (R > 0)
          { /* Stretching map (pincushion deformation). */
            /* The stretching is defined only inside the {R}-circle: */
            if (s >= 1.0) 
              { if (invalidp != NULL) { (*invalidp) = TRUE; }
                return;
              }
            f = 1.0/(1.0 - s);
            df_ds = f*f;
          }
        else
          { /* Shrinking map (barrel deformation): */
            double r = sqrt(1 + 2*s);
            f = 2.0/(1 + r);
            df_ds = -f*f/r/2;
          }
        
        /* Compute the mapped point coordinates: */
        double qX = pX*f; 
        double qY = pY*f;
        
        /* Update the given point {p}: */
        p->c[0] = qX; p->c[1] = qY;

        if (J != NULL)
          { /* Compute the Jacobian {K} of this map: */
            r2x2_t K;
            K.c[0][0]= f + 2*df_ds*pX*pX/(R*R); /* dqX_dpX */ 
            K.c[0][1]= 0 + 2*df_ds*pX*pY/(R*R); /* dqY_dpX */ 
            K.c[1][0]= 0 + 2*df_ds*pX*pY/(R*R); /* dqX_dpY */ 
            K.c[1][1]= f + 2*df_ds*pY*pY/(R*R); /* dqY_dpY */ 

            /* Update the Jacobian {J}: */
            r2x2_mul(J, &K, J);
          }
      }
  }

void fitr_apply_projective_map(r2_t *p, r3x3_t *M, r2x2_t *J, bool_t *invalidp)
  { 
    /* If {p} is already invalid, do nothing: */
    if ((invalidp != NULL) && (*invalidp)) { return; }
    
    /* Convert {p} to homogeneous coordinates {hp}: */
    r3_t hp = (r3_t) {{ 1.0, p->c[0], p->c[1] }}; 

    /* Map {hp} to {hq}, get the homogeneous coordinates {qw,qx,qy}: */
    r3_t hq;
    r3x3_map_row(&hp, M, &hq);
    double qw = hq.c[0]; 
    double qx = hq.c[1]; 
    double qy = hq.c[2]; 
    
    /* Check for validity of result: */
    if (qw <= 0.0)
      { /* Result is at infinity, or beyond: */
        if (invalidp != NULL) { (*invalidp) = TRUE; } 
        return; 
      }
    else
      { /* Compute the Cartesian coordinates {qX,qY} of {q}: */
        double qX = qx/qw; 
        double qY = qy/qw;
        
        /* Update the given point {p}: */
        p->c[0] = qX; p->c[1] = qY;
        
        if (J != NULL)
          { /* Compute the Jacobian of {hq} (homogeneous) w.r.t {p} (Cartesian): */
            double dqw_dpX = M->c[1][0], dqw_dpY = M->c[2][0];
            double dqx_dpX = M->c[1][1], dqx_dpY = M->c[2][1];
            double dqy_dpX = M->c[1][2], dqy_dpY = M->c[2][2];

            /* Compute the Jacobian of {q} (Cartesian) w.r.t {p} (Cartesian): */
            r2x2_t K;
            K.c[0][0]= (dqx_dpX - qX*dqw_dpX)/qw; /* dqX_dpX */ 
            K.c[0][1]= (dqy_dpX - qY*dqw_dpX)/qw; /* dqY_dpX */ 
            K.c[1][0]= (dqx_dpY - qX*dqw_dpY)/qw; /* dqX_dpY */ 
            K.c[1][1]= (dqy_dpY - qY*dqw_dpY)/qw; /* dqY_dpY */ 

            /* Update the Jacobian {J}: */
            r2x2_mul(J, &K, J);
          }
      }
  }

#define NFILTER 1
/* #define NFILTER 2 */

#define DF (NFILTER-1)

void fitr_eval_around_point
  ( float_image_t *img, 
    r2_t *p,
    r2x2_t *J,
    bool_t avg,
    float v[],
    bool_t debug
  )
  {
    /* Get image dimensions: */
    int chns = img->sz[0];
    
    /* Get coordinates {(x,y)} of center point: */
    double x = p->c[0], y = p->c[1];
    
    if (debug) { fprintf(stderr, "    evaluating img(%9.4f,%9.4f)\n", x, y); }
    
    /* If {J} expands the space along some direction, we may need to
      interpolate at various points near {p}, spaced at most 1
      unit apart, and combine those samples with the proper
      weights. To ensure the proper spacing, we place the samples
      at a unit-step rectangular grid aligned with the vectors
      {u*J} and {v*J} where {u} and {v} are the eigenvectors of
      {J*J^t}. These are the directions in the input domain along
      which {J} has the maximum and minimum stretch. */

    /* Compute the eigenvectors {e.c[0..1]}} and eigenvalues {R.c[0..1]} of {J*J^t}: */
    r2_t e;   /* Eigenvalues of {J*J^t}. */
    r2x2_t R; /* Rows are the corresponding eigenvectors (eventually). */
    r2x2_mul_tr(J,J,&R);         /* Now {R = J*J^t}. */
    r2x2_sym_eigen(&R, &e, &R);  /* Now {R} has the eigenvectors of {J*J^t}. */
    if (debug) { fitr_debug_eigen("      e", &e, "\n"); }
    if (debug) { fitr_debug_rows("      R", &R, "\n"); }
    
    /* Compute the directions of maximum stretch: */
    r2x2_mul(&R, J, &R);        /* Now {R} has the input resampling axes. */
    if (debug) { fitr_debug_rows("      U", &R, "\n"); }

    /* Compute the number of steps {ns[i]} to take along each vector,
      and their weight factors. If there is no stretching along 
      the sampling axis {i}, then one sample suffices (with unit weight).
      If there is stretching by a factor {f > 1}, we take samples
      at unit distance apart, and average them, to remove the high
      frequencies. */
      
    double stepsize = 1.0; /* Step size for smoothing of interpolated values. */
    int ns[2];       /* Will step from {-ns[i]} to {+ns[i]} along sampling axis {i}. */
    double sigma[2]; /* Amount of extra smoothing needed along sampling axis {i}. */
    int i;
    for (i = 0; i < 2; i++)
      { double str = hypot(R.c[i][0], R.c[i][1]); /* Stretch along sampling axis {i}. */
        if (debug) 
          { fprintf(stderr, "      str[%d] = %8.2f\n", i, str);
            double sqe = sqrt(e.c[i]);
            fprintf(stderr, "      sqe[%d] = %8.2f\n", i, sqe);
            affirm(fabs(str-sqe) < 1.0e-6*(str+sqe), "bug in eigenvalue");
          }
        if (str <= 1.0)
          { /* Input image is not being squeezed; just interpolate it. */
            sigma[i] = 0.0; ns[i] = 0;
            /* We don't need axis sampling axis {i}: */
            R.c[i][0] = R.c[i][1] = 0.0;
          }
        else
          { /* Input image is being squeezed; must interpolate and smooth. */
            /* Normalize sampling xxis vector {R[i]} to unit length: */
            R.c[i][0] /= str; R.c[i][1] /= str; 
            /* Avoid excessive sampling: */
            if (str > MAX_SAMPLING_RADIUS) { str = MAX_SAMPLING_RADIUS; }
            /* Compute width {sigma[i]} of Gaussian smoothing kernel: */
            sigma[i] = sqrt(str*str - 1.0);
            /* Compute number of steps needed for {sigma[i]}-smoothing: */
            ns[i] = (int)ceil(3*sigma[i]/stepsize) - 1;
            if (ns[i] < 0) { ns[i] = 0; }
          }
        if (debug) 
          { fprintf(stderr, "      sigma[%d] = %8.2f", i, sigma[i]);
            fprintf(stderr, " ns[%d] = %d\n", i, ns[i]);
          }
      }

    /* Clear the result pixel: */
    int ich;
    for (ich = 0; ich < chns; ich++) { v[ich] = 0.0; }

    /* Accumulate samples: */
    double wtot = 0;
    int k0, k1;
    float fs[chns];
    for (k0 = -ns[0]; k0 <= +ns[0]; k0++)
      { double f0 = k0*stepsize; /* Sample displacement along evec {R[0]}. */
        double w0 = fitr_sample_weight(f0, sigma[0]); /* Sample weight for {f0}. */
        for (k1 = -ns[1]; k1 <= +ns[1]; k1++)
          { double f1 = k1*stepsize; /* Sample displacement along evec {R[1]}. */
            double w1 = fitr_sample_weight(f1, sigma[1]); /* Sample weight for {f1}. */
            /* Compute sample point {(xs,ys)}: */
            double xs = x + f0*R.c[0][0] + f1*R.c[1][0];
            double ys = y + f1*R.c[0][1] + f1*R.c[1][1];
            /* Evaluate source image at fractional point {(xs,ys)}: */
            float_image_interpolate_pixels(img, xs, ys, fs);
            if (debug) { fitr_debug_pixel("        is", xs, ys, chns, fs, ""); }
            double ws = w0*w1; /* Weight of sample. */
            if (debug) { fprintf(stderr, "        weight = %8.4f\n", ws); }
            /* Accumulate: */
            fitr_accum_pixel(chns, fs, ws, v, &wtot);
          }
      }
    if (debug) { fitr_debug_pixel("      ac", x, y, chns, v, "\n"); }
    if (debug) { fprintf(stderr, "      wtot = %8.4f\n", wtot); }
    
    /* Compute scale factor to account for non-unit-sum weights: */
    assert(wtot != 0);
    double scale = 1.0/wtot;
    
    /* If {avg} is FALSE, correct for the Jacobian's stretch/shrink: */
    if (! avg) { scale *= fabs(r2x2_det(J)); }

    /* Scale pixel value: */
    if (scale != 1.0) { fitr_scale_pixel(chns, scale, v); } 
    if (debug) { fitr_debug_pixel("    ev", x, y, chns, v, "\n"); }
  }
  
double fitr_sample_weight(double d, double sigma)
  { 
    if ((sigma == 0.0) || (d == 0.0))
      { return 1.0; }
    else
      { double z = d/sigma;
        return exp(-z*z/2);
      }
  }

void fitr_accum_pixel(int chns, float *vs, double wt, float *v, double *wtotP)
  {
    if (wt != 0.0)
      { int ich;
        for (ich = 0; ich < chns; ich++)
          { v[ich] += wt*vs[ich]; }
        (*wtotP) += wt;
      }
  }

void fitr_fill_pixel(int chns, float *v, float val)
  {
    int ich;
    for (ich = 0; ich < chns; ich++) 
      { v[ich] = val; }
  }

void fitr_scale_pixel(int chns, double s, float *v)
  {
    int ich;
    for (ich = 0; ich < chns; ich++) { v[ich] *= s; }
  }

void fitr_debug_pixel(char *label, double x, double y, int chns, float *v, char *tail)
  { 
    int ich;
    fprintf(stderr, "%s(%9.4f,%9.4f) = (", label, x, y);
    for (ich = 0; ich < chns; ich++) 
      { fprintf(stderr, " %7.4f", v[ich]); }
    fprintf(stderr, " )%s", tail);
  }

void fitr_check_jacobian(r2_t *op, r2x2_t *J, fitr_transform_t *map, double eps)
  {
    fprintf(stderr, "    checking the Jacobian ...\n");
    /* Estimate the magnitudes {mag_J} of the Jacobian and {mag_op} of {op}: */
    double mag_J = fabs(J->c[0][0]) + fabs(J->c[0][1]) + fabs(J->c[1][0]) + fabs(J->c[1][1]); 
    double mag_op = fabs(op->c[0]) + fabs(op->c[1]);
    if (mag_J < 1.0e-100) { return; }
    /* Choose a small enough step ... */
    double step = eps*mag_op/(mag_J < 1 ? 1 : mag_J);
    /* ... but not too small: */
    if (step <  1.0e-6*mag_op) { step = 1.0e-6*mag_op; }
    fprintf(stderr, "    step = %12.8f\n", step);
    /* Establish a limit for the relative error: */
    double tol = 1.0e-4;
    /* Check the partial derivatives numerically: */
    bool_t invalid, debug; /* For {map}'s use. */
    int i, j, k;
    for (i = 0; i < 2; i++)
      { /* Evaluate {map} at points {oq[0..1]} , displaced from {op} along axis {i}: */
        r2_t oq[2], iq[2];
        for (k = 0; k < 2; k++)
          { oq[k] = (*op); 
            oq[k].c[i] += (2*k - 1)*step;
            fitr_debug_point("      oq", FALSE, &(oq[k]), NULL, "\n");
            invalid = FALSE; debug = FALSE;
            map(&(oq[k]), &(iq[k]), NULL, &invalid, &debug);
            fitr_debug_point("      iq", invalid, &(iq[k]), NULL, "\n");
            if (invalid) { return; }
          }
        /* Check the derivatives of {ip} w.r.t. coordinate {i} of {op}: */
        for (j = 0; j < 2; j++)
          { /* Compute the derivative of coordinate {j} w.r.t coordinate {i}: */
            double dnum = (iq[1].c[j] - iq[0].c[j])/(2*step);
            /* Compare with the Jacobian returned by {map}: */
            double djac = J->c[i][j]; 
            double err = fabs(dnum-djac);
            if (err > tol*(fabs(dnum)+fabs(djac)))
              { fprintf(stderr, "    ** Jacobian is inconsistent (err = %+12.6e)\n", err); 
                fprintf(stderr, "    J[%d][%d] = %12.5f", i, j, djac);
                fprintf(stderr, "  numeric d_ip[%d]/d_op[%d] = %12.5f", i, j, dnum);
                fprintf(stderr, "\n");
              }
          }
      }
  } 

void fitr_debug_point(char *label, bool_t invalid, r2_t *p, r2x2_t *J, char *tail)
  {
    fprintf(stderr, "    %s", label);
    if (invalid)
      { fprintf(stderr, " is invalid\n"); }
    else
      { fprintf(stderr, " = (%12.5f %12.5f)", p->c[0], p->c[1]);
        if (J != NULL)
          { fprintf
              ( stderr, 
                "  J = [[%6.1f %6.1f] [%6.1f %6.1f]]", 
                J->c[0][0], J->c[0][1], J->c[1][0], J->c[1][1]
              );
          }
      }
    fprintf(stderr, "%s", tail);
  }

void fitr_debug_rows(char *label, r2x2_t *R, char *tail)
  {
    int i;
    for (i = 0; i < 2; i++)
      { fprintf(stderr, "    %s", label);
        fprintf
          ( stderr, 
            "[%d] = [%6.1f %6.1f]", 
            i, R->c[i][0], R->c[i][1]
          );
        fprintf(stderr, "%s", tail);
      }
  }

void fitr_debug_direction(char *label, r2_t *v, char *tail)
  {
    fprintf(stderr, "    %s", label);
    fprintf(stderr, " = (%+8.5f %+8.5f)", v->c[0], v->c[1]);
    fprintf(stderr, "%s", tail);
  }

void fitr_debug_eigen(char *label, r2_t *e, char *tail)
  {
    fprintf(stderr, "    %s", label);
    fprintf(stderr, " = (%12.5f %12.5f)", e->c[0], e->c[1]);
    fprintf(stderr, "%s", tail);
  }
    
