#ifndef float_image_transform_H
#define float_image_transform_H

/* Tools for projective and barrel/pincushion image correction. */
/* Last edited on 2008-03-17 12:53:41 by stolfi */ 

#include <r2.h>
#include <r2x2.h>
#include <r3x3.h>
#include <bool.h>
#include <float_image.h>

typedef void fitr_transform_t (r2_t *op, r2_t *ip, r2x2_t *J, bool_t *invalidp, bool_t *debugp);
  /* Type of a procedure that defines a geometric transformation between images.  
    
    More precisely, a procedure of this type must implement the
    *inverse* of the desired image transformation. Namely, given the
    coordinates {op} of the center of some pixel in the *output*
    image, it must return the corresponding point {ip} in the *input*
    image. If {J} is not NULL, the procedure must also store into {*J}
    the Jacobian of that transformation, evaluated at {op}.
    
    Point coordinates are relative to the standard {float_image}
    coordinate system for the relevant image. Namely, the X axis point
    to the right, the Y axis points up, the origin is at the bottom
    left corner of the image, and pixels are 1 unit apart.  Note 
    that pixel centers have half-integer coordinates.
    
    The procedure may set {*invalid = TRUE} to signal that the point
    {op} has no corresponding input point {ip}. It may set
    {*debugp=TRUE} to request debugging printout for that point. */

void fitr_transform
  ( float_image_t *iimg,   /* Input image. */
    fitr_transform_t *map, /* Output-to-input coordinate transformation. */
    float undef,           /* Sample value for undefined output pixels. */
    bool_t avg,            /* TRUE to average pixels, FALSE to add them. */
    float_image_t *oimg    /* Output image. */
  );
  /* Applies the transformation {map} to the image {iimg} and
    stores the result in {oimg}. 
    
    More precisely, the value of the transformed image {timg} at point
    {op} is assumed to be the value of {iimg} at point {ip}, where
    {ip} is computed by {map(&op,&ip,&J,&invalid,&debug)}. The value
    of {iimg} at a given point is obtained from nearby {iimg} samples
    by suitable interpolation.
    
    If {avg} is true, each sample of {oimg} is computed by a suitable
    average of the {iimg} values around the point {ip} corresponding
    to the pixel's center. If {avg} is FALSE, the procedure computes a
    weighted integral, so as to approximately preserve the total
    integral of {iimg} over its domain. This option is appropriate,
    for instance, when {iimg} is a probability distribution, or a
    weight mask.
    
    The procedure assumes that the given {map} can be approximated by
    an affine (1st degree) map, which is determined by the output
    pixel center {op}, the corresponding point {ip} in the input
    domain, and the Jacobian matrix {J} returned by the {map}
    procedure. This approximation is assumed to be valid within a
    couple of pixels of point {op}.
    
    If {map} sets the {invalid} flag when applied to a pixel center
    {op}, the procedure assums that {op} has no corresponding point in
    the input domain, and sets all pixel's samples to the given
    {undef} value.

    The input image {ip} is assumed to be extended to infinity by
    replication of the pixels around its edges. This assumption is
    relevant when {map} sets its {invalid} argument to FALSE but
    returns a point {ip} that lies outside the domain of {iimg}.
    
    If {map} sets {debug} to TRUE, the procedure prints debugging
    information about the computation of the output pixel with center
    {op}. */

/* SOME GEOMETRIC TRANSFORMATIONS */

void fitr_apply_radial_map(r2_t *p, double R, r2x2_t *J, bool_t *invalidp);
  /* Applies to point {p} a radial deformation map with parameter {R}.
    Also post-multiplies the matrix {J} by the Jacobian of the map. */

void fitr_apply_projective_map(r2_t *p, r3x3_t *M, r2x2_t *J, bool_t *invalidp);
  /* Applies to {p} a projective transformation defined by the matrix {M}.
    Also post-multiplies the matrix {J} by the Jacobian of the map. */

/* DEBUGGING AIDS */

void fitr_debug_pixel(char *label, double x, double y, int chns, float *v, char *tail);
  /* Prints the point {(x,y)} and the float-valued samples
    {v[0..chns-1]} to {stderr}, tagged with the string {label} and
    terminated by the string {tail}. */

void fitr_debug_point(char *label, bool_t invalid, r2_t *p, r2x2_t *J, char *tail);
  /* Prints the point {(x/w,y/w)} to {stderr}, tagged with the
    string {label} and terminated by the string {tail}. If the
    Jacobian {J} is not NULL, prints it too. */

void fitr_debug_rows(char *label, r2x2_t *R, char *tail);
  /* Prints the rows of {R} as vectors to {stderr}. Each row
    is tagged with the string {label} and its index,
    and terminated by the string {tail}. */

#endif
