/* See float_ppm_image.h */
/* Last edited on 2008-11-11 01:21:12 by stolfi */ 

#define _GNU_SOURCE
#include <limits.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <indexing.h>
#include <affirm.h>
#include <bool.h>
 
#include <float_image.h>
#include <float_pnm_image.h>
#include <sample_conv.h>
 
float_image_t *float_image_from_pnm_image
  ( pnm_image_t *iim, 
    double lo[], 
    double hi[], 
    bool_t verbose
  )
  { 
    /* Get image dimensions: */
    int NX = iim->cols;
    int NY = iim->rows;
    
    /* Channel counts: */
    int chns = iim->chns; /* Num of channels. */
    
    /* Allocate float image: */
    float_image_t *fim = float_image_new(chns, NX, NY);
    
    /* Max sample value in integer image: */
    pnm_sample_t maxval = iim->maxval;
    
    /* Input and output range registers: */
    sample_uint_t imin[chns], imax[chns]; /* Input range registers. */ 
    float vmin[chns], vmax[chns];         /* Output range registers. */ 
    int c; /* Channel index. */
    for (c = 0; c < chns; c++) 
      { imin[c] = maxval;
        imax[c] = 0;
        vmin[c] = +INF;
        vmax[c] = -INF;
      }
    
    /* Convert pixels, keep statistics: */
    int x, y;
    for(y = 0; y < NY; y++)
      { int pgmy = NY - 1 - y;
        pnm_sample_t *prow = iim->smp[pgmy];
        for(x = 0; x < NX; x++)
          { for (c = 0; c < chns; c++)
              { /* Convert int sample {*prow} to float {v}, store, keep stats: */
                pnm_sample_t ismp = (*prow);
                double loc = (lo == NULL ? 0.0 : lo[c]);
                double hic = (hi == NULL ? 1.0 : hi[c]);
                float fsmp = sample_conv_floatize
                  ( ismp, maxval, loc, hic, &(imin[c]), &(imax[c]), &(vmin[c]), &(vmax[c]) );
                float_image_set_sample(fim, c, x, y, fsmp);
                prow++;
              }
          }
      }
    
    if (verbose) 
      { /* Print statistics: */
        long int NPIX = ((long int)NX)*((long int)NY);
        fprintf(stderr, "  %ld pixels in PNM image\n", NPIX);
        if (NPIX > 0)
          { for (c = 0; c < chns; c++)
              { double loc = (lo == NULL ? 0.0 : lo[c]);
                double hic = (hi == NULL ? 1.0 : hi[c]);
                sample_conv_print_floatize_stats
                  ( c, c, imin[c], imax[c], maxval, loc, hic, vmin[c], vmax[c] );
              }
          }
      }
    
    return fim;
  }

pnm_image_t *float_image_to_pnm_image
  ( float_image_t *fim, 
    int chns,
    double lo[], 
    double hi[], 
    int ch[],
    pnm_sample_t maxval, 
    bool_t verbose
  )
  { /* Get image dimensions: */
    int NX = fim->sz[1];
    int NY = fim->sz[2];
    
    /* Channel counts: */
    int fchns = fim->sz[0]; /* Num channels in float image. */
    int ichns = chns;       /* Num channels in integer image. */
    
    /* Allocate PGM/PPM image: */
    pnm_image_t *iim = pnm_image_new(NX, NY, ichns);
    
    /* Set max sample value in integer image: */
    iim->maxval = maxval;
    
    /* Channel indexing variables: */
    int k; /* Channel of integer image. */
    int c; /* Channel of float image. */
    
    /* Input and output range registers: */
    float vmin[ichns], vmax[ichns];         /* Float pixel range. */
    sample_uint_t imin[ichns], imax[ichns]; /* Int pixel range. */
    int clo[ichns], chi[ichns];             /* Counts of lo-clipped and hi-clipped pixels. */
    for (k = 0; k < ichns; k++) 
      { clo[k] = chi[k] = 0;
        vmin[k] = +INF;
        vmax[k] = -INF; 
        imin[k] = maxval;
        imax[k] = 0;  
      }
    
    /* Convert pixels, store in {iim}, keep statistics: */
    int x, y;
    for(y = 0; y < NY; y++)
      { int ppmy = NY-1-y;
        pnm_sample_t *prow = iim->smp[ppmy];
        for(x = 0; x < NX; x++)
          { /* Convert float pixel {fpxy[c..c+2]} to integer pixel {ipxy[0..2]}, keep stats: */
            for (k = 0; k < ichns; k++)
              { double lok = (lo == NULL ? 0.0 : lo[k]);
                double hik = (hi == NULL ? 1.0 : hi[k]);
                c = (ch == NULL ? k : ch[k]);
                float v = ((c < 0) || (c >= fchns) ? 0.0 : float_image_get_sample(fim, c, x, y));
                (*prow) = sample_conv_quantize
                  ( v, maxval, lok, hik, 
                    &(vmin[k]), &(vmax[k]), &(clo[k]), &(chi[k]), &(imin[k]), &(imax[k])
                  );
                prow++;
              }
          }
      }
    
    if (verbose)
      { /* Print statistics: */
        long int NPIX = ((long int)NX)*((long int)NY);
        fprintf(stderr, "  %ld pixels in float image\n", NPIX);
        if (NPIX > 0)
          { for (k = 0; k < chns; k++)
              { double lok = (lo == NULL ? 0.0 : lo[k]);
                double hik = (hi == NULL ? 1.0 : hi[k]);
                c = (ch == NULL ? k : ch[k]);
                sample_conv_print_quantize_stats
                  ( c, k, vmin[k], vmax[k], lok, hik, clo[k], chi[k], maxval, imin[k], imax[k]);
              }
          }
      }
    
    return iim;
  }
