/* See {float_pnm_image_buffer.h}. */
/* Last edited on 2006-11-21 02:51:51 by stolfi */

#include <assert.h>
#include <affirm.h>
#include <bool.h>
#include <jspnm.h>
#include <jspnm_image.h>

#include <float_pnm_image_buffer.h>

/* Internal prototypes: */

double **fpnm_allocate_buffer_rows(int bufrows, int smpr, bool_t clear);

fpnm_buffer_t *fpnm_read_buffer_new(FILE *rd, int bufrows)
  { /* Allocate top record: */
    fpnm_buffer_t *im = (fpnm_buffer_t *)notnull(malloc(sizeof(fpnm_buffer_t)), "no mem");
    /* Read the input file header: */
    pnm_read_header
     ( rd, &(im->cols), &(im->rows), &(im->chns), &(im->maxval), 
       &(im->raw), &(im->bits), &(im->format)
     );
    im->smpr = im->cols*im->chns; /* Samples per row. */
    im->smp = pnm_image_alloc_pixel_row(im->cols, im->chns);
    im->bad = (((uint64_t)PNM_MAX_SAMPLE)+1); /* No invalid sample value. */
    im->rowptr = fpnm_allocate_buffer_rows(bufrows, im->smpr, FALSE);
    im->yfrst = 0; im->ynext = 0; /* No rows loaded yet. */
    return im;
  }

void fpnm_buffer_free(fpnm_buffer_t *im)
  { if (im != NULL)
      { if (im->smp != NULL) { free(im->smp); im->smp = NULL; }
        if (im->rowptr != NULL)
          { int k;
            for (k = 0; k < im->bufrows; k++)
              { double **p = &(im->rowptr[k]);
                if (*p != NULL) { free(*p); *p = NULL; }
              }
            free(im->rowptr); im->rowptr = NULL; 
          }
        free(im); im = NULL; 
      }
  }

double **fpnm_allocate_buffer_rows(int bufrows, int smpr, bool_t clear)
  { double **rowptr = (double **)notnull(malloc(bufrows*sizeof(double *)), "no mem");
    int yb;
    for (yb = 0; yb < bufrows; yb++) 
      { double *sP = (double *)notnull(malloc(smpr*sizeof(double)), "no mem");
        rowptr[yb] = sP;
        if (clear) { int k; for (k = 0; k < smpr; k++, sP++) { (*sP) = 0.0; } }
      }
    return rowptr;
  }

double *fpnm_read_buffer_get_row(FILE *rd, fpnm_buffer_t *im, int y) 
  { /* Row index must be valid: */
    if ((y < 0) || (y >= im->rows)) { return NULL; }
    /* Compute index {yyyy} of next row to be read from file: */
    while (im->ynext <= y) { fpnm_read_buffer_load_next_row(rd, im); }
    /* Did we succeed? */
    demand(im->yfrst <= y, "row cannot be read again");
    assert(y < im->ynext);
    return im->rowptr[y % im->bufrows];
  }

void fpnm_read_buffer_load_next_row(FILE *rd, fpnm_buffer_t *im) 
  { demand(im->ynext < im->rows, "no more rows to be read"); 
    /* Read one row of sample values from input image into {im->smp}: */
    pnm_read_pixels(rd, im->smp, im->cols, im->chns, im->maxval, im->raw, im->bits);
    /* Find row in {ibuf} for converted row {im->ynext}: */
    int yb = im->ynext % im->bufrows; /* Row of buffer where to put input row {yyyy}. */
    /* Convert samples {im->smp[0..im->smpr-1]} to [0_1] scale, save in {ibuf}: */
    pnm_sample_t *sP = im->smp; /* Scans raw samples. */
    double *dP = im->rowptr[yb]; /* Start of row {yb} in {ibuf}. */
    int k;
    for (k = 0; k < im->smpr; k++, dP++, sP++)
      { (*dP) = pnm_floatize((*sP), im->maxval, im->bad); }
    /* Update {ynext}: */
    im->ynext++;
    /* Check whether we lost row {yfrst}: */
    if (im->ynext - im->yfrst > im->bufrows) { im->yfrst++; }
  }

fpnm_buffer_t *fpnm_write_buffer_new 
  ( FILE *wr, 
    pnm_sample_t maxval, 
    int rows, 
    int cols, 
    int chns, 
    bool_t forceplain,
    int bufrows
  )
  { /* Allocate top record: */
    fpnm_buffer_t *im = (fpnm_buffer_t *)notnull(malloc(sizeof(fpnm_buffer_t)), "no mem");
    /* Select output format and write image header: */
    im->chns = chns;
    im->rows = rows;
    im->cols = cols;
    im->maxval = maxval;
    pnm_choose_output_format
      ( maxval, chns, forceplain,
        &(im->format), &(im->raw), &(im->bits)
      );
    pnm_write_header(wr, cols, rows, im->maxval, im->format);
    im->smpr = im->cols*im->chns; /* Samples per row. */
    im->smp = pnm_image_alloc_pixel_row(im->cols, im->chns);
    im->bad = (((uint64_t)PNM_MAX_SAMPLE)+1); /* No invalid sample value. */
    im->rowptr = fpnm_allocate_buffer_rows(bufrows, im->smpr, TRUE);
    /* Assume the buffer has all the initial rows: */
    im->yfrst = 0; im->ynext = (bufrows < im->rows ?  bufrows : im->rows); 
    return im;
  }

double *fpnm_write_buffer_get_row(FILE *wr, fpnm_buffer_t *im, int y)
  { /* Row index must be valid: */
    if ((y < 0) || (y >= im->rows)) { return NULL; }
    /* Roll buffer forward until row {y} is in buffer: */
    while (im->ynext <= y) { fpnm_write_buffer_dump_first_row(wr, im); }
    /* Did we succeed? */
    demand(im->yfrst <= y, "cannot get row that has been written out");
    assert(y < im->ynext);
    return im->rowptr[y % im->bufrows];
  }

void fpnm_write_buffer_dump_first_row(FILE *wr, fpnm_buffer_t *im) 
  { demand(im->yfrst < im->rows, "no more rows to write out");
    /* Convert first row, write it, clear it: */
    pnm_sample_t *sP = im->smp; /* Scans raw samples. */
    int yb = im->yfrst % im->bufrows; /* index of row {yfrst} in buffer. */
    double *dP = im->rowptr[yb]; /* Start of row {yb} in {ibuf}. */
    int k;
    for (k = 0; k < im->smpr; k++, dP++, sP++)
      { (*sP) = pnm_quantize((*dP), im->maxval, im->bad); (*dP) = 0.0; }
    pnm_write_pixels(wr, im->smp, im->cols, im->chns, im->maxval, im->raw, im->bits);
    /* Remove it from the buffer: */
    im->yfrst++;
    /* Allocate its space to the next image row, if any: */
    if (im->ynext < im->rows) { im->ynext++; }
  }
