#ifndef float_pnm_image_buffer_H
#define float_pnm_image_buffer_H

/* Multipurpose float-format image buffer */
/* Last edited on 2008-05-24 23:04:40 by stolfi */

#include <jspnm.h>
#include <bool.h>

typedef struct fpnm_buffer_t
  { /* Image file data: */
    pnm_format_t format; /* Image file format (PGM_FORMAT, RPGM_FORMAT, etc.). */
    int rows;            /* Image width. */
    int cols;            /* Image height. */
    int chns;            /* Image channels. */
    int smpr;            /* Image samples per row { = cols*chns}. */
    pnm_sample_t maxval; /* Maximum pixel value */
    bool_t raw;          /* TRUE if `raw' format variant. */
    bool_t bits;         /* TRUE for PBM format (`raw' or `plain'). */
    /* Sample conversion parameters: */
    uint32_t bad;        /* Integer sample value meaning `invalid', or {>maxval} if none. */
    /* PNM sample buffer: */
    pnm_sample_t *smp;   /* Buffer for one row of pixels. */
    /* Float sample buffer: */
    int bufrows;         /* Number of rows in buffer. */
    int yfrst;           /* Index of first image row currently in buffer. */
    int ynext;           /* Index of first image row that has never been in buffer. */
    double **rowptr;     /* Pointers to rows of floated pixels. */
  } fpnm_buffer_t;
  /* A {fpnm_buffer_t} stores a horizontal band from the input image,
    consisting of rows {[yfrst..ynext-1]} (maximum of {bufrows} rows).
    The address of row {y}, for {y} in that range, is stored in {p =
    rowptr[y % bufrows]}. The pixel with abscissa {j} in that row is
    stored in {p[chns*j+r]} for {r = 0..chns-1}.
    
    When reading, {im->ynext} is the next image row to read from the file,
    of 0 if no rows were read yet. When writing, {im->yfrst}
    is the next row to write to the image file, or {im->rows} if
    the entire image was written.  The buffer is empty when 
    {im->yfrst == im->ynext} .*/

/* ********************************************************************** */
/* INPUT BUFFERS */

fpnm_buffer_t *fpnm_read_buffer_new(FILE *rd, int bufrows);
  /* Creates a new buffer structure {im} primed for reading from the
    PBM/PGM/PPM image file {rd}.
    
    Initializes the fields {rows,cols,chns,maxval,raw,bits,format} of
    {im} by reading the image header from {rd}. Allocates the sample
    row buffer {im->smp} and the array {im->rowptr} with {bufrows} row
    buffers. Sets {bad} to {PNM_NO_BADVAL}; clients may modify
    it, usually before the first row gets read or written. */
    
double *fpnm_read_buffer_get_row(FILE *rd, fpnm_buffer_t *im, int y);
  /* Makes sure that row {y} of pixels from file {rd} is in the buffer,
   and returns its address there.
   
   If row {y} it is not already in memory, requires {y} be a
   row that has not yet been read from file {rd}, i.e. {y >=
   im->ynext}. In that case, loads into {im} every row from
   row {im->ynext} up to row {y}, with {fpnm_read_buffer_load_next_row}
   (using {im->smp} as a temproary buffer). May cause rows
   starting with {im->yfrst} to disappear from the buffer. 
   Returns NULL if {y} is not in {0..rows-1}. */
   
void fpnm_read_buffer_load_next_row(FILE *rd, fpnm_buffer_t *im);
  /* Reads the next image row (namely row {im->ynext}) from file {rd}
    into {im->smp}, converts its samples to floating point, stores them
    into {*(im->rowptr[k])} for the appropriate index {k}. May cause
    row {im->yfrst} to disappear from the buffer. 
    Fails if {im->ynext >= im->rows}.

    Pixel samples are converted from integer to [double} with
    {pnm_floatize(ival,im->maxval,im->bad)} from {jspnm.h}. Note
    that the {im->bad} integer value, if in {0..im->maxval}, gets
    mapped to {NAN}. */

/* ********************************************************************** */
/* OUTPUT BUFFERS */

fpnm_buffer_t *fpnm_write_buffer_new
  ( FILE *wr, 
    pnm_sample_t maxval, 
    int rows, 
    int cols, 
    int chns, 
    bool_t forceplain,
    int bufrows
  );
  /* Initializes the image buffer {im} for the given
    {maxval,rows,cols,chns}. Chooses a suitable {format} and sets
    {raw,bits} accordingly, then writes the corresponding PNM image
    file header to {wr}. Allocates the sample row buffer {im->smp} and
    the array {im->rowptr} with {bufrows} row buffers.
    Sets {bad} to {PNM_MAX_MAXVAL + 1}. */

double *fpnm_write_buffer_get_row(FILE *rd, fpnm_buffer_t *im, int y);
  /* Makes sure that there is space in the buffer {im} reserved for
   row {y} of the image, and returns the address of that space. Row
   {y} must not have been written yet, i.e. {y > yfrst}. This may
   cause some rows to be written out with
   {fpnm_write_buffer_dump_first_row}, starting with {im->yfrst}. New
   buffer rows are initialized with zeros. Fails if requested to
   return a row that has been written out. Returns NULL if {y} is
   outside {0..rows-1}. */

void fpnm_write_buffer_dump_first_row(FILE *wr, fpnm_buffer_t *im);
  /* Quantizes the first pixel row in the buffer, namely row
    {im->yfrst}, writes it to {wr}, and removes it from the buffer. Fails if
    {im->yfrst >= im->rows}.  To flush the buffered lines,
    up to row {y}, call this procedue until {y < im->yfrst}.
   
   Pixel samples are converted from {double} to integers with
   {pnm_quantize(fval,im->maxval,im->bad)} from {jspnm.h}. Note that
   the {im->bad} integer value, if in {0..im->maxval}, is used to
   encode {NAN} samples only.*/

/* ********************************************************************** */
/* FREEING BUFFER STORAGE */

void fpnm_buffer_free(fpnm_buffer_t *im);
  /* Frees all storage associated with {im}, icluding {*im} itself. */

#endif
