/* see frgb_ops.h 
** Last edited on 2008-07-19 21:12:31 by stolfi
**
** Copyright (C) 2003 by Jorge Stolfi, the University of Campinas, Brazil.
** See the rights and conditions notice at the end of this file.
*/

#define _GNU_SOURCE
#include <math.h>
#include <limits.h>
#include <float.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <frgb_ops.h>

#include <argparser.h>
#include <frgb.h>
#include <bool.h>
#include <fget.h>

/* Test for NaN - in case it is not defined in <math.h> */
#ifdef isnan
#else
int __isnanf(float x);
int __isnan(double x);
int __isnanl(long double x);
#define isnan(x) \
  ( sizeof(x) == sizeof(float) ? __isnanf(x)	: \
    sizeof(x) == sizeof(double) ? __isnan(x) : \
    __isnanl(x) )
#endif

int debug = 0;

frgb_t frgb_scale(double s, frgb_t *a)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++) { p.c[i] = s * a->c[i]; }
    return p;
  }

frgb_t frgb_add(frgb_t *a, frgb_t *b)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++) { p.c[i] = a->c[i] + b->c[i]; }
    return p;
  }
  
frgb_t frgb_sub(frgb_t *a, frgb_t *b)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++) { p.c[i] = a->c[i] - b->c[i]; }
    return p;
  }
  
frgb_t frgb_mul(frgb_t *a, frgb_t *b)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++) { p.c[i] = a->c[i]*b->c[i]; }
    return p;
  }

frgb_t frgb_mix(double ca, frgb_t *a, double cb, frgb_t *b)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++) { p.c[i] = ca*a->c[i] + cb*b->c[i]; }
    return p;
  }

bool_t frgb_is_all_zeros(frgb_t *a)
  { return ((a->c[0] == 0) && (a->c[1] == 0) && (a->c[2] == 0)); }

bool_t frgb_is_all_ones(frgb_t *a)
  { return ((a->c[0] == 1) && (a->c[1] == 1) && (a->c[2] == 1)); }

bool_t frgb_eq(frgb_t *a, frgb_t *b)
  { return ((a->c[0] == b->c[0]) && (a->c[1] == b->c[1]) && (a->c[2] == b->c[2])); }

frgb_t frgb_parse(argparser_t *pp, double lo, double hi)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++)
      { p.c[i] = argparser_get_next_double(pp, lo, hi); }
    return p;
  }

frgb_t frgb_read(FILE *rd, double lo, double hi)
  { frgb_t p;
    int i;
    for (i = 0; i < 3; i++)
      { double pi = fget_double(rd);
        if ((pi < lo) || (pi > hi))
          { fprintf(stderr, "  component [%d] = %25.16e\n", i, pi);
            demand(FALSE, "bad {frgb_t} component value");
          }
        p.c[i] = pi;
      }
    return p;
  }

frgb_t frgb_parse_color(argparser_t *pp)
  { frgb_t p;
    double scale;
    int i;
    for (i = 0; i < 3; i++)
      { p.c[i] = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX); }
    if (argparser_keyword_present_next(pp, "/"))
      { scale = argparser_get_next_double(pp, -DBL_MAX, +DBL_MAX); }
    else
      { scale = 1.0; }
    for (i = 0; i < 3; i++)
      { p.c[i] /= scale; }
    return p;
  }

frgb_t frgb_read_color(FILE *rd)
  { frgb_t p;
    double scale;
    int i;
    for (i = 0; i < 3; i++)
      { p.c[i] = fget_double(rd); }
    fget_skip_spaces(rd);
    if (fget_test_char(rd, '/'))
      { scale = fget_double(rd); }
    else
      { scale = 1.0; }
    for (i = 0; i < 3; i++)
      { p.c[i] /= scale; }
    return p;
  }

double frgb_floatize(int ival, int maxval, double zero, double scale)
  { 
    return ((double)ival - zero)/scale;
  }

double frgb_undo_gamma_gray(double y, double gamma)
  { if (gamma == 1.0)
      { return y; }
    else if (y == 0.0)
      { return 0.0; }
    else if (y == 1.0)
      { return 1.0; }
    else
      { double yin = fabs(y);
        double yout = exp(log(yin)*gamma);
        return (y < 0.0 ? -yout : yout);
      }
  }

frgb_t frgb_correct_arg(frgb_t *p, frgb_t *inGamma, int gray)
  { frgb_t res = *p;
    int i;
    if (inGamma != NULL)
      { for (i = 0; i < 3; i++)
          { res.c[i] = frgb_undo_gamma_gray(res.c[i], inGamma->c[i]); }
      }
    if (gray) { res.c[0] = res.c[1] = res.c[2] = frgb_luminance_CIE_XYZrec601_1(&(res)); }
    return res;
  }
  
double frgb_log_scale_gray(double x)
  { if (x < VAL_EPS) { x = VAL_EPS; }
    return log(x);
  }

void frgb_log_scale(frgb_t *p, int chns)
  { int i;
    for (i = 0; i < chns; i++)
      { double xi = p->c[i];
        if (xi < VAL_EPS) { xi = VAL_EPS; }
        p->c[i] = log(xi);
      }
  }

double frgb_clip_gray(double x)
  {  if (x < 0.0)
      { return 0.0; }
    else if (x > 1.0)
      { return 1.0; }
    else
      { return x; }
  }

double frgb_apply_gamma_gray(double y, double gamma)
  { if (gamma == 1.0)
      { return y; }
    else if (y == 0.0)
      { return 0.0; }
    else if (y == 1.0)
      { return 1.0; }
    else
      { double yin = fabs(y);
        double yout = exp(log(yin)/gamma);
        return (y < 0.0 ? -yout : yout);
      }
  }

void frgb_clip_rgb(frgb_t *p)
  { double lum = frgb_luminance_CIE_XYZrec601_1(p); /* Luminance */
    double obs = 1.0 - lum;  /* "Obscurance" = complement of luminance */
    /* Check for extreme values: */
    if (lum <= 0.0)
      { *p = frgb_Black; }
    else if (obs <= 0.0)
      { *p = frgb_White; }
    else
      { /* Clip {p} to the unit cube, preserving hue: */
        frgb_t x;
        int i;
        /* Compute chrominance {x = lum*White - p},
          and the minimum scale factor {s \geq 1} that leaves 
          {x/s + lum*White} inside the unit cube: */
        double s = 1.0;
        for (i = 0; i < 3; i++)
          { double pi = p->c[i];
            double xi = pi - lum;
            if (s*lum < -xi)
              { s = -xi/lum; }
            else if (s*obs < +xi)
              { s = +xi/obs; }
            x.c[i] = xi;
          }
        if (s > 1.0) 
          { /* Compute output pixel {p = x/s + lum*White}: */
            for (i = 0; i < 3; i++) 
              { double ri = x.c[i]/s + lum;
                /* Guarding against roundoff: */
                if (ri < 0) { ri = 0; }
                if (ri > 1) { ri = 1; }
                p->c[i] = ri;
              }
          }
      }
  }
  
void frgb_clip_rgb_towards_grey(frgb_t *p)
  { frgb_t q = (frgb_t){{ 0.5, 0.5, 0.5 }};
    return frgb_clip_rgb_towards(p, &q);
  }

void frgb_clip_rgb_towards(frgb_t *p, frgb_t *q)
  { int i;
    /* Compute the min {s >= 1} such that {q + (p - q)/s} is in the cube: */
    double s = 1.0; /* L-inf distance from {gv} to middle gray. */
    for (i = 0; i < 3; i++)
      { double pi = p->c[i];
        double qi = q->c[i];
        double qf = 1 - qi;
        demand((qi > 0) && (qf > 0), "q is not in the cube's interior");
        double di = pi - qi;
        if (s*qi < -di) 
          { s = -di/qi; }
        else if (s*qf < +di)
          { s = +di/qf; }
      }
    if (s > 1.0)
      { /* Pull {p} towards {q} until inside the unit cube: */
        for (i = 0; i < 3; i++) 
          { double ri = q->c[i] + (p->c[i] - q->c[i])/s;
            /* Guarding against roundoff: */
            if (ri < 0) { ri = 0; }
            if (ri > 1) { ri = 1; }
            p->c[i] = ri;
          }
      }
  }

double frgb_apply_kappa_gray(double y, double kappa)
  { if (y <= 0.0)
      { return 0.0; }
    else if (y >= 1.0)
      { return 1.0; }
    else if (kappa == 1.0)
      { return y; }
    else
      { return kappa*y/((kappa-1.0)*y + 1.0); }
  }

void frgb_apply_glob_kappa_sat_clip(frgb_t *p, double kap, double satf)
  { double lumI = frgb_luminance_CIE_XYZrec601_1(p);
    double obsI = 1.0 - lumI;  /* "Obscurance" = complement of luminance */
    double satI = 0.0;
    double lumO, obsO;
    double satO = 0.0;
    /* Compute output luminance {lumO} and its complement {obsO}: */
    if (kap != 1.0)
      { lumO = frgb_apply_kappa_gray(lumI, kap); obsO = 1.0 - lumO; }
    else
      { lumO = lumI; obsO = obsI; }
    /* Check for extreme values: */
    if (lumO <= 0.0)
      { *p = frgb_Black; }
    else if (obsO <= 0.0)
      { *p = frgb_White; }
    else
      { /* Replace {p} by its chrominance component {lumI*White - p}
          If {satf < 1.0}, scale the chrominance right away,
          and set {satf} to 1.0; else leave this adjustment for later.
          Then compute the input saturation {satI}, relative to the maximum
          possible saturation for {p}'s hue and luminosity {lumI}:
            { satI = MAX{ s : lumI*White + p/s IN [0_1]^3 } }
          Then clip {satI} to the range {[0_1]}.
          Compute also the saturation {satO} of the same chrominance
          vector, relative to the maximum saturation for {p}'s hue and  
          for the desired luminosity {lumO}:
            { satO = MAX{ s : lumO*White + p/s IN [0_1]^3 } }
        */
        int i;
        for (i = 0; i < 3; i++)
          { double pi = p->c[i];
            pi = pi - lumI;
            if (satf < 1.0) { pi = pi * satf; }
            if (satI < 1.0)
              { if ((pi > +obsI) || (pi < -lumI))
                  { satI = 1.0; }
                else if (satI*lumI < -pi)
                  { satI = -pi/lumI; }
                else if (satI*obsI < +pi)
                  { satI = +pi/obsI; }
              }
            if (satO*lumO < -pi)
              { satO = -pi/lumO; }
            else if (satO*obsO < +pi)
              { satO = +pi/obsO; }
            p->c[i] = pi;
          }
        if (satf < 1.0) { satf = 1.0; }
        /* Compute output pixel {p} from new luminance {lumO} plus the 
          input chroma vector {p}, the latter scaled by
          {f == r/((1-satI) + r*satO)} where {r == satf*lumO/lumI}: */
        { double r = satf*lumO/lumI;
          if (r == 0.0)
            { *p = (frgb_t){{lumO, lumO, lumO}}; }
          else
            { double f = r/((1.0 - satI) + r*satO);
              for (i = 0; i < 3; i++) { p->c[i] = lumO + f * p->c[i]; }
            }
        }
      }
  }

int frgb_quantize(double fval, double zero, double scale, int maxval)
  { if (fval == +INF)
      { return maxval; }
    else if (fval == -INF)
      { return 0; }
    else
      { double ival = fval*scale + zero;
        if (isnan(ival))
          { return maxval/2; }
        else if (ival <= 0.0)
          { return 0; }
        else if (ival >= (double)maxval)
          { return maxval; }
        else
          { return (int)(floor(ival + 0.5)); }
      }
  }
  
int frgb_dequal(double *a, double *b, int chns)
  { int i;
    for (i = 0; i < chns; i++)
      { if (a[i] != b[i]) return 0; }
    return 1;
  }
  
int frgb_fequal(float *a, float *b, int chns)
  { int i;
    for (i = 0; i < chns; i++)
      { if (a[i] != b[i]) return 0; }
    return 1;
  }

void frgb_print(FILE *f, char *pref, frgb_t *p, char *suff)
  { int k;
    fprintf(f, "%s", pref);
    for (k = 0; k < 3; k++) 
      { /* For 16-bit samples, 5 decimals are are nec. & suff. (1/65536 = 0.0000152+): */
        fprintf(f, "%s%8.5f", (k == 0 ? "" : " "), p->c[k]);
      }
    fprintf(f, "%s", suff);
  }

void frgb_print_int_pixel(FILE *f, char *pref, int *p, int chns, char *suff)
  { 
    int k;
    fprintf(f, "%s", pref);
    for (k = 0; k < chns; k++) 
      { fprintf(f, "%s%3d", (k == 0 ? "" : " "), p[k]); }
    fprintf(f, "%s", suff);
  }

void frgb_debug(char *label, int col, int row, frgb_t *p, char *tail)
  { 
    if (debug) 
      { fprintf(stderr, "%s[%3d][%3d] = ", label, row, col);
        frgb_print(stderr, "( ", p, " )");
        fprintf(stderr, "%s", tail);
      }
  }

void frgb_debug_int_pixel(char *label, int col, int row, int *p, int chns, char *tail)
  { 
    if (debug) 
      { fprintf(stderr, "%s[%3d][%3d] = ", label, row, col);
        frgb_print_int_pixel(stderr, "( ", p, chns, " )");
        fprintf(stderr, "%s", tail);
      }
  }

/* COLORSPACE CONVERSIONS */

/*
    Some RGB color space parameters (from Susstrunk, Buckley and Swen 2005)
    as reproduced in Wikipedia:

      Color Space        | WhtPt | Primaries
                         |       |  xR    | yR      | xG      | yG     | xB     | yB
      ISO RGB            | ANY   | 
      Extended ISO RGB   | ANY   | floating
      sRGB, HDTV[1]      | D65   | 0.64   | 0.33    | 0.30    | 0.60   | 0.15   | 0.06
      ROMM RGB           | D50   | 0.7347 | 0.2653  | 0.1596  | 0.8404 | 0.0366 | 0.0001
      Adobe RGB 98       | D65   | 0.64   | 0.34    | 0.21    | 0.71   | 0.15   | 0.06
      Apple RGB          | D65   | 0.625  | 0.34    | 0.28    | 0.595  | 0.155  | 0.07
      NTSC [2]           | C     | 0.67   | 0.33    | 0.21    | 0.71   | 0.14   | 0.08
      NTSC (1979) [3]    | D65   | 0.63   | 0.34    | 0.31    | 0.595  | 0.155  | 0.07
      PAL/SECAM [4]      | D65   | 0.64   | 0.33    | 0.29    | 0.60   | 0.15   | 0.06

      [1] ITU-R BT.709-3
      [2] FCC 1953 
      [3] SMPTE C, SMPTE-RP 145
      [4] EBU 3213, ITU-R BT.470-6
      
    It is not clear which RGB space is assumed by the formulas below.
    Someday we should replace this interface by one with the CIE XYZ
    system as the central standard, and the various RGB systems as 
    peripheral.
*/

void frgb_to_CIE_XYZrec601_1(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double X = +0.606881*R +0.173505*G +0.200336*B;
    double Y = +0.298911*R +0.586611*G +0.114478*B;
    double Z = 00.000000*R +0.066097*G +1.116157*B;
    
    p->c[0] = (float)X;
    p->c[1] = (float)Y; 
    p->c[2] = (float)Z;
  }
  
void frgb_from_CIE_XYZrec601_1(frgb_t *p)
  { 
    double X = (double)p->c[0];
    double Y = (double)p->c[1];
    double Z = (double)p->c[2];

    double R = +1.910027*X -0.532464*Y -0.288214*Z;
    double G = -0.984645*X +1.999130*Y -0.028308*Z;
    double B = +0.058309*X -0.118385*Y +0.897608*Z;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

double frgb_luminance_CIE_XYZrec601_1(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    return +0.298911*R +0.586611*G +0.114478*B;
  }

void frgb_to_CIE_XYZccir709(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double X = +0.412411*R +0.357585*G +0.180454*B;
    double Y = +0.212649*R +0.715169*G +0.072182*B;
    double Z = +0.019332*R +0.119195*G +0.950390*B;
   
    p->c[0] = (float)X;
    p->c[1] = (float)Y; 
    p->c[2] = (float)Z;
  }
  
void frgb_from_CIE_XYZccir709(frgb_t *p)
  { 
    double X = (double)p->c[0];
    double Y = (double)p->c[1];
    double Z = (double)p->c[2];

    double R = +3.240811*X -1.537310*Y -0.498586*Z;
    double G = -0.969241*X +1.875966*Y +0.041554*Z;
    double B = +0.055638*X -0.204007*Y +1.057130*Z;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

double frgb_luminance_CIE_XYZccir709(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    return +0.212649*R +0.715169*G +0.072182*B;
  }


void frgb_to_CIE_XYZitu_D65(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double X = +0.430574*R +0.341550*G +0.178325*B;
    double Y = +0.222015*R +0.706655*G +0.071330*B;
    double Z = +0.020183*R +0.129553*G +0.939180*B;  
   
    p->c[0] = (float)X;
    p->c[1] = (float)Y; 
    p->c[2] = (float)Z;
  }
  
void frgb_from_CIE_XYZitu_D65(frgb_t *p)
  { 
    double X = (double)p->c[0];
    double Y = (double)p->c[1];
    double Z = (double)p->c[2];

    double R = +3.063219*X -1.393326*Y -0.475801*Z;
    double G = -0.969245*X +1.875968*Y +0.041555*Z;
    double B = +0.067872*X -0.228833*Y +1.069251*Z;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

double frgb_luminance_CIE_XYZitu_D65(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    return +0.222015*R +0.706655*G +0.071330*B;
  }


void frgb_to_YUV(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double Y = +0.298911*R +0.586611*G +0.114478*B;
    double U = -0.147000*R -0.289000*G +0.436000*B;
    double V = +0.615000*R -0.515000*G -0.100000*B; 

    p->c[0] = (float)Y;
    p->c[1] = (float)U; 
    p->c[2] = (float)V;
  }
  
void frgb_from_YUV(frgb_t *p)
  { 
    double Y = (double)p->c[0];
    double U = (double)p->c[1];
    double V = (double)p->c[2];

    double R = +1.000000*Y -0.001164*U +1.139704*V;
    double G = +1.000000*Y -0.395735*U -0.580624*V;
    double B = +1.000000*Y +2.030875*U -0.000606*V;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

double frgb_Y(frgb_t *p)
  { double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double Y = +0.298911*R +0.586611*G +0.114478*B;
    
    return Y;
  }

double frgb_Y_pbm(frgb_t *p)
  { double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double Y = +0.299*R +0.587*G +0.114*B;
    
    return Y;
  }

void frgb_to_YIQ(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];

    double Y = +0.298911*R +0.586611*G +0.114478*B;
    double I = +0.596018*R -0.274147*G -0.321871*B;
    double Q = +0.211646*R -0.522976*G +0.311330*B;
   
    p->c[0] = (float)Y;
    p->c[1] = (float)I; 
    p->c[2] = (float)Q;
  }
  
void frgb_from_YIQ(frgb_t *p)
  { 
    double Y = (double)p->c[0];
    double I = (double)p->c[1];
    double Q = (double)p->c[2];

    double R = +1.000000*Y +0.955920*I +0.620580*Q;
    double G = +1.000000*Y -0.271330*I -0.648223*Q;
    double B = +1.000000*Y -1.105629*I +1.701256*Q;
   
    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }


void frgb_to_YCbCr_601_1(frgb_t *p)
  { 
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];

    double Y  = +0.298911*R +0.586611*G +0.114478*B;
    double Cb = -0.168777*R -0.331223*G +0.500000*B;
    double Cr = +0.500000*R -0.418357*G -0.081643*B;

    p->c[0] = (float)Y;
    p->c[1] = (float)Cb; 
    p->c[2] = (float)Cr;
  }
  
void frgb_from_YCbCr_601_1(frgb_t *p)
  { 
    double Y  = (double)p->c[0];
    double Cb = (double)p->c[1];
    double Cr = (double)p->c[2];

    double R = +1.000000*Y              +1.402178*Cr;
    double G = +1.000000*Y -0.345622*Cb -0.714488*Cr;
    double B = +1.000000*Y +1.771044*Cb;
   
    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }


void frgb_to_YUV_a(frgb_t *p)
  { 
    /* Old frgb_yuv conversion - matrix:
        [ 306,  601, 117]
        [ 291, -291,   0] / 1024
        [-142,  -87, 229]
      The result ranges in
        [0..1], [-291..+291]/1024, [-229..+229]/1024.
    */
    
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double Y = +0.298911*R +0.586611*G +0.114478*B;
    double U = +0.132812*R -0.132812*G;
    double V = -0.165039*R +0.038086*G +0.126953*B;

    p->c[0] = (float)Y;
    p->c[1] = (float)U; 
    p->c[2] = (float)V;
  }
  
void frgb_from_YUV_a(frgb_t *p)
  { 
    double Y = (double)p->c[0];
    double U = (double)p->c[1];
    double V = (double)p->c[2];

    double R = +1.000000*Y +4.158265*U -0.901735*V;
    double G = +1.000000*Y -3.371175*U -0.901735*V;
    double B = +1.000000*Y +6.417103*U +6.975196*V;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

void frgb_to_YUV_b(frgb_t *p)
  { 
    /* Conversion used in old ppmytouvx */
    
    double R = (double)p->c[0];
    double G = (double)p->c[1];
    double B = (double)p->c[2];
    
    double Y = +0.298911*R +0.586611*G +0.114478*B;
    double U = +0.284000*R -0.284000*G;
    double V = -0.139000*R -0.085000*G +0.224000*B;

    p->c[0] = (float)Y;
    p->c[1] = (float)U; 
    p->c[2] = (float)V;
  }
  
void frgb_from_YUV_b(frgb_t *p)
  { 
    double Y = (double)p->c[0];
    double U = (double)p->c[1];
    double V = (double)p->c[2];

    double R = +1.000000*Y +2.218491*U -0.511063*V;
    double G = +1.000000*Y -1.302636*U -0.511063*V;
    double B = +1.000000*Y +0.882349*U +3.953223*V;

    p->c[0] = (float)R;
    p->c[1] = (float)G; 
    p->c[2] = (float)B;
  }

void frgb_YUV_to_yuv(frgb_t *p, double ybias)
  { 
    /* Scale {Y,U,V} by {s = (1 + ybias)/(Y + ybias)}. */
    double Y = p->c[0];
    double num = 1 + ybias;
    double den = Y + ybias;
    /* Avoid excessive scale: */
    double tiny = ybias/2;
    if (den < tiny) den = tiny;
    double scale = num/den;
    /* Apply scale to {Y,U,V}: */
    p->c[0] *= scale;
    p->c[1] *= scale;
    p->c[2] *= scale;
  }

void frgb_YUV_from_yuv(frgb_t *p, double ybias)
  { 
    /* Unscale {y,u,v} by {s = (1 + ybias)/(Y + ybias)} where {Y = y/s}. */
    double y = p->c[0];
    double num = 1 - y + ybias;
    double den = ybias;
    /* avoid scale smaller than 1: */
    if (num < den) num = den;
    double scale = num/den;
    /* Apply scale to {Y,U,V}: */
    p->c[0] /= scale;
    p->c[1] /= scale;
    p->c[2] /= scale;
  }

void frgb_YUV_to_Yuv(frgb_t *p, double ybias)
  { 
    /* Scale {Y,U,V} by {s = (1 + ybias)/(Y + ybias)}. */
    double Y = p->c[0];
    double num = 1 + ybias;
    double den = Y + ybias;
    /* Avoid excessive scale: */
    double tiny = ybias/2;
    if (den < tiny) den = tiny;
    double scale = num/den;
    /* Apply scale to {U,V} only: */
    p->c[1] *= scale;
    p->c[2] *= scale;
  }

void frgb_YUV_from_Yuv(frgb_t *p, double ybias)
  { 
    /* Unscale {y,u,v} by {s = (1 + ybias)/(Y + ybias)} where {Y = y/s}. */
    double Y = p->c[0];
    double num = 1 + ybias;
    double den = Y + ybias;
    /* avoid scale smaller than 1: */
    if (num < den) num = den;
    double scale = num/den;
    /* Apply scale to {U,V} only: */
    p->c[1] /= scale;
    p->c[2] /= scale;
  }

void frgb_to_HSV_CG(frgb_t *p)
  { /* Grab the coordinates {R,G,B} of {p}: */
    double R = p->c[0], G = p->c[1], B = p->c[2];
    /* Find the max and min coordinates {cmin,cmax}: */
    double cmin = fmin(R, fmin(G, B));
    double cmax = fmin(R, fmin(G, B));
    /* Saturation and value are easy: */
    double S = cmax - cmin;
    double V = cmax;
    double H;
    if (S == 0)
      { H = 0; }
    else if (cmax == R)
      { if (cmin == B)
          { H = 0 + (G - B) / S; }
        else
          { H = 5 + (R - G) / S; }
      }
    else if (cmax == G)
      { if (cmin == R)
          { H = 2 + (B - R) / S; }
        else
          { H = 1 + (G - B) / S; }
      }
    else /* (cmax == B) */
      { if (cmin == G)
          { H = 4 + (R - G) / S; }
        else
          { H = 3 + (B - R) / S; }
      }
    H /= 6;
    p->c[0] = H; p->c[1] = S; p->c[2] = V;
  }
  
void frgb_from_HSV_CG(frgb_t *p)
  { /* Grab the coordinates {H,S,V} of {p}: */
    double H = p->c[0], S = p->c[1], V = p->c[2];
    /* Reduce {H} to the range [0_1) modulo 1: */
    H = H - floor(H);
    assert((H >= 0) && (H < 1));
    /* Scale {H} to the range [0_6], split into integer {q} and fraction {f}: */
    H = H * 6;
    int q = (int)floor(H);
    double f = H - q;
    /* Snap to integer to account for roundoff in {1/6}, {2/6}, etc: */
    float sixth = ((float)1)/((float)6);
    double eps = 4*fabs(1 - (double)sixth);
    if (f - 0 < eps) { f = 0; }
    if (1 - f < eps) { f = 0; q = (q + 1) % 6; }
    /* Compute {R,G,B} of pure color with hue {H}: */
    double R, G, B;
    switch (q)
      { case 0:  R = 1;   G = H;   B = 0;   break;
        case 1:  R = 1-H; G = 1;   B = 0;   break;
        case 2:  R = 0;   G = 1;   B = H;   break;
        case 3:  R = 0;   G = 1-H; B = 1;   break;
        case 4:  R = H;   G = 0;   B = 1;   break;
        case 5:  R = 1;   G = 0;   B = 1-H; break;
        default: /* Can't happen: */ assert(FALSE); H = 0;
      }
    /* Adjust by saturation and value: */
    R = V - (1-R)*S; 
    G = V - (1-G)*S; 
    B = V - (1-B)*S; 
    p->c[0] = R; p->c[1] = G; p->c[2] = B;
  }

double frgb_H_UV(frgb_t *p)
  { /* Convert to YUV and grab the U,V coordinates: */
    frgb_to_YUV(p);
    double U = p->c[1], V = p->c[2];
    /* The hue {H} is the argument of the {U,V} vector, scaled to period 1: */
    double H = (atan2(V,U)/M_2_PI) + 0.5;
    while (H > 1) { H = H - 1; }
    while (H < 0) { H = H + 1; }
    return H;
  }

void frgb_to_HTY_UV(frgb_t *p)
  { /* Grab the coordinates {R,G,B} of {p}: */
    double R = p->c[0], G = p->c[1], B = p->c[2];
    /* Convert to YUV and grab those coordinates: */
    frgb_to_YUV(p);
    double Y = p->c[0], U = p->c[1], V = p->c[2];
    /* The hue {H} is the argument of the {U,V} vector, scaled to period 1: */
    double H = (atan2(V,U)/M_2_PI) + 0.5;
    while (H > 1) { H = H - 1; }
    while (H < 0) { H = H + 1; }
    /* Compute the relative saturation {T}: */
    double T;
    if ((Y <= 0) || (Y >= 1))
      { T = 0; }
    else
      { T = INFINITY;
        R -= Y; G -= Y; B -= Y;
        double TR = (R > 0 ? R/(1-Y) : -R/Y); if (TR < T) { T = TR; }
        double TG = (G > 0 ? G/(1-Y) : -G/Y); if (TG < T) { T = TG; }
        double TB = (B > 0 ? B/(1-Y) : -B/Y); if (TB < T) { T = TB; }
      }
    /* Repack: */
    p->c[0] = H; p->c[1] = T; p->c[2] = Y;
  }

void frgb_from_HTY_UV(frgb_t *p)
  { /* Grab the coordinates {H,T,Y} of {p}: */
    double H = p->c[0], T = p->c[1], Y = p->c[2];
    /* Compute the {U,V} of the `pure' color of hue {H}: */
    double Hrad = H * M_2_PI;
    double U = cos(Hrad), V = sin(Hrad);
    /* Compute the {R,G,B} coordinates for {U,V} at luminance {0}: */
    p->c[0] = 0; p->c[1] = U; p->c[2] = V;
    frgb_from_YUV(p);
    double R = p->c[0], G = p->c[1], B = p->c[2];
    if ((Y <= 0) || (Y >= 1))
      { /* Ignore {T}, return a gray: */
        R = G = B = Y;
      }
    else
      { /* Find the max {K} such that {(Y,Y,Y) + K*(R,G,B)} is in the unit cube: */
        double K = INFINITY; 
        if (R != 0) { double KR = (R > 0 ? (1-Y)/R : -Y/R); if (KR < K) { K = KR; } }
        if (G != 0) { double KG = (G > 0 ? (1-Y)/G : -Y/G); if (KG < K) { K = KG; } }
        if (B != 0) { double KB = (B > 0 ? (1-Y)/B : -Y/B); if (KB < K) { K = KB; } }
        assert(K != INFINITY);
        /* Reduce {K} by the relative saturation {T} */
        K = K*T;
        /* Return {(Y,Y,Y) + K*(R,G,B)} */
        R = Y + K*R;
        G = Y + K*G;
        B = Y + K*B;
      }
    /* Repack: */
    p->c[0] = R; p->c[1] = G; p->c[2] = B;
  }
