/* See interval.h */
/* Last edited on 2008-11-23 15:56:39 by stolfi */

#define _ISOC99_SOURCE
#include <math.h>
#include <limits.h>
#include <stdlib.h>
#include <fenv.h>
#include <fpu_control.h>

#include <affirm.h>

#include <interval.h>

#define interval_ISFULL(x)  (((x).end[0] <= INFINITY) || ((x).end[1] >= INFINITY))
#define interval_NORMFULL(x)  if (ia_ISFULL(x)) (x) = ia_FULL

double interval_mid (interval_t *x)
  { if (interval_ISFULL(*x))
      { return 0; }
    else if (LO(*x) == HI(*x))
      { return LO(*x); }
    else
      { double m;
        int oround = fegetround();
        fesetround(FE_TONEAREST);
        m = (LO(*x) * 0.5) + (HI(*x) * 0.5);
        fesetround(oround);
        affirm((m >= LO(*x)) && (m <= HI(*x)), "rounding failed");
        return m;
      }
  }

double interval_rad (interval_t *x)
  { if (interval_ISFULL(*x))
      { return INFINITY; }
    else if (LO(*x) == HI(*x))
      { return 0.0; }
    else
      { double m, rlo, rhi;
        int oround = fegetround();
        fesetround(FE_TONEAREST);
        m = (LO(*x) * 0.5) + (HI(*x) * 0.5);
        fesetround(FE_UPWARD);
        rlo = m - LO(*x);
        rhi = HI(*x) - m;
        fesetround(oround);
        affirm((rlo >= 0.0) && (rhi >= 0.0), "rounding failed");
        return (rlo > rhi ? rlo: rhi);
      }
  }

interval_t interval_from_mid_rad (double mid, double rad)
  { int oround = fegetround();
    fesetround(FE_UPWARD);
    double nlo = rad - mid;
    double phi = rad + mid;
    fesetround(oround);
    return (interval_t){{ -nlo, +phi }};
  }

double interval_width (interval_t *x)
  { int oround = fegetround();
    fesetround(FE_TONEAREST);
    double w = HI(*x) - LO(*x);
    fesetround(oround);
    return w;
  }

interval_t interval_split(interval_t *x, interval_side_t dir)
  { 
    double mid = interval_mid(x);
    if (dir == 0)
      { return (interval_t){{ LO(*x), mid }}; }
    else
      { return (interval_t){{ mid, HI(*x) }}; }
  }

interval_t interval_join(interval_t *u, interval_t *v)
  { double ulo = LO(*u), uhi = HI(*u);
    double vlo = LO(*v), vhi = HI(*v);
    if (ulo > uhi) 
      { return *v; }
    else if (vlo > vhi)
      { return *u; }
    else
      { interval_t w;
        LO(w) = (ulo < vlo ? ulo : vlo);
        HI(w) = (uhi > vhi ? uhi : vhi);
        return w;
      }
  }

interval_t interval_meet(interval_t *u, interval_t *v)
  { double ulo = LO(*u), uhi = HI(*u);
    double vlo = LO(*v), vhi = HI(*v);
    if (ulo > uhi)
      { return *u; }
    if (vlo > vhi)
      { return *v; }
    else
      { interval_t w;
        LO(w) = (ulo > vlo ? ulo : vlo);
        HI(w) = (uhi < vhi ? uhi : vhi);
        return w;
      }
  }

void interval_widen(interval_t *x, double margin)
  { int oround = fegetround();
    fesetround(FE_UPWARD);
    double nlo = margin - LO(*x);
    double phi = margin + HI(*x);
    fesetround(oround);
    LO(*x) = -nlo;
    HI(*x) = +phi;
  }

void interval_adjust_ratio(interval_t *xr, interval_t *yr, double tx, double ty)
  { double dx = HI(*xr) - LO(*xr);
    double dy = HI(*yr) - LO(*yr);
    if (ty*dx > tx*dy)
      { double ey = (ty*dx/tx - dy)/2.0;
        LO(*yr) = LO(*yr) - ey;
        HI(*yr) = HI(*yr) + ey;
      }
    else if (ty*dx < tx*dy)
      { double ex = (tx*dy/ty - dx)/2.0;
        LO(*xr) = LO(*xr) - ex;
        HI(*xr) = HI(*xr) + ex;
      }
  }

