#ifndef interval_H
#define interval_H

/* Intervals with {double} endpoints. */
/* Last edited on 2008-11-23 15:51:46 by stolfi */ 

/* Should be merged with (or replaced by) {ia.h}. */
 
/* INTERVALS
  
  An {interval_t} {I} is a pair of {double}s, the /inferior/ or /low
  endpoint/ {LO(I) = I.end[0]} and the /superior/ or /high endpoint/
  {HI(I) = I.end[1]}. It usually stands for all real numbers {x} such
  that {LO(I) < x < HI(I)}. Depending on the context, the endpoints
  {LO(I)} and/or {HI(I)} may be included too.
  
  An open or half-open interval {I} is empty iff {LO(I) \geq HI(I)}. 
  A closed interval {I} is empty iff {LO(I) > HI(I)}. */
  
typedef struct interval_t { double end[2]; } interval_t;
   /* A real interval, open or closed. */
     
#define LO(x) ((x).end[0])
#define HI(x) ((x).end[1])
  /* Handy macros for the two endpoints of an interval. */

typedef enum { interval_BLO = 0, interval_BHI = 1 } interval_side_t;
  /* Binary direction along an axis : {BLO} towards {-oo}, {BHI}
    towards {+oo}. Used e. g. to identify the two halves or endpoints
    of an interval. */

double interval_mid (interval_t *x);
  /* Approximate midpoint of {x}, guaranteed to be in {x}.
    Finite as long as {x} is finite. */

double interval_rad (interval_t *x);
  /* Radius of {x} from its midpoint {m = ia_mid(x)}.
    I.e. a value {r} such that {[m-r _ m+r]} contains {x}.
    Finite as long as {x} is finite. */

interval_t interval_from_mid_rad (double mid, double rad);
  /* Returns an interval that includes {[mid-rad, mid+rad]}.
    If either endpoint overflows, sets it to infinity.*/

double interval_width (interval_t *x);
  /* Width of {x}, i.e. {HI(x) - LO(x)}, rounded up. Returns {+INF}
    if the subtraction overflows. */

interval_t interval_split(interval_t *x, interval_side_t dir);
  /* Returns the lower or upper half of {x}, depending on {dir}.
    The splitting point is {interval_mid(x)}, and is included
    in both halves. */

interval_t interval_join(interval_t *u, interval_t *v);
  /* Returns the smallest interval enclosing both {u} and {v}. */

interval_t interval_meet(interval_t *u, interval_t *v);
  /* Returns the intersection of {u} and {v}. */

void interval_widen(interval_t *x, double margin);
  /* Widens {*x} by the specified {margin} on both sides.
    The {margin} may be negative. */

void interval_adjust_ratio(interval_t *xr, interval_t *yr, double tx, double ty);
  /* Widens either {*xr} or {*yr}, as needed, to ensure that 
    their widths are in the ratio {tx:ty}. */

#endif
