/* See jsmath.h */
/* Last edited on 2008-09-29 05:42:08 by stolfi */

#define _GNU_SOURCE
#include <math.h>
#include <limits.h>
#include <assert.h>

#include <jsmath.h>

int64_t ipow(int64_t x, unsigned int y)
  { if (y == 0) { return 1; }
    if (x == 0) { return 0; }
    if (x == 1) { return 1; }
    /* Find largest power of 2 {b} not exceeding {y}: */ 
    unsigned int b = UINT_MAX/2 + 1;
    while (b > y) { b /= 2; }
    /* Compute result by binary method: */ 
    int64_t p = x;
    y -= b; b /= 2;
    while (b > 0)
      { p = p*p; 
        if (y >= b) { p *= x; y -= b; } 
        b /= 2;
      }
    return p;
  }

int64_t imod(int64_t x, int64_t y)
  { /* If {y} is negative, force a divide-by-zero: */
    if (y < 0) { y = 0; }
    x -= (x/y)*y;
    if (x < 0) { x += y; }
    return x;
  }

int64_t ifloor(int64_t x, int64_t y)
  { /* If {y} is negative, force a divide-by-zero: */
    if (y < 0) { y = 0; }
    int64_t z = (x/y)*y;
    if (z > x) { z -= y; }
    return z;
  }

int64_t iceil(int64_t x, int64_t y)
  { /* If {y} is negative, force a divide-by-zero: */
    if (y < 0) { y = 0; }
    int64_t z = -((-x)/y)*y;
    if (z < x) { z += y; }
    return z;
  }

uint64_t gcd(uint64_t a, uint64_t b)
  { while (b != 0)  { uint64_t r = a % b; a = b; b = r; }
    return a;
  }

uint64_t lcm(uint64_t a, uint64_t b)
  { uint64_t g = gcd(a, b);
    if (g == 0) { return 0; }
    /* The following math overflows only if overflow is unavoidable: */
    return ((uint64_t)a/g) * (uint64_t)b;
  }

int64_t imin(int64_t x, int64_t y)
  { return (x < y ? x : y); }

int64_t imax(int64_t x, int64_t y) 
  { return (x > y ? x : y); }
  
uint64_t umin(uint64_t x, uint64_t y)
  { return (x < y ? x : y); }

uint64_t umax(uint64_t x, uint64_t y) 
  { return (x > y ? x : y); }
  
#define LO32(x) (((uint64_t)(x)) & ((1LLU << 32) - 1LLU))
  /* The lowest 32 bits of an integer, as the lowest 32 bits of a 64-bit uint. */

#define HI32(x) (((uint64_t)(x)) >> 32)
  /* The highest 32 bits of an integer, as the lowest 32 bits of a 64-bit uint. */

void uint64_mul(uint64_t x, uint64_t y, uint64_t Z[])
  {
    uint32_t x0 = LO32(x), x1 = HI32(x);
    uint32_t y0 = LO32(y), y1 = HI32(y);
    
    uint64_t A00 = ((uint64_t)x0)*y0;
    uint64_t A01 = ((uint64_t)x0)*y1;
    uint64_t A10 = ((uint64_t)x1)*y0;
    uint64_t A11 = ((uint64_t)x1)*y1;
    
    Z[0] = A00 + ((A01 + A10) << 32);
    uint64_t P = HI32(A00) + LO32(A10) + LO32(A01);
    Z[1] = (P >> 32) + HI32(A10) + HI32(A01) + A11; /* Should not overflow */
  }

unsigned int digits(uint64_t x)
  { unsigned int d = 1;
    while (x > 9) { x /= 10; d++; }
    return d;
  }

double rel_diff(double x, double y)
  { double mx = fabs(x), my = fabs(y);
    double m = (mx > my ? mx : my);
    if (m == 0.0) 
      { return 0.0; }
    else
      { double rx = x/m, ry = y/m;
        return 0.5*(rx - ry)/sqrt(rx*rx + ry*ry);
      }
  }
  
double abs_rel_diff(double x, double y, double abs_tol, double rel_tol)
  { double x2 = x*x;
    double y2 = y*y;
    double D2 = abs_tol*abs_tol + rel_tol*rel_tol*(x2+y2)/2 + 1.0e-300;
    return (x - y)/sqrt(D2);
  }

