/* See pqueue.h */
/* Last edited on 2008-01-10 01:19:19 by stolfi */

#include <stdlib.h>
#include <assert.h>

#include <bool.h>
#include <affirm.h>

#include <pqueue.h>
#include <pqueue_rep.h>

/* !!! Consider combining this module with a simpler list {L} of
  entries, sharing storage with the heap queue {Q} but growing at the
  opposite end. Each item would be either in {Q} or in {L}, but not in
  both. The list {L} would not have the heap property, so insert and
  delete can be {O(1)}, and append can be {O(1)} and
  position-preserving. The pair {Q,L} would be useful for heapsort and
  Dikstra-type algorithms. */

/* INTERNAL PROTOS */

void pqueue_expand(pqueue_t *Q, pqueue_item_t z);
  /* Reallocates the {Q} vectors as needed to allow the insertion
    of item {z}. The size at least doubles, so that the 
    total alloc time is O(1). */

void pqueue_bubble_up(unsigned *itm,  double *val, unsigned *pos, unsigned i);
  /* Restores the heap invariant after a decrease in {Q->val[i]}. */

void pqueue_bubble_down(unsigned *itm, double *val, unsigned *pos, unsigned i, unsigned n);
  /* Restores the heap invariant after an increase in {Q->val[i]}. */

void pqueue_check(pqueue_t *Q);
  /* Checks whether {Q} satisfies the invariants. */

/* IMPLS */

pqueue_t *pqueue_new(void)
  { pqueue_t *Q = (pqueue_t *)notnull(malloc(sizeof(pqueue_t)), "out of memory");
    Q->order = +1;
    Q->n = 0;
    Q->itm = NULL; Q->val = NULL; Q->nmax = 0;
    Q->pos = NULL; Q->zlim = 0;
    return Q;
  }
  
void pqueue_set_order(pqueue_t *Q, int order)
  {
    demand(order != 0, "zero order");
    int old = Q->order;
    Q->order = (order < 0 ? -1 : +1);
    if (old != Q->order)
     { /* Negate all values and re-bubble all elements: */
       int n = Q->n;
       int i;
       for (i = 0; i < n; i++)
         { Q->val[i] = - Q->val[i];
           pqueue_bubble_up(Q->itm, Q->val, Q->pos, i);
         }
     }
  }

void pqueue_realloc(pqueue_t *Q, unsigned nmax, unsigned zlim)
  { 
    demand(Q->n == 0, "queue not empty");
    /* If we are using too much or too little storage, free it: */
    if ((Q->nmax < nmax) || (Q->nmax/2 >= nmax))
      { /* Free {itm,val}: */
        free(Q->itm); Q->itm = NULL; 
        free(Q->val); Q->val = NULL;
        Q->nmax = 0;
      }
    if ((Q->zlim < zlim) || (Q->zlim/2 >= zlim))
      { /* Free {pos}: */
        free(Q->pos); Q->pos = NULL;
        Q->zlim = 0;
      }
    /* If we are using too little, expand it: */
    if (Q->nmax < nmax)
      { demand(nmax <= pqueue_NMAX, "to many items");
        assert(Q->nmax == 0);
        Q->itm = (unsigned *)notnull(malloc(nmax*sizeof(unsigned)), "no mem"); 
        Q->val = (double *)notnull(malloc(nmax*sizeof(double)), "no mem"); 
        Q->nmax = nmax;
      }
    if (Q->zlim < zlim)
      { demand(zlim-1 <= pqueue_ITEM_MAX, "item too big");
        assert(Q->zlim == 0);
        Q->pos = (unsigned *)notnull(malloc(zlim*sizeof(unsigned)), "no mem");
      }
    Q->zlim = zlim;
  }
  
void pqueue_free(pqueue_t *Q)
  { pqueue_reset(Q); pqueue_realloc(Q,0,0); free(Q); }

/* QUERIES (WITHOUT SIDE EFFECTS) */

unsigned pqueue_count(pqueue_t *Q)
  { return Q->n; }

pqueue_item_t pqueue_head(pqueue_t *Q)
  { return Q->itm[0]; }

unsigned pqueue_position(pqueue_t *Q, pqueue_item_t z)
  { if ((z < 0) || (z >= Q->zlim)) { return Q->n; }
    unsigned i = Q->pos[z];
    if ((i > Q->n) || (Q->itm[i] != z)) { return Q->n; }
    return i;
  }

bool_t pqueue_has(pqueue_t *Q, pqueue_item_t z)
  { unsigned i = pqueue_position(Q, z);
    return (i < Q->n);
  }

pqueue_val_t pqueue_value(pqueue_t *Q, pqueue_item_t z)
  { unsigned i = pqueue_position(Q, z);
    demand(i < Q->n, "item not in queue");
    return Q->order * Q->val[i];
  }

pqueue_val_t pqueue_item(pqueue_t *Q, unsigned i)
  { demand(i < Q->n, "no such position in queue");
    return Q->itm[i];
  }

int pqueue_order(pqueue_t *Q)
  { return Q->order; }

/* MODIFYING */

void pqueue_expand(pqueue_t *Q, pqueue_item_t z)
  { if ((Q->itm == NULL) || (Q->n >= Q->nmax))
      { /* Reallocate {Q->itm,Q->val}: */
        unsigned nmax_new = 2*Q->nmax + 15;
        Q->itm = realloc(Q->itm, nmax_new*sizeof(unsigned));
        affirm(Q->itm != NULL, "out of mem");
        Q->val = realloc(Q->val, nmax_new*sizeof(double));
        affirm(Q->val != NULL, "out of mem");
        Q->nmax = nmax_new;
      }
    if ((Q->pos == NULL) || (z >= Q->zlim))
      { /* Reallocate {Q->pos}: */
        unsigned zlim_new = Q->zlim + z + 15;
        Q->pos = realloc(Q->pos, zlim_new*sizeof(unsigned));
        affirm(Q->pos != NULL, "out of mem");
        Q->zlim = zlim_new;
      }
  }

void pqueue_bubble_up(unsigned *itm, double *val, unsigned *pos, unsigned i)
  { unsigned tel = itm[i];
    double tva = val[i];
    unsigned k = i; /* Index of bubble. */
    unsigned j;     /* {j} is the parent of {k}. */
    /* fprintf(stderr, "bubble up %d:", i); */
    while ((k > 0) && (tva < val[(j = (k-1)/2)])) 
      { itm[k] = itm[j]; val[k] = val[j]; pos[itm[k]] = k; k = j; /* fprintf(stderr, " %d", k); */ }
    if (k != i) { itm[k] = tel; val[k] = tva; pos[tel] = k; }
    /* fprintf(stderr, "\n"); */
  }

void pqueue_bubble_down(unsigned *itm, double *val, unsigned *pos, unsigned i, unsigned n)
  { unsigned tel = itm[i];
    double tva = val[i];
    unsigned k = i; /* Index of bubble. */
    unsigned ja;    /* {ja} is the first child of {k}. */
    /* fprintf(stderr, "bubble down %d:", i); */
    while ((ja = 2*k + 1) < n)
      { /* Find smallest child {itm[j]} of {itm[k]}: */
        unsigned jb = ja + 1; /*  {itm[jb]} is the second child of {itm[k]}. */
        unsigned j = ((jb >= n) || (val[ja] <= val[jb]) ? ja : jb);
        if (tva <= val[j]) { break; }
        /* Promote smallest child into hole: */
        itm[k] = itm[j]; val[k] = val[j]; pos[itm[k]] = k;
        k = j; /* fprintf(stderr, " %d", k); */
      }
    if (k != i) { itm[k] = tel; val[k] = tva; pos[tel] = k; }
    /* fprintf(stderr, "\n"); */
  }

void pqueue_insert(pqueue_t *Q, pqueue_item_t z, pqueue_val_t v)
  { unsigned i = pqueue_position(Q, z);
    demand(i == Q->n, "item already in queue");
    pqueue_expand(Q, z);
    Q->itm[i] = z;
    Q->val[i] = Q->order * v;
    Q->pos[z] = i;
    Q->n++;
    pqueue_bubble_up(Q->itm, Q->val, Q->pos, i);
    /* pqueue_check(Q); */
  }

void pqueue_delete(pqueue_t *Q, pqueue_item_t z)
  { unsigned i = pqueue_position(Q, z);
    demand(i < Q->n, "item not in queue");
    
    unsigned n = Q->n;
    unsigned *itm = Q->itm;
    double *val = Q->val;
    unsigned *pos = Q->pos;
    
    /* Now slot {itm[i],val[i]} is vacant. */
    /* Promote children into vacancy {itm[i],val[i]} until {i} reaches the fringe: */
    unsigned ja; /* {itm[ja]} is the first child of {itm[i]}. */
    while ((ja = 2*i + 1) < n)
      { /* Find smallest child {itm[j]} of {itm[i]}: */
        unsigned jb = ja + 1; /*  {itm[jb]} is the second child of {itm[i]}. */
        unsigned j = ((jb >= n) || (val[ja] <= val[jb]) ? ja : jb);
        /* Promote smallest child into hole: */
        itm[i] = itm[j]; val[i] = val[j]; pos[itm[i]] = i;
        i = j;
      }
    /* One less element in queue: */
    n--;
    if (i < n)
      { /* The vacancy {itm[i]} did not end up at {itm[n]}, so fill it with {itm[n]}: */        
        unsigned tel = itm[n];
        double tva = val[n];
        unsigned j; 
        /* Bubble it up to the proper place: */
        while ((i > 0) && (tva < val[(j = (i-1)/2)]))
          { itm[i] = itm[j]; val[i] = val[j]; pos[itm[i]] = i;
            i = j;
          }
        itm[i] = tel; val[i] = tva; pos[itm[i]] = i;
      }
    Q->n = n;
    /* pqueue_check(Q); */
  }

void pqueue_set_value(pqueue_t *Q, pqueue_item_t z, pqueue_val_t v)
  { unsigned i = pqueue_position(Q, z);
    demand(i < Q->n, "item not in queue");
    v *= Q->order;
    double old_v = Q->val[i];
    Q->val[i] = v;
    if (v < old_v)
      { pqueue_bubble_up(Q->itm, Q->val, Q->pos, i); }
    else if (v > old_v)
      { pqueue_bubble_down(Q->itm, Q->val, Q->pos, i, Q->n); }
    /* pqueue_check(Q); */
  }

void pqueue_set_all_values(pqueue_t *Q, pqueue_value_function_t *f)
  { unsigned i;
    for (i = 0; i < Q->n; i++)
      { pqueue_item_t z = Q->itm[i];
        double v = Q->val[i]; 
        Q->val[i]= f(z, v);
        pqueue_bubble_up(Q->itm, Q->val, Q->pos, i);
      }
  }

void pqueue_reset(pqueue_t *Q)
  { Q->n = 0; }

void pqueue_check(pqueue_t *Q)
  {
    /* Validate {Q->n}: */
    assert((Q->n >= 0) && (Q->n <= Q->nmax)); 
    unsigned i;
    for (i = 0; i < Q->n; i++)
      { /* Get item {z} in slot {i}: */
        pqueue_item_t z = Q->itm[i];
        /* Items must be in range {0..Q->zlim-1}: */
        assert((z >= 0) && (z < Q->zlim)); 
        /* Get claimed position of item {z}: */
        unsigned k = Q->pos[z];
        /* Must be this position: */
        assert(k == i);
        /* Check heap invatiant: */
        if (i > 0)
          { int p = (k-1)/2; /* Parent of slot {k}. */
            assert(Q->order*pqueue_dblcmp(Q->val[p], Q->val[i]) >= 0);
          }
      }
  }

  
int pqueue_dblcmp(double x, double y) 
  { return (x < y ? -1 : ( x > y ? +1 : 0)); }
