/* See {sample_conv.h}. */
/* Last edited on 2008-10-18 19:14:16 by stolfi */

#define _GNU_SOURCE
#include <math.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>

#include <sample_conv.h>

#include <affirm.h>
#include <bool.h>

#define BT709_ENCODE_GAMMA 0.450
#define BT709_ENCODE_A     4.500
#define BT709_ENCODE_R     0.018053968511
#define BT709_ENCODE_B     0.099296826809
  /* Parameters for luminance encoding according to ITU-R Recommendation BT.709. */

float sample_conv_encode_BT709(float Y)
  { float a = fabsf(Y);
    if ((a == 0) || (a == 1)) { return Y; }
    if (a <= BT709_ENCODE_R)
      { a = a * BT709_ENCODE_A; }
    else
      { a = (1 + BT709_ENCODE_B)*pow(a, BT709_ENCODE_GAMMA) - BT709_ENCODE_B; }
    return (Y < 0 ? -a : a);
  }

#define BT709_DECODE_GAMMA 2.222222222222
#define BT709_DECODE_A     0.222222222222
#define BT709_DECODE_R     0.081242858299
#define BT709_DECODE_B     0.099296826809
  /* Parameters for luminance decoding according to ITU-R Recommendation BT.709. */

float sample_conv_decode_BT709(float V)
  { float a = fabsf(V);
    if ((a == 0) || (a == 1)) { return V; }
    if (a <= BT709_DECODE_R)
      { a = a * BT709_DECODE_A; }
    else
      { a = pow((a + BT709_DECODE_B)/(1 + BT709_DECODE_B), BT709_DECODE_GAMMA); }
    return (V < 0 ? -a : a);
  }

float sample_conv_gamma(float z, double gamma, double bias)
  { demand(bias >= 0, "negative bias not implemented yet");
    if (gamma == 1) { return z; }
    float a = fabsf(z);
    if ((a == 0) || (a == 1)) { return z; }
    if (bias == 0)
      { a = pow(a, gamma); }
    else
      { double sg = sqrt(gamma);
        double c = pow(bias, 1/sg);
        double d = pow(bias, sg);
        double u = a*(1 - c) + c;
        double v = pow(u, gamma);
        double w = (v - d)/(1 - d);
        a = w;
        if (a < 0) { a = 0; }
      }
    return (z < 0 ? -a : a);
  }

float sample_conv_log(float u, double uref, double logBase)
  {
    if ((! isfinite(logBase)) || (logBase == 0)) { return NAN; }
    if ((! isfinite(uref)) || (uref <= 0)) { return NAN; }
    if (u == +INFINITY) { return +INFINITY; }
    if ((! isfinite(u)) || (u < 0)) { return NAN; }
    if (u <= uref) { return 0.0; }
    double ulog = log(u/uref)/logBase;
    return (float)ulog;
  }

float sample_conv_interp(float z, int np, double U[], double V[])
  { if (np == 0) { return z; }
    float a = fabsf(z);
    /* Binary search consecutive indices {ilo,ihi} such that {U[ilo] <= a <= U[ihi]}. */
    /* Assume that {U[-1]==V[-1]==0} and {U[np]==V[np]==1}. */
    int ilo = -1, ihi = np;
    if (a <= 0) 
      { ihi = 0; }
    else if (a >= 1)
      { ilo = np-1; }
    else
      { while (ihi - ilo >= 2)
          { int imd = (ilo + ihi)/2; 
            /* assert((ilo < imd) && (imd < ihi)); */
            /* assert((0 <= imd) && (imd < np)); */
            if (a < U[imd])
              { ihi = imd; }
            else 
              { ilo = imd; }
          }
      }
    /* Get points {(ulo,vlo)} and {uhi,vhi)} that define the function for {a}: */
    double ulo, uhi, vlo, vhi;
    if (ilo < 0)   { ulo = vlo = 0; } else { ulo = U[ilo]; vlo = V[ilo]; }
    if (ihi >= np) { uhi = vhi = 1; } else { uhi = U[ihi]; vhi = V[ihi]; }
    
    /* Apply linear interpolation: */
    double s = (a - ulo)/(uhi - ulo); 
    float v = (1-s)*vlo + s*vhi;
    return (z < 0 ? -v : +v);
  }

float sample_conv_floatize
  ( sample_uint_t iv, 
    sample_uint_t maxval, 
    double lo,
    double hi, 
    sample_uint_t *imin, 
    sample_uint_t *imax, 
    float *vmin,
    float *vmax
  )
  /* Convert integer intensity to float, kep stats: */
  { if ((imin != NULL) && (iv < (*imin))) { (*imin) = iv; }
    if ((imax != NULL) && (iv > (*imax))) { (*imax) = iv; }
    double rv = ((double)iv + 0.5)/((double)maxval + 1.0);
    float fv = lo + rv*(hi - lo);
    if ((vmin != NULL) && (fv < (*vmin))) { (*vmin) = fv; }
    if ((vmax != NULL) && (fv > (*vmax))) { (*vmax) = fv; }
    return fv;
  }

void sample_conv_print_floatize_stats
  ( int iChan,           /* Channel index in input image. */
    int oChan,           /* Channel index in output image. */
    sample_uint_t imin,   /* Minimum integer sample seen. */
    sample_uint_t imax,   /* Maximum integer sample seen. */
    sample_uint_t maxval, /* Maximum possible integer sample. */
    double lo,           /* Low end of float scaling range. */
    double hi,           /* High end of float scaling range. */
    float vmin,          /* Minimum float sample seen. */
    float vmax           /* Maximum float sample seen. */
  )
  { fprintf(stderr, "  converted int channel %d to float channel %d:\n", iChan, oChan);
    fprintf(stderr, "    mapped [ 0 .. %u ] to [ %12.5e _ %12.5e ]\n", maxval, lo, hi);
    fprintf(stderr, "    actual input range  = [ %5u .. %5u ]\n", imin, imax);
    fprintf(stderr, "    actual output range = [ %12.5e _ %12.5e]\n", vmin, vmax);
  }

sample_uint_t sample_conv_quantize
  ( float fv, 
    sample_uint_t maxval, 
    double lo,
    double hi, 
    float *vmin,
    float *vmax, 
    int *clo,
    int *chi,
    sample_uint_t *imin, 
    sample_uint_t *imax
  )
  { demand(! isnan(fv), "{fv} is NaN"); 
    if ((vmin != NULL) && (fv < (*vmin))) { (*vmin) = fv; }
    if ((vmax != NULL) && (fv > (*vmax))) { (*vmax) = fv; }
    double fdelta = hi - lo;
    double rv = (fdelta == 0 ? 0.5 : (fv - lo)/fdelta);
    if (rv < 0.0) { rv = 0.0;  if (clo != NULL) { (*clo)++; } }
    if (rv > 1.0) { rv = 1.0;  if (chi != NULL) { (*chi)++; } }
    sample_uint_t zv = (rv == 1.0 ? maxval : (int)floor(rv*(maxval + 1)));
    demand((zv >= 0) && (zv <= (int)maxval), "bad {zv}");
    sample_uint_t iv = zv;
    if ((imin != NULL) && (iv < (*imin))) { (*imin) = iv; }
    if ((imax != NULL) && (iv > (*imax))) { (*imax) = iv; }
    return iv;
  }
  
void sample_conv_print_quantize_stats
  ( int iChan,           /* Channel index in input image. */
    int oChan,           /* Channel index in output image. */
    float vmin,          /* Minimum float sample seen. */
    float vmax,          /* Maximum float sample seen. */
    double lo,           /* Low end of float scaling range. */
    double hi,           /* High end of float scaling range. */
    int clo,             /* Number of samples seen below {lo}. */
    int chi,             /* Number of samples seen above {hi}. */
    sample_uint_t maxval, /* Maximum possible integer sample. */
    sample_uint_t imin,   /* Minimum integer sample seen. */
    sample_uint_t imax    /* Maximum integer sample seen. */
  )
  { fprintf(stderr, "  converted float channel %d to int channel %d:\n", iChan, oChan);
    fprintf(stderr, "    mapped [ %12.5e _ %12.5e ] to [ 0 .. %u ]\n", lo, hi, maxval);
    fprintf(stderr, "    actual input range  = [ %12.5e _ %12.5e ]\n", vmin, vmax);
    fprintf(stderr, "    actual output range = [ %5u .. %5u ]\n", imin, imax);
    if ((clo > 0) || (chi > 0)) 
      { fprintf(stderr, "    clipped pixels: too low = %d  too high = %d\n", clo, chi); }
  }

