#ifndef sample_conv_H
#define sample_conv_H

/* {sample_conv.h} - conversion between floating-point and integer samples. */
/* Last edited on 2008-09-23 14:37:21 by stolfi */

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>

typedef uint32_t sample_uint_t;
  /* A quantized sample value. */

float sample_conv_encode_BT709(float Y);
  /* Applies the luminance encoding (a.k.a. `gamma correction')
    according to ITU-R Recommendation BT.709. Namely, returns the
    sample value {V} that must be stored in an image in order to
    produce intensity {Y} on the idealized monitor assumed by BT.709.
    
    The encoding is monotonic for all {Y}, and maps {-1}, 0, and {+1}
    to themselves. It is approximately {V = Y^0.45} for positive {Y},
    and {V = -((-Y)^0.45)} for negative {Y}, but replaced by a linear
    function {V = 4.5*Y} near zero ({|Y| < 0.01805}). */

float sample_conv_decode_BT709(float V);
  /* The inverse of {sample_conv_encode_BT709}. Namely, returns the 
    intensity {Y} produced on the idealized monitor assumed by BT.709
    when displaying a sample value {V} taken from an image.
    
    The encoding is monotonic for all {V}, and maps {-1}, 0, and {+1}
    to themselves. It is approximately {Y = V^2.222} for positive {V},
    and {V = -((-V)^2.222)} for negative {V}, but replaced by a linear
    function {V = 0.2222*z} near zero ({|V| < 0.08124}). */

float sample_conv_gamma(float z, double gamma, double bias);
  /* Returns the result of applying the power correction with exponent
    {gamma} and offset {bias} to sample value {z}.
    
    If the offset {bias} is zero, the power correction is {z^gamma}.
    If the offset {bias} is positive, the procedure aplies affine
    corrections before and after this power correction, so that the
    slope at the origin is neither zero or infinity.
    
    In any case, the function maps {[0_1]} to {[0_1]}; and using
    exponent {1/gamma} instead of {gamma}, with the same offset
    {bias}, gives the inverse of this function.

    The exponent {gamma} and the offset {bias} must be positive. For
    negative samples, the procedure assumes {z^gamma = -(-z)^gamma},
    even if {gamma} is an even integer. */

float sample_conv_log(float u, double uref, double logBase);
  /* Converts {u} from linear to logarithmic scale, relative to the
    reference value {uref} and the base {exp(logBase)}. In particular,
    {logBase == 1} gives natural logarithms, {logBase == M_LOG2} gives
    result in octaves, {logBase == M_LOG10} gives result in decades, etc.
    
    More precisely, returns {log(u/uref)/logBase} if {u > 0}, {+INF}
    if {u == +INF}, {-INF} if {u} is zero, and {NAN} if {u} is
    negative or {NAN}. Requires {uref} to be finite and positive, and
    {logBase} to be finite and nonzero; otherwise returns {NAN} for
    any {u}. */
    
float sample_conv_interp(float u, int np, double U[], double V[]);
  /* Computes a piecewise affine function of {u} defined by the 
    nodal points {(0,0)}, {(U[i],V[i])} for {i = 0..np-1}, and {(1,1)}. 
    
    The values {U[0..np-1]} must be strictly increasing and 
    in {(0 _ 1)}, and ditto for {V[0..np-1]}.  The function 
    is anti-symmetric across zero, i.e. {f(u) = -f(-u)}. */

float sample_conv_floatize
  ( sample_uint_t iv, 
    sample_uint_t maxval, 
    double lo,
    double hi, 
    sample_uint_t *imin, 
    sample_uint_t *imax, 
    float *vmin,
    float *vmax
  );
  /* Converts an integer sample {iv} to float, mapping 
    affinely {[0..maxval+1]} to {[lo _ hi]}.  Also updates 
    the range {imin,imax} to enclose the input value {iv}, and 
    the range {vmin,vmax} to enclose the output value {fv}. */
    
void sample_conv_print_floatize_stats
  ( int iChan,           /* Channel index in input image. */
    int oChan,           /* Channel index in output image. */
    sample_uint_t imin,       /* Minimum integer sample seen. */
    sample_uint_t imax,       /* Maximum integer sample seen. */
    sample_uint_t maxval,     /* Maximum possible integer sample. */
    double lo,           /* Low end of float scaling range. */
    double hi,           /* High end of float scaling range. */
    float vmin,          /* Minimum float sample seen. */
    float vmax           /* Maximum float sample seen. */
  );
  /* Prints statistics for floatizing channel {iChan} of a PGM/PPM image
    into channel {oChan} of a float image. */

sample_uint_t sample_conv_quantize
  ( float fv, 
    sample_uint_t maxval, 
    double lo,
    double hi, 
    float *vmin,
    float *vmax, 
    int *clo,
    int *chi,
    sample_uint_t *imin, 
    sample_uint_t *imax
  );
  /* Converts a float sample {fv} to integer, mapping affinely 
    {[lo _ hi]} to {[0..maxval + 1)} with clipping. Also updates the range 
    {vmin,vmax} to enclose the input value {fv}, and the
    range {imin,imax} to enclose the output value {iv}. Increments 
    {clo} or {chi} if the input value was below {lo} or above {li}, 
    respectively. */

void sample_conv_print_quantize_stats
  ( int iChan,           /* Channel index in input image. */
    int oChan,           /* Channel index in output image. */
    float vmin,          /* Minimum float sample seen. */
    float vmax,          /* Maximum float sample seen. */
    double lo,           /* Low end of float scaling range. */
    double hi,           /* High end of float scaling range. */
    int clo,             /* Number of samples seen below {lo}. */
    int chi,             /* Number of samples seen above {hi}. */
    sample_uint_t maxval,     /* Maximum possible integer sample. */
    sample_uint_t imin,       /* Minimum integer sample seen. */
    sample_uint_t imax        /* Maximum integer sample seen. */
  );
  /* Prints statistics for quantizing channel {iChan} of a {sample_conv_t}
    into channel {oChan} of a PGM/PPM image. */

#endif
