#include <animacao.h>
#include <sys/stat.h>
#include <candidatos.h>
#include <gnuplot.h>
#include <unistd.h>
#include <time.h>

struct Plotter_{
  // Parametros gerais:
  int online;

  // Parametros para gravacao em arquivo:
  char *dir;
  char *subdir;

  // Parametros para plotagem on-line:
  gnuplot_t *gpl; // Processo gnuplot

  // Parametros do plot corrente:
  int iteracao;
  int nBoxes;
};

// PROCEDIMENTOS INTERNOS:

void writeLine(Plotter_t *plotter, char *which, float val);

// IMPLEMENTACOES:

Plotter_t *openPlotter(char *dir, char *subdir, int online)
{
  Plotter_t *plotter = malloc(sizeof(Plotter_t));
  plotter->online = online;
  plotter->dir = dir;
  plotter->subdir = subdir;
  // Por vai das duvidas:
  plotter->nBoxes = 0;
  plotter->iteracao = -1;
  if (online) 
    { 
      // Inicializa processo gnuplot:
      fprintf(stderr, "criando processo gnuplot para plot online...\n");
      plotter->gpl = gnuplot_start(640,480);
      //plotter->gpl = gnuplot_start(0,0);
      gnuplot_set_terminal(plotter->gpl, /*x11*/ 0);
      fprintf(stderr, "pronto.\n");
    }
  else
    {
      // Cria diretórios: 
      fprintf(stderr, "criando diretorios para plot offline...\n");
      makeDir(dir,subdir,"Animacao");
      makeDir(dir,subdir,"Animacao/rectangles");
      makeDir(dir,subdir,"Animacao/niveis");
      makeDir(dir,subdir,"Animacao/line_dmin");
      makeDir(dir,subdir,"Animacao/line_dmax");
      fprintf(stderr, "pronto.\n");
    }
  return plotter;
}

void closePlotter(Plotter_t *plotter)
{
   if (plotter->online)
    {
      // Encerra o processo gnuplot a fecha a janela
      gnuplot_stop(plotter->gpl);
    }
  else
    {  
      // Finaliza plot em disco:
      // (Nada a fazer)
    }
}

// ----------------------------------------------------------------------

void beginPlot(Plotter_t *plotter, int nBoxes, int iteracao)
{
  
  plotter->iteracao = iteracao;
  plotter->nBoxes = nBoxes;

  if (plotter->online)
    {
      // Inicializa plot no gnuplot:
      gnuplot_set_axis_labels(plotter->gpl, "i", "D", "Z");
      gnuplot_set_ranges(plotter->gpl, -1.0, nBoxes + 1.0,  -0.1, +1.3, -0.1, +1.1);
    }
  else
    {  
      // Inicializa plot em disco:
    }
}

void endPlot(Plotter_t *plotter)
{
   if (plotter->online)
    {
      // Inicializa plot no gnuplot:
      // (Nada a fazer)
    }
  else
    {  
      // Finaliza plot em disco:
      // (Nada a fazer)
    }
}

// ----------------------------------------------------------------------

void drawQueue(Plotter_t *plotter, candidato_t *prim, QFILA_t qual, int res_max, float dAstLo, float dAstHi)
{
  if (plotter->online)
    {
      // Desenha retangulos no gnuplot:

      char *filename=NULL;
      asprintf(&filename,"%s/%s/Animacao/plots/%07d.eps", plotter->dir, plotter->subdir, plotter->iteracao);
      char *title = NULL;
      asprintf(&title,"Iteration %07d", plotter->iteracao);
      

      float dLo[plotter->nBoxes];
      float dHi[plotter->nBoxes];
      float dMd[plotter->nBoxes];
      int res[plotter->nBoxes];
      candidato_t *aux = prim;
      int cont = 0;
      while ((aux != NULL) && (cont < plotter->nBoxes))
        { dLo[cont] = aux->distLo;
          dHi[cont] = aux->distHi;
          dMd[cont] = aux->distMd;
	  res[cont] = aux->k;
          aux = aux->prox[qual];
          cont++;
        }
      gnuplot_plot_interval_queue
        ( plotter->gpl, filename, dAstLo, dAstHi, cont, dLo, dHi, dMd, res_max, res, title );
      struct timespec req;
      if (plotter->iteracao == 0) 
        { req = (struct timespec){ 3, 0 }; }
      else if (cont <= 30)
        { req = (struct timespec){ 1, 00 }; }
      else 
        { req = (struct timespec){ 0, 100000000 }; }
      nanosleep(&req, NULL);
      free(filename);
      free(title);
    }
  else
    {  
      // Grava retangulos em disco:
    
      char *filename=NULL;
      asprintf(&filename,"%s/%s/Animacao/rectangles/%07d.txt", plotter->dir, plotter->subdir, plotter->iteracao);
      FILE *arq = fopen(filename,"w");
      if (arq == NULL) { fprintf(stderr, "** falhou abertura de '%s'\n", filename); assert(0); }

      candidato_t *aux = prim;
      int cont = 0;
      while ((aux != NULL) && (cont < plotter->nBoxes))
        {
          fprintf(arq,"%d  %f %f  %f\n", cont, aux->distLo, aux->distHi, aux->distMd);
          aux = aux->prox[qual];
          cont++;
        }
      fclose(arq);
      free(filename);
      writeLine(plotter, "dmin", dAstLo);
      writeLine(plotter, "dmax", dAstHi);
    }
}

void writeLine(Plotter_t *plotter, char *which, float val)
{
  char *filename = NULL;
  asprintf(&filename,"%s/%s/Animacao/line_%s/%07d.txt",plotter->dir,plotter->subdir,which,plotter->iteracao);
  FILE *arq = fopen(filename,"w");
  if (arq == NULL) { fprintf(stderr, "** falhou abertura de '%s'\n", filename); assert(0); }
  fprintf(arq,"%d  %f\n", -1, val);
  fprintf(arq,"%d  %f\n", plotter->nBoxes+1, val);
  fclose(arq);
  free(filename);
}

void makeDir(char *dir, char *subdir, char *subsubdir)
{
  char *pathname = NULL;
  if (strlen(subsubdir) == 0)
    { asprintf(&pathname,"%s/%s", dir, subdir); }
  else
    { asprintf(&pathname,"%s/%s/%s", dir, subdir, subsubdir); }
  mkdir(pathname, S_IRWXU+S_IXGRP+S_IXOTH);
  free(pathname);
}
