#ifndef GNUPLOT_H
#define GNUPLOT_H

/* Last edited on 2007-01-10 19:08:37 by stolfi */ 

/* Plots fields and meshes (using Gnuplot "splot"). */

typedef struct gnuplot_t gnuplot_t;
  /* A {gnuplot_t} object contains a connection to an active
    {gnuplot} process, which may plot data on the screen or
    write it to a file.

    An {gnuplot_t} object also contains several plot parameters, such as
    terminal type, plot styles, data ranges, etc.. The client can use
    the various {gnuplot_set_XXX} functions below to change those
    parameters.  Those changes will affect all subsequent {gnuplot_plot_XXX}
    calls, until overriden by other {gnuplot_set_XXX} calls. */

/* GNUPLOT PROCESS CONTROL  */

gnuplot_t *gnuplot_start(int hSize, int vSize);
  /* Starts a {gnuplot} process and returns a handle to it. 

    If the plotter opens any windows, they will have width {hSize} and
    height {vSize} (in pixels).

    If both parameters are zero, the {gnuplot} process is not started,
    and all plot commands will be written to a file called
    "gnuplot.plt". */

void gnuplot_stop(gnuplot_t *wp);
  /* Kills the {gnuplot} process attached to {wp}, and reclaims
    its storage. */

/* OUTPUT DEVICE TYPE */

void gnuplot_set_terminal(gnuplot_t *wp, int termKind);
  /* This function sets gnuplot's ``terminal'' (output device type),
    according to {termKind}: 0 -> x11, 10 -> eps grayscale, 
    11 -> eps color, 101 -> png grayscale, 111 -> png color. */

/* AXIS LABELS AND RANGES */

void gnuplot_set_axis_labels(gnuplot_t *wp, char *xLabel, char *yLabel, char *zLabel);
  /* This function specifies the labels for the X, Y, and Z axes
    of the plot. If any of them is NULL, the corresponding axis
    is omitted. */
    
void gnuplot_set_ranges
  ( gnuplot_t *wp, 
    double xMin, double xMax, 
    double yMin, double yMax,
    double zMin, double zMax
  );
  /* This function sets the data ranges, that wil define the plot
    scales for subsequent plot commands.
    
    If any of these parameters is NaN, the corresponding limit will be
    left uspecified, so that {gnuplot} will choose it based on the
    data values to be provided. */

/* STYLE OPTIONS FOR 3D GRAPHS */

void gnuplot_set_3dgraph_palette(gnuplot_t *wp, int palette);
  /* This function sets the color palette for 3D graphs of bivariate
    functions, according to the {palette} parameter.
    Palette 0 is the default. */

void gnuplot_set_3dgraph_style(gnuplot_t *wp, int style);
  /* This function sets the {gnuplot} options that are relevant for 3D
    graphs of bivariate functions, according to the {style} parameter:
    
       0 -> graph as a surface in 3D, with hidden parts removed.
      10 -> graph as a ???.
      20 -> graph as a ???.  */

/* STYLE OPTIONS FOR GRIDMAP PLOTS */

void gnuplot_set_gridmap_palette(gnuplot_t *wp, int palette);
  /* This function sets the color palette for 2D grid-map
    plots of bivariate functions, according to the {palette} 
    parameter. Palette 0 is the default. */
 
void gnuplot_set_gridmap_style(gnuplot_t *wp, int style);
  /* This function sets the {gnuplot} options that are relevant for 2D
    grid-map plots of bivariate functions, according to the {style}
    parameter: 0 -> grid of uniformly colored squares. */
   
/* PLOTTING COMMANDS

  The commands {gnuplot_plot_XXX} functions below will 
  sed the data to {gnuplot}, and cause it to generate
  the desired plot (displayed or written to disk). */

void gnuplot_plot_interval_queue
  ( gnuplot_t *wp,
    char *fileName,
    float dAstLo,
    float dAstHi,
    int n, 
    float dLo[], 
    float dHi[], 
    float dMd[], 
    int maxLevel, 
    int level[], 
    char *title
    );
  /* This function makes a 2D plot showing a list of intervals
    {(dLo[i],dHi[i])} for {i} from 1 to {n-1}.  Each interval is
    plotted as an error bar with a dot at {dMd[i]}. Also plots the
    relative level {level[i]/maxLevel} of each interval as a dot over
    it. Also plots two horizontal lines at ordinates {dAstLo} and
    {dAstHi}.
    
    The {fileName} is used only if the current terminal type
    requires it.  If the current terminal is a window
    (e.g., type "x11"), the {fileName} is ignored. */

void gnuplot_plot_3dgraph
  ( gnuplot_t *wp,            
    char *fileName,  
    int nx,        /* Number of rows of {A} */
    int ny,        /* Number of cols of {A} */
    double dx,     /* Sampling step in X. */
    double dy,     /* Sampling step in Y. */
    double **A,    /* {A[ix][iy]} is the field value at {(ix*dx,iy*dy)}. */
    char* title    /* Name of the function being plotted. */
  );
  /* This function plots the 3D fraph of a bivariate function
    {f(x,y)}, given as an array {A[0..nx][0..ny]} of sample values.
    The output device and plot style options should have been set
    previosly with the {gnuplot_set_XXX} functions.
    
    The array {A} is assumed to have {nx+1} rows, each with {ny+1}
    elements {A[ix][iy]}. The first index {ix} increases along X, the
    second {iy} increases along Y. The parameters {dx} and {dy} define
    the spacing of the elements along each axis. Thus, the appropriate
    plotting ranges are {[-0.5*dx : (nx+0.5)*dx]} fo X, and similarly
    for Y.
    
    The {fileName} is used only if the current terminal type
    requires it.  If the current terminal is a window
    (e.g., type "x11"), the {fileName} is ignored. */
 
void gnuplot_plot_gridmap
  ( gnuplot_t *wp,            
    char *fileName,  
    int nx,        /* Number of rows of {A} */
    int ny,        /* Number of cols of {A} */
    double **A,    /* {A[ix][iy]} is the field value at {(ix*dx,iy*dy)}. */
    char* title    /* Name of the function being plotted. */
  );
  /* This function plots the elements of array {A[0..nx][0..ny]} as a
    2D grid of colored squares. The output device and plot style
    options should have been set previosly with the {gnuplot_set_XXX}
    functions.
    
    The array {A} is assumed to have {nx+1} rows, each with {ny+1}
    elements {A[ix][iy]}. The first index {ix} increases along X, the
    second {iy} increases along Y.
    
    This command ignores any X or Y ranges previously specified 
    with {gnuplot_set_ranges}, and uses {[0:nx+1]} and {[0:ny+1]} 
    instead.  The previously defined Z range is honored.
    
    The {fileName} is used only if the current terminal type
    requires it.  If the current terminal is a window
    (e.g., type "x11"), the {fileName} is ignored. */
 
#endif
