#define _GNU_SOURCE
#include "vetorn.h"
#include "normais.h"
#include "tabela.h"
#include <affirm.h>
#include <float_image.h>
#include <float_image_masked.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <float.h>
#include <r2.h>
#include <r3.h>
#include <rn.h>
#include <r3x3.h>

#define MINMAG 0.01

struct LinhaTabela{
  r3_t normal;      // normal - normal da superfície.
  double *intdir;   // intdir[num_luzes] - assinatura normalizada
  double intmag;    // intmag norma euclidiana da assinatura
};

struct Tabela{
  int num_linhas;
  int num_luzes;
  r3_t view_dir; //stores the camera direction as seen from the center of the gauge
  LinhaTabela* tabela;
};


r3_t calcula_normal_esfera(r2_t *uv)
  { double u = uv->c[0];
    double v = uv->c[1];
    /* Check if inside unit {u,v} disk: */
    double uv2 = u*u + v*v;
    if (uv2 >= 0.999999)
      { /* Outside - return null normal: */
        return (r3_t) {{ 0.0, 0.0, 0.0 }};
      }
    else
      { /* Inside - compute {w} coordinate: */
        double w = sqrt(1.000001 - uv2);
        /* The normal is the points's position: */
        return (r3_t) {{ u, v, w }};
      }
  }

void gera_pontos_no_gabarito_old(int resolucao, r2_t centro, double raio, r2_t **pontoP, r3_t **normalP, int *num_pontosP)
{
  assert(sizeof(int) >= 4*sizeof(char));

  fprintf(stderr,  "Gerando pontos no gabarito\n");
  fprintf(stderr,  "  Resolução: %d\n", resolucao);
  fprintf(stderr,  "  Centro: %f %f\n", centro.c[0], centro.c[1]);
  fprintf(stderr,  "  Raio: %f\n", raio);
  fprintf(stderr,  "\n");

  r2_t *ponto = NULL;
  r3_t *normal = NULL; 
  int np = 0;
  int passo;
  for (passo = 0; passo < 2; passo++) {
    /* Gera pontos no disco unitário, em {M} círculos concentricos. */
    int kp = 0;
    int kr, kt;
    for (kr = 0; kr < resolucao; kr++){
      /* Escolhe o raio do círculo número {kr}: */
      /* double r = ((double) kr)/resolucao;  Igualmente espaçados em X,Y */
      /* Igualmente espaçados na esfera. */
      /* Escolhe o número de pontos {N} nesse círculo. Note que {N=1} quando {kr=0}: */
      int N = N = floor((M_PI*kr)) +1;
      if (ponto == NULL) {
        np += N; 
      } else {
      	for (kt = 0; kt < N; kt++){
      	  /* Escolhe ângulo {t} no círculo, com deslocamento alternado: */
	  double r = (kr)/(double)resolucao;
	  double t = (2*M_PI*kt)/N;
      	      	  /* Calcula normal {dx,dy,dz} da esfera nesse ponto: */
      	  double dx = r*cos(t);
      	  double dy = r*sin(t);
      	  double dz = sqrt(1 - (dx*dx) - (dy*dy));
      	  /* Calcula coordenadas {dx,dy} do ponto no disco unitário: */
      	  double qX = centro.c[0] + raio*dx;
      	  double qY = centro.c[1] + raio*dy;
      	  ponto[kp] = (r2_t){{qX, qY}}; 
      	  normal[kp] = (r3_t){{dx, dy, dz}}; 
      	  fprintf(stderr, "%d %6.3lf %6.3lf\n",kp,qX,qY);
      	  kp++;
      	} 
      }
    }
    if (ponto == NULL) {
      ponto = (r2_t *)malloc(np*sizeof(r2_t)); 
      normal = (r3_t *)malloc(np*sizeof(r3_t)); 
    } else {
      assert(kp == np); 
    }
  }
  fprintf(stderr,  "  Num pontos: %d\n", np);
  (*pontoP) = ponto;
  (*normalP) = normal;
  (*num_pontosP) = np;
}

r3_t compute_view_dir_from_stretch(r2_t gauge_stretch, double radius){
	r3_t view_dir;
	double m = r2_norm(&gauge_stretch);
	if(m != 0 ){
		double cR = radius/(radius + m);
  		double sR = sqrt(1 - cR*cR);
		view_dir = (r3_t){{ (gauge_stretch.c[0]*sR)/m, -(gauge_stretch.c[1]*sR)/m , cR}};
		(void)r3_dir(&view_dir,&view_dir);//just in case...
		
	}else{
		view_dir = (r3_t){{ 0,0,1}};
	}
	
	return view_dir;

}

r3x3_t compute_normal_correction_matrix(r3_t view_dir){
	r3x3_t roda_normal;
	double sR = hypot(view_dir.c[0],view_dir.c[1]);
	double cR = view_dir.c[2];
	if(sR != 0) {
		
		r3x3_t A,R;
		double ux = view_dir.c[0]/sR;
		double uy = view_dir.c[1]/sR;
	
	
		A = (r3x3_t){
			{
			{ux,0.0,-uy},
			{uy, 0, ux},
			{0.0,1.0,0.0}
			}
		};
		fprintf(stderr,"A\n");
		r3x3_print(stderr,&A);
		fprintf(stderr,"Determinant = %lf \n",r3x3_det(&A));
		r3x3_t temp;
		r3x3_mul_tr(&A,&A,&temp);
		fprintf(stderr,"A*Atr\n");
		r3x3_print(stderr,&temp);
		R = (r3x3_t){
			{
				{cR,-sR, 0.0},
				{sR,cR,0.0},
				{0.0,0.0,1.0}
			}
		};
		r3x3_mul(&A,&R,&roda_normal);
		r3x3_mul_tr(&roda_normal,&A,&roda_normal);
		fprintf(stderr,"Roda Normal\n");
		r3x3_print(stderr,&roda_normal);
		fprintf(stderr,"Determinant = %lf \n",r3x3_det(&roda_normal));
		r3x3_mul_tr(&roda_normal,&roda_normal,&temp);
		fprintf(stderr,"M*Mtr\n");
		r3x3_print(stderr,&temp);
	}else {
		r3x3_ident(&roda_normal);
	}
	return roda_normal;
}

void gera_normais_na_esfera_uniforme(int resolucao, r3_t** normalP, int* num_pontosP, r3_t view_dir){
	double step = (M_PI*0.5)/((double) resolucao);
	int k;
	int np = 0;
	r3_t* normal = NULL;
	int passo;
	int kp = 0;
	for(passo = 0; passo < 2; passo++){
		for(k = 0; k < resolucao; k++){
			double theta = (M_PI*0.5)*(k/(double)resolucao);
			double r = sin(theta);
			double N = (2.0*M_PI*r)/step;
			int i;
			if(normal == NULL){
				np+= N;
			}else{
				for(i = 0; i < (N-1); i++){
					double phi = (2.0*M_PI*i)/N;
					//double phi = (M_PI*(2*i + (k%2)))/N;
					double x,y,z;
					x = r*cos(phi);
					y = r*sin(phi);
					z = cos(theta);
					r3_t normal_bruta = (r3_t){{ x,y,z }};
					normal[kp] = normal_bruta;
					r3_print(stderr,&normal_bruta);
					fprintf(stderr,"\n");
					kp++;
				}
			}
		}
		if(normal == NULL){
			normal = (r3_t*)malloc(sizeof(r3_t)*np);
		}
	}
	*normalP = normal;
	*num_pontosP = np;
}

void gera_normais_no_gabarito(int resolucao, r3_t** normalP, int* num_pontosP, r3_t view_dir){
  fprintf(stderr,  "Gerando normais no gabarito\n");
  fprintf(stderr,  "  Resolução: %d\n", resolucao);
  fprintf(stderr,  "\n");

  r3_t *normal = NULL; 
  int np = 0;
  int passo;
  
   r3x3_t roda_normal = compute_normal_correction_matrix(view_dir);
   for (passo = 0; passo < 2; passo++) {
    	/* Gera pontos no disco unitário, em {M} círculos concentricos. */
    	int kp = 0;
    	int kr, kt;
    	for (kr = 0; kr < resolucao; kr++){
      		/* Escolhe o raio do círculo número {kr}: */
      		/* double r = ((double) kr)/resolucao;  Igualmente espaçados em X,Y */
      		double r = sin(M_PI/2*((double) kr)/resolucao); /* Igualmente espaçados na esfera. */
      		/* Escolhe o número de pontos {N} nesse círculo. Note que {N=1} quando {kr=0}: */
      		int N = (int)floor(M_PI*r*resolucao) + 1;
      		if (normal == NULL) {
	        	np += N; 
      		}else {
      			for (kt = 0; kt < N; kt++){
      	  			/* Escolhe ângulo {t} no círculo, com deslocamento alternado: */
      	  			double t = (M_PI*(2*kt + (kr%2)))/N;
      	   			/* Calcula a posição {uv} relativa ao círculo unitário */
      	  			r2_t uv;
	  			uv.c[0] = r*cos(t);
      	  			uv.c[1] = r*sin(t);
      	  			r3_t normal_bruta = calcula_normal_esfera(&uv);
      	  			
				r3x3_map_row(&(normal_bruta),&roda_normal,&(normal[kp]));
				double dot = r3_dot(&view_dir,&normal[kp] );
				if(dot < -1.0e-6) {
					fprintf(stderr,"entry %d\n",kp);
					fprintf(stderr,"normal_bruta = "); r3_print(stderr,&normal_bruta); fprintf(stderr,"\n");
					fprintf(stderr,"normal_ajustada = "); r3_print(stderr,&normal[kp]); fprintf(stderr,"\n");
					assert(FALSE);
				}
      	       			//  fprintf(stderr, "%d %6.3lf %6.3lf\n",kp,qX,qY);
				kp++;
      			} 
		}
    	}
    	if (normal == NULL) {
      		normal = (r3_t *)malloc(np*sizeof(r3_t)); 
    	} else {
      		assert(kp == np); 
    	}
   }

  fprintf(stderr,  "  Num normais: %d\n", np);
  (*normalP) = normal;
  (*num_pontosP) = np;
	
}

void gera_pontos_no_gabarito_eliptico(int resolucao, r2_t centro, double raio, r2_t estica, r2_t **pontoP, r3_t **normalP, int *num_pontosP,r3_t view_dir){
	
  assert(sizeof(int) >= 4*sizeof(char));

  fprintf(stderr,  "Gerando pontos no gabarito eliptico\n");
  fprintf(stderr,  "  Resolução: %d\n", resolucao);
  fprintf(stderr,  "  Centro: %f %f\n", centro.c[0], centro.c[1]);
  fprintf(stderr,  "  Raio: %f\n", raio);
  fprintf(stderr,  "  Estica: %f %f\n", estica.c[0], estica.c[1]);
  fprintf(stderr,  "\n");

  r2_t *ponto = NULL;
  r3_t *normal = NULL; 
  int np = 0;
  
  r2_t dir_estica;
  double en = r2_norm(&estica);
  fprintf(stderr,"EN = %24.16e\n",en);
  if((estica.c[0] == 0.0) && (estica.c[1] == 0.0)){
	dir_estica = (r2_t){{1.0,0.0}};
  }
  else{
	r2_dir(&estica,&dir_estica);
  }

  gera_normais_no_gabarito(resolucao,&normal,&np,view_dir);
  ponto = (r2_t *)malloc(np*sizeof(r2_t)); 
  int k;
  for(k = 0; k < np; k++){
	r3_t uvw = normal[k];
	r2_t uv;
	uv.c[0] = uvw.c[0];
      	uv.c[1] = uvw.c[1];
	/* Calcula o fator de esticamento (b) */
	double b = r2_dot(&uv,&dir_estica);
	//calcula a posição q do ponto no gabarito
      	r2_t q;
	q.c[0] = (raio*uv.c[0] + b*estica.c[0]) + centro.c[0];
	q.c[1] = (raio*uv.c[1] + b*estica.c[1]) + centro.c[1];
      	ponto[k] = q; 
  }

  fprintf(stderr,  " Num pontos: %d\n", np);
  (*pontoP) = ponto;
  (*normalP) = normal;
  (*num_pontosP) = np;

		
}


void gera_pontos_com_mascara(float_image_t* mask,r2_t centro, double raio, r2_t estica, r2_t **pontoP, r3_t **normalP, int *num_pontosP,r3_t view_dir){
	
  assert(sizeof(int) >= 4*sizeof(char));

  fprintf(stderr,  "Gerando pontos no gabarito com mascara\n");
  fprintf(stderr,  "  Centro: %f %f\n", centro.c[0], centro.c[1]);
  fprintf(stderr,  "  Raio: %f\n", raio);
  fprintf(stderr,  "  Estica: %f %f\n", estica.c[0], estica.c[1]);
  fprintf(stderr,  "\n");

 
  r2_t dir_estica;
  double compr_estica = r2_norm(&estica);
  fprintf(stderr,"COMPR_ESTICA = %24.16e\n",compr_estica);
  if((estica.c[0] == 0.0) && (estica.c[1] == 0.0)){
	dir_estica = (r2_t){{1.0,0.0}};
  }
  else{
	r2_dir(&estica,&dir_estica);
  }

  r2_t dir_perp = (r2_t){{-dir_estica.c[1],dir_estica.c[0]}};

  r3x3_t roda_normal = compute_normal_correction_matrix(view_dir);
 /*conta quantos pontos a imagem possui dentro da máscara*/
  int x,y;
  
  demand(mask->sz[0] == 1,"Mask File with more than 1 channel");
  int NX = mask->sz[1];
  int NY = mask->sz[2];
   r2_t * ponto = NULL;
  r3_t * normal = NULL;
  int passo;
  int np = 0;
  for(passo = 0; passo < 2; passo++){
	if(passo == 1){
		ponto = (r2_t *)malloc(np*sizeof(r2_t)); 
		normal = (r3_t *)malloc(np*sizeof(r3_t));
	}
	int kp = 0;	
	bool_t debug  = FALSE;
	for(y = 0; y < NY; y++){
		for(x = 0; x < NX; x++){
			if(float_image_get_sample(mask,0,x,y) != 0){
				r2_t p = (r2_t){{x+0.5,y+0.5}};
				r2_t q; 
				r2_sub(&p,&(centro),&q);
				double u = r2_dot(&dir_estica,&q)/(raio + compr_estica);
				double v = r2_dot(&dir_perp,&q)/(raio);
				r2_t nxy;
				r2_mix(u,&(dir_estica),v,&(dir_perp),&(nxy));
				double nxy2 = r2_norm_sqr(&nxy);
				assert(nxy2 <= (1.00001+(1.42/raio)) );
				if( nxy2 <  1.00 ){
					double nz = sqrt( 1.0 - fmin(nxy2, 1.0));
					r3_t normal_bruta = (r3_t){{ nxy.c[0],nxy.c[1], nz}};
					(void)r3_dir(&normal_bruta,&normal_bruta);
					if(debug){
						fprintf(stderr,"x=%d y=%d ",x,y);
						fprintf(stderr,"NXY= %9.6f %9.6f ",nxy.c[0],nxy.c[1]);
						fprintf(stderr,"ABS= %9.6f ",sqrt(nxy2));
						r3_print(stderr,&normal_bruta);
						fprintf(stderr,"\n");
					
						debug = FALSE;
					}
					r3_t normal_rodada;
					r3x3_map_row(&(normal_bruta),&roda_normal,&(normal_rodada));
					if(passo == 0){ np++;}
					else{
						ponto[kp] = p;
						normal[kp] = normal_rodada;
						kp++;
					}
				}
			}
		}
	}
  }

  fprintf(stderr,  "  Num pontos: %d\n", np);
  (*pontoP) = ponto;
  (*normalP) = normal;
  (*num_pontosP) = np;

		
}

void gera_pontos_no_gabarito(int resolucao, r2_t centro, double raio, r2_t **pontoP, r3_t **normalP, int *num_pontosP)
{
  assert(sizeof(int) >= 4*sizeof(char));

  fprintf(stderr,  "Gerando pontos no gabarito\n");
  fprintf(stderr,  "  Resolução: %d\n", resolucao);
  fprintf(stderr,  "  Centro: %f %f\n", centro.c[0], centro.c[1]);
  fprintf(stderr,  "  Raio: %f\n", raio);
  fprintf(stderr,  "\n");

  r2_t *ponto = NULL;
  r3_t *normal = NULL; 
  int np = 0;
  int passo;
  for (passo = 0; passo < 2; passo++) {
    /* Gera pontos no disco unitário, em {M} círculos concentricos. */
    int kp = 0;
    int kr, kt;
    for (kr = 0; kr < resolucao; kr++){
      /* Escolhe o raio do círculo número {kr}: */
      /* double r = ((double) kr)/resolucao;  Igualmente espaçados em X,Y */
      double r = sin(M_PI/2*((double) kr)/resolucao); /* Igualmente espaçados na esfera. */
      /* Escolhe o número de pontos {N} nesse círculo. Note que {N=1} quando {kr=0}: */
      int N = (int)floor(M_PI*r*resolucao) + 1;
      if (ponto == NULL) {
        np += N; 
      } else {
      	for (kt = 0; kt < N; kt++){
      	  /* Escolhe ângulo {t} no círculo, com deslocamento alternado: */
      	  double t = (M_PI*(2*kt + (kr%2)))/N;
      	  /* Calcula normal {dx,dy,dz} da esfera nesse ponto: */
      	  double dx = r*cos(t);
      	  double dy = r*sin(t);
      	  double dz = sqrt(1 - (dx*dx) - (dy*dy));
      	  /* Calcula coordenadas {dx,dy} do ponto no disco unitário: */
      	  double qX = centro.c[0] + raio*dx;
      	  double qY = centro.c[1] + raio*dy;
      	  ponto[kp] = (r2_t){{qX, qY}}; 
      	  normal[kp] = (r3_t){{dx, dy, dz}}; 
      	//  fprintf(stderr, "%d %6.3lf %6.3lf\n",kp,qX,qY);
      	  kp++;
      	} 
      }
    }
    if (ponto == NULL) {
      ponto = (r2_t *)malloc(np*sizeof(r2_t)); 
      normal = (r3_t *)malloc(np*sizeof(r3_t)); 
    } else {
      assert(kp == np); 
    }
  }
  fprintf(stderr,  "  Num pontos: %d\n", np);
  (*pontoP) = ponto;
  (*normalP) = normal;
  (*num_pontosP) = np;
}

int ponto_dentro_do_gabarito(int radius, r2_t stretch,r2_t dp);
int ponto_dentro_do_gabarito(int radius, r2_t stretch,r2_t dp){
	r2_t e; //direção do stretch
  	double en = r2_norm(&stretch);
  	if(en == 0.0){
		e = (r2_t){{1.0,0.0}};
  	}
  	else{
		r2_dir(&stretch,&e);
  	}
	
	r2_t f = (r2_t){{-e.c[1],e.c[0]}}; //perpendicular ao {e}
	double dpe = r2_dot(&dp,&e)/radius;
	double dpf = r2_dot(&dp,&f)/(radius + en);
	 
	return ((dpe*dpe + dpf*dpf) <= 1);
	
  
}

float_image_t  *gera_mascara_do_gabarito(int nx, int ny, r2_t centro, double raio,r2_t stretch)
{
  float_image_t  *M = float_image_new(1, nx, ny);
  int x, y;
  for (x = 0; x < nx; x++) {
    for (y = 0; y < ny; y++) {
      /* Determina o centro {px,py} do pixel {x,y}: */
      double px = x + 0.5; 
      double py = y + 0.5;
      /* Determina o ponto {px,py} do pixel que está mais longe do centro do gabarito: */
      if (px < centro.c[0]) { px -= 0.5; } else { px += 0.5; }
      if (py < centro.c[1]) { py -= 0.5; } else { py += 0.5; }
      /* Decide se o pont está dentro do gabarito: */
      double dx = px - centro.c[0];
      double dy = py - centro.c[1];
      //int dentro = (dx*dx + dy*dy <= raio*raio);
      int dentro = ponto_dentro_do_gabarito(raio,stretch,(r2_t){{dx,dy}});
      /* Guarda na máscara: */
      float v = (dentro ? 1.0 : 0.0);
      float_image_set_sample(M, 0, x, y, v);
    }
  }
  return M;
}

Tabela* aloca_tabela_vazia(int num_luzes, int num_linhas, r3_t view_dir){
	Tabela* tab = (Tabela*)malloc(sizeof(Tabela));
  	tab->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*num_linhas);
	tab->num_linhas = num_linhas;
  	tab->num_luzes = num_luzes;
  	tab->view_dir = view_dir;
	int i;
	for (i = 0; i < num_linhas; i++){
		tab->tabela[i].intdir = (double*)malloc(sizeof(double)*num_luzes);
	}
	return tab;
}

Tabela* cria_tabela
  ( float_image_t  *G[],
    float_image_t  *M, 
    double albedo, 
    int num_luzes, 
    int canal, 
    r2_t ponto[], 
    r3_t normal[], 
    int num_pontos,
    r3_t view_dir,
    bool_t interpolate_pixels
  )
{

  fprintf(stderr,  "Criando tabela fotométrica\n");
  fprintf(stderr,  "  Num luzes: %d\n", num_luzes);
  fprintf(stderr,  "  Canal: %d\n", canal);
  fprintf(stderr,  "  Num pontos dados: %d\n", num_pontos);
  fprintf(stderr,  "  View Dir = "); r3_print(stderr,&view_dir); fprintf(stderr,"\n");

  Tabela* tab = (Tabela*)malloc(sizeof(Tabela));
  tab->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*num_pontos);

  tab->num_luzes = num_luzes;
  tab->view_dir = view_dir;
  /* Guarda vetores de observação úteis na tabela: */
  int k, luz;
  int count_minus =0;
  int count_norm =0;
  int count_plus = 0;
  int count_wzero = 0;
  int count_small = 0;
  int num_ok = 0;  /* Conta pontos OK. */
  int num_rej = 0; /* Conta pontos rejeitados. */
  for (k = 0; k < num_pontos; k++){
    double GO[num_luzes]; /* Vetor de observações. */
    double GW[num_luzes]; /* Vetor de pesos. */
    double Gmag2 = 0;
    r2_t *q = &(ponto[k]);
    double x = q->c[0];
    double y = q->c[1];
    r3_t *u = &(normal[k]); 
    //normal must be a unitary vector
    r3_dir(u,u);
    assert(!isnan(u->c[0]));
    int nbugs = 0; /* Numero de erros neste ponto. */
    for(luz = 0; luz < num_luzes; luz++){
      float_image_masked_t GM = (float_image_masked_t){ .img = G[luz], .msk = M };
      float val, wht;
      if(interpolate_pixels){
      	float_image_masked_interpolate(&GM, canal, x, y, 2, &val, &wht);
      }else{
	int ix,iy;
	ix = floor(x);
	iy = floor(y);
	val = float_image_get_sample(G[luz],canal,ix, iy);
	wht = float_image_get_sample(M,0,ix, iy);
      }
     // float_image_masked_interpolate_exclusive(&GM, canal, x, y, 1, &val, &wht);
      GO[luz] = val; GW[luz] = wht;
      if (GW[luz] == 0.0) { count_wzero++; nbugs++; }
      if (GO[luz] < 0.0) { count_minus++; nbugs++; }
      if (GO[luz] > 1.0) { count_plus++; nbugs++; }
      if (u->c[2] < 0.0) { count_norm++; nbugs++; }
      Gmag2 += GO[luz]*GO[luz];
    }
    double Gmag = sqrt(Gmag2);
    if (Gmag < MINMAG) { count_small++; nbugs++; }
    if ((nbugs > 0) && (count_wzero+count_minus+count_plus+count_small == nbugs)) {
      /* Primeiro erro nesta tabela. */
      fprintf(stderr, "pixel problematico no gabarito");
      fprintf(stderr, "  luz = %2d  q = (%7.2f %7.2f)\n", luz, x, y);
      for(luz = 0; luz < num_luzes; luz++){
        fprintf(stderr, "  GO[%2d] = %+9.6f wht = %12.6e\n", luz, GO[luz], GW[luz]);
      }
    }
    
    if (nbugs == 0) {
      /* Aloca vetor de observação normalizado: */
      double *go = (double*)malloc(sizeof(double)*num_luzes);
      for(luz = 0; luz < num_luzes; luz++){ go[luz] = GO[luz]/Gmag; }
      tab->tabela[num_ok].intdir = go;
      tab->tabela[num_ok].intmag = Gmag / albedo;
      tab->tabela[num_ok].normal = (*u);
      num_ok++;
    } else {
      num_rej++;
    }
  }
  tab->num_linhas = num_ok;
  fprintf(stderr,  "  Num pontos aceitos: %d\n", num_ok);
  fprintf(stderr,  "  Num pontos rejeitados: %d\n", num_rej);
  fprintf(stderr,  "  Num pontos com intensidade abaixo de zero: %d\n",count_minus);
  fprintf(stderr,  "  Num pontos com intensidade acima de um: %d\n",count_plus);
  fprintf(stderr,  "  Num pontos com normal Z negativa: %d\n",count_norm);
  fprintf(stderr,  "\n");
  return tab;
}


Tabela* cria_tabela_virtual
  ( r3_t  directions[],
    double radius,r2_t centro,
    double albedo, 
    int num_luzes, 
    int canal, 
    r2_t ponto[], 
    r3_t normal[], 
    int num_pontos,
    r3_t view_dir
  )
{

  fprintf(stderr,  "Criando tabela fotométrica virtual\n");
  fprintf(stderr,  "  Num luzes: %d\n", num_luzes);
  fprintf(stderr,  "  Canal: %d\n", canal);
  fprintf(stderr,  "  Num pontos dados: %d\n", num_pontos);
  fprintf(stderr,  "  View Dir = "); r3_print(stderr,&view_dir); fprintf(stderr,"\n");
 
  Tabela* tab = (Tabela*)malloc(sizeof(Tabela));
  tab->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*num_pontos);

  tab->num_luzes = num_luzes;
  tab->view_dir = view_dir;
  /* Guarda vetores de observação úteis na tabela: */
  int k, luz;
  int count_minus =0;
  int count_plus = 0;
  int count_wzero = 0;
  int count_small = 0;
  int num_ok = 0;  /* Conta pontos OK. */
  int num_rej = 0; /* Conta pontos rejeitados. */
  /*entorta direcoes da luz*/
  r3_t true_directions[num_luzes];
  r3x3_t roda_normal = compute_normal_correction_matrix(view_dir);
  for(luz = 0; luz < num_luzes;luz++){
	r3x3_map_row(&(directions[luz]),&roda_normal,&(true_directions[luz]));
  }
  

  for (k = 0; k < num_pontos; k++){
    double GO[num_luzes]; /* Vetor de observações. */
    double GW[num_luzes]; /* Vetor de pesos. */
    double Gmag2 = 0;
    r2_t *q = &(ponto[k]);
    double x = q->c[0];
    double y = q->c[1];
    r3_t *u = &(normal[k]); 
    int nbugs = 0; /* Numero de erros neste ponto. */
    for(luz = 0; luz < num_luzes; luz++){
      float val, wht;
      //float_image_masked_interpolate(&GM, canal, x, y, 2, &val, &wht);
      //val = virtual_gab_intensity(x,y,radius,cor.c[0],cor.c[1], cor.c[2], centro.c[0], centro.c[1]);
      val = lambertian_shading(true_directions[luz],albedo,*u);
      wht = 1.0;
     // float_image_masked_interpolate_exclusive(&GM, canal, x, y, 1, &val, &wht);
      GO[luz] = val; GW[luz] = wht;
      if (GW[luz] == 0.0) { count_wzero++; nbugs++; }
      if (GO[luz] < 0.0) { count_minus++; nbugs++; }
      if (GO[luz] > 1.0) { count_plus++; nbugs++; }
      Gmag2 += GO[luz]*GO[luz];
    }
    double Gmag = sqrt(Gmag2);
    if (Gmag < MINMAG) { count_small++; nbugs++; }
    if ((nbugs > 0) && (count_wzero+count_minus+count_plus+count_small == nbugs)) {
      /* Primeiro erro nesta tabela. */
      fprintf(stderr, "pixel problematico no gabarito");
      fprintf(stderr, "  luz = %2d  q = (%7.2f %7.2f)\n", luz, x, y);
      for(luz = 0; luz < num_luzes; luz++){
        fprintf(stderr, "  GO[%2d] = %+9.6f wht = %12.6e\n", luz, GO[luz], GW[luz]);
      }
    }
    
    if (nbugs == 0) {
      /* Aloca vetor de observação normalizado: */
      double *go = (double*)malloc(sizeof(double)*num_luzes);
      for(luz = 0; luz < num_luzes; luz++){ go[luz] = GO[luz]/Gmag; }
      tab->tabela[num_ok].intdir = go;
      tab->tabela[num_ok].intmag = Gmag / albedo;
      tab->tabela[num_ok].normal = (*u);
      num_ok++;
    } else {
      num_rej++;
    }
  }
  tab->num_linhas = num_ok;
  fprintf(stderr,  "  Num pontos aceitos: %d\n", num_ok);
  fprintf(stderr,  "  Num pontos rejeitados: %d\n", num_rej);
  fprintf(stderr,  "  Num pontos intensidade abaixo de zero: %d\n",count_minus);
  fprintf(stderr,  "  Num pontos intensidade acima de um: %d\n",count_plus);
  fprintf(stderr,  "\n");
  return tab;
}


Tabela* criaSubTabela(Tabela* tab,int subsetSize,int* subSets, int** index){
	Tabela* nova = (Tabela*)malloc(sizeof(Tabela));
        nova->num_luzes = subsetSize;
	//first count how lines would be used
	int num_linhas = tab->num_linhas;
	int i,valid_lines;
	valid_lines = 0;
	int is_valid_line[num_linhas];
	fprintf(stderr,  "Criando subTabela fotométrica a partir da Tabela principal\n");
	
	for(i = 0; i < num_linhas;i++){
		double GO[subsetSize];
		double Gmag = 0;
		const double* goTab = get_intdir(tab, i);
		double gmagTab = get_intmag(tab, i);
		int j;
		for(j = 0; j < subsetSize; j++){
			int ind = subSets[j];
			GO[j] = goTab[ind]*gmagTab;
			Gmag+= GO[j]*GO[j];
		}
		Gmag = sqrt(Gmag);
		is_valid_line[i] = 0;
		if (Gmag > MINMAG) { valid_lines++; is_valid_line[i] = 1;}
	}
	fprintf(stderr,  "Linhas selecionadas: %d de %d\n",valid_lines,num_linhas);
	fprintf(stderr,  "Preenchendo estrutura da tabela\n");
	//now we fill our table
	*index = (int*)malloc(sizeof(int)*valid_lines);
	nova->num_linhas = valid_lines;
	nova->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*valid_lines);
        nova->view_dir = tab->view_dir;
	int k = 0;
	for(i = 0; i < num_linhas; i++){
		if(is_valid_line[i] == 1){
			double* go = (double*)malloc(sizeof(double)*subsetSize);
			double gmag = 0;
			//get GO and gmag from Table
			const double* goTab = get_intdir(tab, i);
			double gmagTab = get_intmag(tab, i);
			int j;
			for(j = 0; j < subsetSize; j++){
				int ind = subSets[j];
				go[j] = goTab[ind]*gmagTab;
				gmag+= go[j]*go[j];
			}
			gmag = sqrt(gmag);
			//normalize vector
			for(j = 0; j < subsetSize; j++){ go[j] = go[j]/gmag; }
			//store data 
			nova->tabela[k].intdir = go;
			nova->tabela[k].intmag = gmag;
			nova->tabela[k].normal = get_normal(tab, i);
			(*index)[k] = i;
			k++;
		}
	}
  	fprintf(stderr,  "SubTabela fotométrica gerada\n");
	return nova;
	
}




double distPoints(double x1, double x2, double y1, double y2);
double get_intdir_selecionado(Tabela* tab,int num_linha,int arquivo,int canal);
void normaliza_assinatura(double **intdir, double *intmag, int num_luzes);
double ObservationMagnitude(double** intdir,int num_luzes,int canal);

double distPoints(double x1, double x2, double y1, double y2){
  double X = (x1 - x2)*(x1 - x2);
  double Y = (y1 - y2)*(y1 - y2);
  return sqrt(X+Y);
}

/* FUNÇÕES DE INTERFACE DA TABELA*/

const double *get_intdir(Tabela* tab, int linha)
{
  return tab->tabela[linha].intdir;
}

double get_intmag(Tabela* tab, int linha)
{
  return tab->tabela[linha].intmag;
}


void set_intdir(Tabela* tab, int linha,double g[]){
	int i;
	for(i = 0; i < tab->num_luzes; i++){
		tab->tabela[linha].intdir[i] = g[i];
	}
}

void set_intmag(Tabela* tab, int linha,double Gmag){
	tab->tabela[linha].intmag = Gmag;
}

r3_t get_view_dir(Tabela* tab){
	return tab->view_dir;
}

void print_linha(Tabela* tab, int linha)
{
  int i;
  fprintf(stderr,  "Linha %d Assinatura", linha);
  const double *go =tab->tabela[linha].intdir;
  for (i = 0; i < tab->num_luzes; i++) { fprintf(stderr,  " %f", go[i]); }
  fprintf(stderr,  " Magnitude %f \n", tab->tabela[linha].intmag);
	
}

int get_num_linhas(Tabela* tab)
{
  return (tab == NULL ? 0 : tab->num_linhas);
}

void set_num_linhas(Tabela* tab,int num_linhas)
{
	tab->num_linhas = num_linhas; 
}


int get_num_luzes(Tabela* tab)
{
  return tab->num_luzes;
}

r3_t get_normal(Tabela* tab, int linha)
{
  return tab->tabela[linha].normal;
}

void set_normal(Tabela* tab, int linha, r3_t normal){
	tab->tabela[linha].normal = normal;	
}

double ObservationMagnitude(double** intdir,int num_luzes,int canal){
	int i;
	double soma = 0.000001;
    	for(i = 0; i < num_luzes; i++){
      		double vi = intdir[i][canal];
      		soma += vi*vi;
    	}
    	return sqrt(soma);
    	
	
}

void normaliza_assinatura(double **intdir, double *intmag, int num_luzes){
  int canal;
  for (canal = 0; canal < 3; canal++){
    double soma = 0.000001;
    int i;
    for(i = 0; i < num_luzes; i++){
      double vi = intdir[i][canal];
      soma += vi*vi;
    }
    intmag[canal] = sqrt(soma);
    for(i = 0; i < num_luzes; i++){
      intdir[i][canal] /= intmag[canal];
    }
  }
}

int localiza_linha_por_normal(Tabela* tab, r3_t *sn)
{
  int tam = get_num_linhas(tab);
  int imin = -1;
  int i;
  double distmin = HUGE_VAL;
  for(i = 0; i < tam; i++) {
    double dist;
    r3_t gn = get_normal(tab,i);
    dist = r3_dist(&gn, sn);
    if (dist < distmin) { distmin = dist; imin = i;	}
		
  }
  return imin;
}

void ShowTableData(char* prefix, Tabela* tab,r2_t ponto[], r3_t normal[], int num_pontos){
	char* filename = NULL;
	asprintf(&filename,"%s_TableData.txt",prefix);
	FILE* arq;
	arq = fopen(filename,"wt");
	if(arq == NULL){
		fprintf(stderr,"showTableData %s FAILED !\n",filename);
		return;
	}
	fprintf(arq,"# Numero linhas\n");
	fprintf(arq,"%d\n",tab->num_linhas);
	fprintf(arq,"# Numero luzes\n");
	fprintf(arq,"%d\n",tab->num_luzes);
	fprintf(arq,"#(x,y) (normal) (intmag) (intdir)\n");
	int i;
	int linha = 0;
	for(i = 0; i < num_pontos; i++){
		int j;
		int test = 1;
		for(j = 0; j < 3; j++){
			if( normal[i].c[j] != tab->tabela[linha].normal.c[j] ){
				test = 0;
			}
		}
		if(test){
			fprintf(arq,"%9.6f %9.6f     ",ponto[i].c[0], ponto[i].c[1]);
			for(j = 0; j < 3; j++){
				fprintf(arq,"%9.6f ",tab->tabela[linha].normal.c[j]);
			}
			fprintf(arq,"    %9.6f    ",tab->tabela[linha].intmag);
			for(j = 0; j < tab->num_luzes;j++){
				fprintf(arq,"%9.6f ",tab->tabela[linha].intdir[j]);
			}
			fprintf(arq,"\n");
			linha++;
		}
	}
	fclose(arq);

}

void SaveTable(char* filename,Tabela* tab, bool_t check){
	FILE* arq;
	arq = fopen(filename,"wt");
	if(arq == NULL){
		fprintf(stderr,"SaveTable %s FAILED !\n",filename);
		return;
	}
	fprintf(arq,"# Numero linhas\n");
	fprintf(arq,"%d\n",tab->num_linhas);
	fprintf(arq,"# Numero luzes\n");
	fprintf(arq,"%d\n",tab->num_luzes);
        fprintf(arq,"# View dir\n");
	r3_gen_print (arq, &tab->view_dir, "%+10.7f", ""," ", "\n");
	fprintf(arq,"#(x,y) (normal) (intmag) (intdir)\n");
	int i;
	bool_t bug = FALSE;
	for(i = 0; i < tab->num_linhas; i++){
	  const double *intdir = tab->tabela[i].intdir;
	  r3_t *normal = &(tab->tabela[i].normal);
	  double intmag = tab->tabela[i].intmag;

	  int j;
	  for(j = 0; j < 3; j++){
	    fprintf(arq,"%e ",normal->c[j]);
	  }
	  fprintf(arq,"    %e    ",intmag);
	  for(j = 0; j < tab->num_luzes;j++){
	    fprintf(arq,"%e ",intdir[j]);
	  }
	  fprintf(arq,"\n");
	  if (check){
	    // Check normal:
	    double len = r3_norm(normal);
	    if (fabs(len - 1.0) > 0.000001){
	      fprintf(stderr, "normal %d is not normalized\n", i); bug = TRUE;
	    }
	    double dot = r3_dot(&tab->view_dir,normal);
	    if(dot < -1.0e-6) {
	      fprintf(stderr, "normal %d is not visible\n", i); bug = TRUE;
	    }
	    // Check signature:
            for(j = 0; j < tab->num_luzes;j++){
	      if (intdir[j] < 0.0){
	      fprintf(stderr, "signature %d light %d is negative\n", i, j); bug = TRUE;
	      }
	    }
            double mag = rn_norm(tab->num_luzes, (double *)intdir);
            if (fabs(mag - 1.0) > 0.000001){
	      fprintf(stderr, "signature %d is not normalized\n", i); bug = TRUE;
	    }
	    if (intmag < 0.000001){
	      fprintf(stderr, "magniture %d = %9.6f is not positive\n", i,intmag); bug = TRUE;
	    }
            fflush(arq);
	    assert(! bug);
	  }
	}
	fclose(arq);
}


void loadTableData(char* filename, Tabela** table,r2_t** pontos_P, r3_t** normals_P ){
	
	auto void advanceUntilEqual(FILE* arq,char ch);
	void advanceUntilEqual(FILE* arq,char ch){
		char c;
		c = fgetc(arq);
		while( c != EOF){
			if(c == ch) break;
			c = fgetc(arq);
		}
	}
	
	FILE* arq;
	arq = fopen(filename,"rt");
	if(arq == NULL){
		*table = NULL;
		*pontos_P = NULL;
		*normals_P = NULL;
		return ;
	}
	Tabela* tab;
	r2_t* pontos;
	r3_t* normals;
	tab = (Tabela*)malloc(sizeof(Tabela));
	advanceUntilEqual(arq,'\n');
	fscanf(arq,"%d",&(tab->num_linhas));	
 	advanceUntilEqual(arq,'#');
	advanceUntilEqual(arq,'\n');
	fscanf(arq,"%d",&(tab->num_luzes));
	advanceUntilEqual(arq,'#');
	advanceUntilEqual(arq,'\n');
	pontos = (r2_t*)malloc(sizeof(r2_t)*(tab->num_linhas));
	normals = (r3_t*)malloc(sizeof(r3_t)*(tab->num_linhas));
	tab->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*(tab->num_linhas));
	
	int i;
	for(i = 0; i < tab->num_linhas; i++){
		r2_t pt;
		r3_t norm;
		double mag;		

		fscanf(arq,"%lf %lf",&pt.c[0],&pt.c[1]);
		pontos[i] = pt;
		fscanf(arq,"%lf %lf %lf",&norm.c[0],&norm.c[1],&norm.c[2]);
		normals[i] = norm;
		fscanf(arq,"%lf",&mag);
		tab->tabela[i].intdir = (double*)malloc(sizeof(double)*(tab->num_luzes));
		tab->tabela[i].normal = norm;
		tab->tabela[i].intmag = mag;
		int j;
		for(j = 0; j < tab->num_luzes; j++){
			fscanf(arq,"%lf",&(tab->tabela[i].intdir[j]));
		}
	}
	fclose(arq);
	*table = tab;
	*pontos_P = pontos;
	*normals_P = normals;
	return ;
	
}

void LoadTable(char* filename, Tabela** table){
	
	auto void advanceUntilEqual(FILE* arq,char ch);
	void advanceUntilEqual(FILE* arq,char ch){
		char c;
		c = fgetc(arq);
		while( c != EOF){
			if(c == ch) break;
			c = fgetc(arq);
		}
	}
	
	FILE* arq;
	arq = fopen(filename,"rt");
	if(arq == NULL){
		*table = NULL;
		return ;
	}
	Tabela* tab;
	tab = (Tabela*)malloc(sizeof(Tabela));
	advanceUntilEqual(arq,'\n');
	fscanf(arq,"%d",&(tab->num_linhas));	
 	advanceUntilEqual(arq,'#');
	advanceUntilEqual(arq,'\n');
	fscanf(arq,"%d",&(tab->num_luzes));
        advanceUntilEqual(arq,'#');
	advanceUntilEqual(arq,'\n');
	int res = fscanf(arq,"%lf %lf %lf",&(tab->view_dir.c[0]),&(tab->view_dir.c[1]),&(tab->view_dir.c[2]) );
	assert(res == 3);
	advanceUntilEqual(arq,'#');
	advanceUntilEqual(arq,'\n');
	tab->tabela = (LinhaTabela*)malloc(sizeof(LinhaTabela)*(tab->num_linhas));
	
	int i;
	for(i = 0; i < tab->num_linhas; i++){
		r3_t norm;
		double mag;		
		fscanf(arq,"%lf %lf %lf",&norm.c[0],&norm.c[1],&norm.c[2]);
		fscanf(arq,"%lf",&mag);
		tab->tabela[i].intdir = (double*)malloc(sizeof(double)*(tab->num_luzes));
		tab->tabela[i].normal = norm;
		tab->tabela[i].intmag = mag;
		int j;
		for(j = 0; j < tab->num_luzes; j++){
			fscanf(arq,"%lf",&(tab->tabela[i].intdir[j]));
		}
	}
	fclose(arq);
	*table = tab;
	return ;
	
}

void PrintTableStats(Tabela* tab){
	fprintf(stderr,"Lights: %d\nLines: %d\n View Dir (%lf,%lf,%lf)\n",tab->num_luzes, tab->num_linhas,tab->view_dir.c[0],tab->view_dir.c[1],tab->view_dir.c[2]);
}

void LiberaTabela(Tabela* tab){
	int i ;
	for( i = 0; i < tab->num_linhas; i++){
		free(tab->tabela[i].intdir);
	}
	free(tab->tabela);
	free(tab);
	
}
